package org.gcube.indexmanagement.common;

import java.io.InputStream;
import java.io.FilterInputStream;
import java.io.IOException;
import java.util.HashMap;

/**
 * An InputStream used to read LZW compressed data written by a
 * CompressingOutputStream. See http://en.wikipedia.org/wiki/LZW for an
 * explanation of the algorithm.
 * 
 * @see org.gcube.indexmanagement.common.CompressingOutputStream
 * 
 * @version 0.1
 */

public class DecompressingInputStream extends FilterInputStream {

    /** The previously read code */
    private int previous = -1;

    /**
     * The number of codes currently registered. Also forms the basis for the
     * next code.
     */
    private int codeCount;

    /** The current bit size of codes ( ie sizeCode(255)=8, sizeCode(256)=9 ) */
    private int codeSize;

    /**
     * The max number of codes which can fit into the curent codeSize. Always
     * 2^codeSize
     */
    private int maxCodes;

    /** The bytes read, but not assembled and returned as a code yet */
    private int remainingPart = 0;

    /** The size of the remainingPart */
    private int remainingSize = 0;

    /**
     * The maximum allowed number of codes (from CompressingOutputStream). A
     * compromize: the more codes, the larger compression rate, but also the
     * more memory and cpu is used...
     */
    private int maxAllowedCodes = (1 << 19);

    /**
     * The code map, mapping codes to values from which decompressed byte arrays
     * can be deduced
     */
    private HashMap<Integer, Integer> codes;

    /** The index of the next byte to be returned of the decompressedBytes array */
    private int decompressedBytesIdx;

    /**
     * A buffer to hold bytes which have been decompressed, until they are all
     * returned
     */
    private byte decompressedBytes[];

    /**
     * A snapshot of the "previous" variable in order to support mark/reset
     * functionality
     */
    private int previousMark;

    /**
     * A snapshot of the "codeCount" variable in order to support mark/reset
     * functionality
     */
    private int codeCountMark;

    /**
     * A snapshot of the "codeSize" variable in order to support mark/reset
     * functionality
     */
    private int codeSizeMark;

    /**
     * A snapshot of the "remainingPart" variable in order to support mark/reset
     * functionality
     */
    private int remainingPartMark;

    /**
     * A snapshot of the "remainingSize" variable in order to support mark/reset
     * functionality
     */
    private int remainingSizeMark;

    /**
     * A snapshot of the "decompressedBytesIdx" variable in order to support
     * mark/reset functionality
     */
    private int decompressedBytesIdxMark;

    /**
     * A snapshot of the "decompressedBytes" variable in order to support
     * mark/reset functionality
     */
    private byte decompressedBytesMark[];

    /**
     * Constructor which wraps an input stream, and initializes the code map
     * with all possible values of a single byte, before increasing the codesize
     * to the size of one byte + 1 to make place for the next code
     * 
     * @param in -
     *            the input stream to read the compressed data from
     */
    public DecompressingInputStream(InputStream in) {
        super(in);
        codes = new HashMap<Integer, Integer>();
        for (codeCount = 0; codeCount < 256; codeCount++) {
            codes.put(codeCount, codeCount);
        }
        decompressedBytes = new byte[] {};
        decompressedBytesIdx = 0;
        setCodeSize(9);
    }

    /**
     * {@inheritDoc}
     */
    public int read() throws IOException {
        byte[] array;
        if ((array = readDecompressed(1)) == null) {
            return -1;
        }
        return array[0] & 0xff;
    }

    /**
     * {@inheritDoc}
     */
    public int read(byte[] b) throws IOException {
        return read(b, 0, b.length);
    }

    /**
     * {@inheritDoc}
     */
    public int read(byte[] b, int off, int len) throws IOException {
        byte[] array = readDecompressed(len);
        if (array == null) {
            return -1;
        }
        System.arraycopy(array, 0, b, off, array.length);
        return array.length;
    }

    /**
     * {@inheritDoc}
     */
    public void mark(int i) {
        if (markSupported()) {
            previousMark = previous;
            codeCountMark = codeCount;
            codeSizeMark = codeSize;
            remainingPartMark = remainingPart;
            remainingSizeMark = remainingSize;
            decompressedBytesMark = new byte[decompressedBytes.length];
            System.arraycopy(decompressedBytes, 0, decompressedBytesMark, 0,
                    decompressedBytes.length);
            decompressedBytesIdxMark = decompressedBytesIdx;
            super.mark(i);
        }
    }

    /**
     * {@inheritDoc}
     */
    public void reset() throws IOException {
        if (markSupported()) {
            super.reset();
            previous = previousMark;
            codeCount = codeCountMark;
            codeSize = codeSizeMark;
            remainingPart = remainingPartMark;
            remainingSize = remainingSizeMark;
            decompressedBytes = new byte[decompressedBytesMark.length];
            System.arraycopy(decompressedBytesMark, 0, decompressedBytes, 0,
                    decompressedBytesMark.length);
            decompressedBytesIdx = decompressedBytesIdxMark;
        }
    }

    /**
     * {@inheritDoc}
     */
    public long skip(long n) throws IOException {
        return skipDecompressed(n);
    }

    /**
     * Reads codes and decompresses them to a byte sequence
     * 
     * @param len -
     *            the number of bytes to return
     * @return the decompressed byte sequence
     * @throws IOException
     */
    private byte[] readDecompressed(int len) throws IOException {
        if (previous == -1) {
            previous = readCode();
            if (previous == -1) {
                return null;
            }
            decompressedBytes = new byte[] { (byte) previous };
        }

        byte[] returnArray = new byte[len];
        int lenToCopy, copiedLen;

        lenToCopy = Math.min(len, decompressedBytes.length
                - decompressedBytesIdx);
        System.arraycopy(decompressedBytes, decompressedBytesIdx, returnArray,
                0, lenToCopy);
        decompressedBytesIdx += lenToCopy;
        copiedLen = lenToCopy;

        while (copiedLen < len) {
            int current = readCode();
            if (current == -1) {
                if (copiedLen > 0) {
                    byte[] lastPart = new byte[copiedLen];
                    System.arraycopy(returnArray, 0, lastPart, 0, copiedLen);
                    return lastPart;
                }
                return null;
            }

            int newCode;
            if (codeCount < maxAllowedCodes) {
                if (current == codeCount) { // LZW cScSc issue
                    newCode = (previous << 8)
                            + (expandCode(previous, codes)[0] & 0xff);
                } else {
                    newCode = (previous << 8)
                            + (expandCode(current, codes)[0] & 0xff);
                }
                codes.put(codeCount++, newCode);

                if (codeCount == maxCodes && codeCount < maxAllowedCodes) {
                    setCodeSize(codeSize + 1);
                }
            }

            previous = current;

            decompressedBytes = expandCode(previous, codes);
            decompressedBytesIdx = 0;

            lenToCopy = Math.min(len - copiedLen, decompressedBytes.length
                    - decompressedBytesIdx);
            System.arraycopy(decompressedBytes, 0, returnArray, copiedLen,
                    lenToCopy);
            decompressedBytesIdx += lenToCopy;
            copiedLen += lenToCopy;
        }

        return returnArray;
    }

    /**
     * A method used to skip len bytes of decompressed data
     * 
     * @param len -
     *            the number of bytes to skip
     * @return the number of bytes actually skipped
     * @throws IOException -
     *             an error reading bytes
     */
    private long skipDecompressed(long len) throws IOException {
        if (previous == -1) {
            previous = readCode();
            if (previous == -1) {
                return 0;
            }
        }

        long skippedLen;
        int lenToSkip;

        lenToSkip = new Long(Math.min(len, new Integer(decompressedBytes.length
                - decompressedBytesIdx).longValue())).intValue();
        decompressedBytesIdx += lenToSkip;
        skippedLen = lenToSkip;

        while (skippedLen < len) {
            int current = readCode();
            if (current == -1) {
                return skippedLen;
            }

            int newCode;
            if (codeCount < maxAllowedCodes) {
                if (current == codeCount) { // LZW cScSc issue.See wikipedia for
                                            // expl.
                    newCode = (previous << 8)
                            + (expandCode(previous, codes)[0] & 0xff);
                } else {
                    newCode = (previous << 8)
                            + (expandCode(current, codes)[0] & 0xff);
                }
                codes.put(codeCount++, newCode);
                if (codeCount == maxCodes && codeCount < maxAllowedCodes) {
                    setCodeSize(codeSize + 1);
                }
            }

            previous = current;

            decompressedBytes = expandCode(previous, codes);
            decompressedBytesIdx = 0;

            lenToSkip = new Long(Math.min(len - skippedLen, new Integer(
                    decompressedBytes.length - decompressedBytesIdx)
                    .longValue())).intValue();
            decompressedBytesIdx += lenToSkip;
            skippedLen += lenToSkip;
        }
        return skippedLen;
    }

    /**
     * A method used to expand a code to the decompressed byte sequence, using
     * the specified code map.
     * 
     * @param code -
     *            the code to expand
     * @param codes -
     *            the code map to use for the expansion
     * @return the decompressed sequence of bytes
     */
    private byte[] expandCode(int code, HashMap<Integer, Integer> codes) {
        int value = codes.get(code);
        if (code < 256) {
            return new byte[] { (byte) value };
        } else {
            byte lastValue = (byte) (value & 0xff);
            int prevCode = value >> 8;
            byte[] prevValues = expandCode(prevCode, codes);
            byte[] returnValues = new byte[prevValues.length + 1];
            System.arraycopy(prevValues, 0, returnValues, 0, prevValues.length);
            returnValues[returnValues.length - 1] = lastValue;
            return returnValues;
        }
    }

    /**
     * The method in charge of reading codes (of codeSize # of bits) from the
     * input bytes
     * 
     * @return the read code
     * @throws IOException -
     *             an error reading bytes
     */
    private int readCode() throws IOException {
        int code;

        int inByte;
        while (remainingSize < codeSize) {
            if ((inByte = in.read()) == -1) {
                return -1;
            }
            remainingPart = (remainingPart << 8) | inByte;
            remainingSize += 8;
        }
        code = remainingPart >> (remainingSize - codeSize);
        remainingPart = remainingPart & ((1 << (remainingSize - codeSize)) - 1);
        remainingSize -= codeSize;

        return code;
    }

    /**
     * A helper method to change the codeSize and maxCodes variables in one go.
     * 
     * @param newSize =
     *            The new codeSize
     */
    private void setCodeSize(int newSize) {
        codeSize = newSize;
        maxCodes = 1 << codeSize;
    }

}
