/**
 * 
 */
package org.gcube.indexmanagement.common;

import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;

import org.globus.wsrf.Topic;

/**
 * @author Spyros Boutsis, NKUA
 *
 */
public abstract class IndexManagementWSResource extends IndexWSResource {

	/** The list of standard resource properties defined by every index management resource */
    private static final String RP_UPDATER = "Updater";
    private static final String RP_INDEX_STATUS = "IndexStatus";
    private static final String RP_CONNECTION_COUNT = "ConnectionCount";
    private static final String RP_DOCUMENT_COUNT = "DocumentCount";
    
    /** The list of possible management status values */
    public static final String MANAGER_STATUS_EMPTY = "EMPTY";
    public static final String MANAGER_STATUS_UPDATING = "UPDATING";
    public static final String MANAGER_STATUS_FINISHED = "FINISHED";
    
	/** The name of the state change notification topic */
	private static final String STATE_CHANGE_TOPIC_NAME = "SharedStateChange";
	
	/** The state change notification topic */
	protected Topic sharedStateChangeTopic;
	
	/**
	 * Initializes the index management resource.
	 * 
	 * @param namespace the namespace of the service that this resource belongs to
	 * @param indexID the indexID of this resource
	 * @param indexTypeName the index type name of this resource
	 * @param collectionID the list of collection IDs of this resource
	 * @throws Exception an error occured
	 */
	public void initialise(String namespace, String indexID, String indexTypeName, String collectionID[]) throws Exception {
		super.initialise(namespace, indexID, indexTypeName, collectionID);

		/* Create and initialize the resource properties */
		createProperty(RP_INDEX_STATUS);
		this.getResourcePropertySet().get(RP_INDEX_STATUS).clear();
    	this.getResourcePropertySet().get(RP_INDEX_STATUS).add(MANAGER_STATUS_EMPTY);

		createProperty(RP_UPDATER);

		createProperty(RP_CONNECTION_COUNT);
		this.getResourcePropertySet().get(RP_CONNECTION_COUNT).clear();
    	this.getResourcePropertySet().get(RP_CONNECTION_COUNT).add(0);

		createProperty(RP_DOCUMENT_COUNT);
		this.getResourcePropertySet().get(RP_DOCUMENT_COUNT).clear();
    	this.getResourcePropertySet().get(RP_DOCUMENT_COUNT).add(0);

		/* Register the state change notification topic to the IS */
        this.sharedStateChangeTopic = this.createTopic(STATE_CHANGE_TOPIC_NAME, namespace + "/" + indexID);
	}
	
	/**
	 * Invoked when a resource is being created from a serialized, previously saved state.
	 * 
	 * @param ois the input stream through which the state can be read
	 * @param indicates if the resource is being loaded for the first time (hard load) or not (soft load)
	 * @throws Exception an error occured during resource deserialization
	 */
	protected void onLoad(ObjectInputStream ois, boolean firstLoad) throws Exception {
		super.onLoad(ois, firstLoad);
		
		/* Create and initialize the resource properties from the given input stream */
		String status = (String) ois.readObject();
		createProperty(RP_INDEX_STATUS);
		this.getResourcePropertySet().get(RP_INDEX_STATUS).clear();
    	this.getResourcePropertySet().get(RP_INDEX_STATUS).add(status);

    	int size = ois.readInt();
		createProperty(RP_UPDATER);
		this.getResourcePropertySet().get(RP_UPDATER).clear();
		for (int i=0; i<size; i++)
			this.getResourcePropertySet().get(RP_UPDATER).add((Integer) ois.readInt());

		Integer connectionCount = ois.readInt();
		createProperty(RP_CONNECTION_COUNT);
		this.getResourcePropertySet().get(RP_CONNECTION_COUNT).clear();
    	this.getResourcePropertySet().get(RP_CONNECTION_COUNT).add(connectionCount);

    	Integer docCount = ois.readInt();
		createProperty(RP_DOCUMENT_COUNT);
		this.getResourcePropertySet().get(RP_DOCUMENT_COUNT).clear();
    	this.getResourcePropertySet().get(RP_DOCUMENT_COUNT).add(docCount);

		/* Register the state change notification topic to the IS */
        this.sharedStateChangeTopic = this.createTopic(STATE_CHANGE_TOPIC_NAME, namespace + "/" + this.getIndexID());
	}
	
	/**
	 * Invoked when the state of the resource must be saved (resource serialization)
	 * 
	 * @param oos the output stream to write the resource state to
	 * @throws Exception an error occured during resource serialization
	 */
	protected void onStore(ObjectOutputStream oos) throws Exception {
		super.onStore(oos);
		
		oos.writeObject((String) this.getResourcePropertySet().get(RP_INDEX_STATUS).get(0));
		
		int size = this.getResourcePropertySet().get(RP_UPDATER).size();
		oos.writeInt(size);
		for (int i=0; i<size; i++)
			oos.writeInt((Integer) this.getResourcePropertySet().get(RP_UPDATER).get(i));
			
		oos.writeInt((Integer) this.getResourcePropertySet().get(RP_CONNECTION_COUNT).get(0));
		
		oos.writeInt((Integer) this.getResourcePropertySet().get(RP_DOCUMENT_COUNT).get(0));
	}
	
	/**
	 * Sends a state change notification message
	 * @param message the notification message to send
	 */
	public void sendSharedStateChangeNotification(Object message) {
		try {
			sharedStateChangeTopic.notify(message);   
			logger.debug("State change notification sent on topic: " + sharedStateChangeTopic.getName());
		} catch(Exception e) {
			logger.error("Exception while trying to send state change notification.", e);
		}
	}

/*
	public void onResourceRemoval() {
		try {
			super.onRemove();
		} catch (Exception e) {
			logger.error("Failed to remove management resource with ID = " + this.getID());
		}
		
		try{
			ISNotifier notifier= GHNContext.getImplementation(ISNotifier.class);
            ArrayList<Topic> brokeredTopicList = new ArrayList<Topic>();
            brokeredTopicList.add(sharedStateChangeTopic);
            notifier.unregisterISNotification(this.getEPR(), brokeredTopicList, this.getServiceContext(),
            		this.getServiceContext().getScope());
            logger.debug("Producer unregistered topic: " + sharedStateChangeTopic.getName() + " from broker for Index: " + this.getIndexID());
        }
        catch(Exception e){
            logger.debug("Failed to unregister notification topics from the IS.", e);
        }
	}
*/

	/**
     * Getter method for the IndexStatus Resource Property
     * @return <code>String</code> the requested status indicator
     */
    public String getIndexStatus() {
        return (String) this.getResourcePropertySet().get(RP_INDEX_STATUS).get(0);
    }
    
    /**
     * Getter method for the ConnectionCount Resource Property
     * @return <code>int</code> the requested ConnectionCount 
     */
    public int getConnectionCount() {
    	return (Integer) this.getResourcePropertySet().get(RP_CONNECTION_COUNT).get(0);
    }

    /**
     * Getter method for the DocumentCount Resource Property
     * @return <code>int</code> the requested DocumentCount 
     */
    public int getDocumentCount() {
    	return (Integer) this.getResourcePropertySet().get(RP_DOCUMENT_COUNT).get(0);
    }
    
    /**
     * Getter method for the Updater Resource Property
     * @return <code>int[]</code> the values of the requested Updater property 
     */
    public int[] getUpdater() {    	
    	int size = this.getResourcePropertySet().get(RP_UPDATER).size();
    	int[] upds = new int[size];
    	for (int i=0; i<size; i++)
    		upds[i] = (Integer) this.getResourcePropertySet().get(RP_UPDATER).get(i);
    	return upds;
    }
    
    /**
     * Returns the number of values in the Updater Resource Property
     * @return the number of updaters
     */
    public int getUpdaterCount() {
    	return this.getResourcePropertySet().get(RP_UPDATER).size();
    }
    
    /**
     * Setter method for the IndexStatus Resource Property
     * @param indexStatus <code>String</code> the new status of the resource
     */
    public void setIndexStatus(String indexStatus) {
    	this.getResourcePropertySet().get(RP_INDEX_STATUS).clear();
    	this.getResourcePropertySet().get(RP_INDEX_STATUS).add(indexStatus);
    }
    
    /**
     * Setter method for the ConnectionCount Resource Property
     * @param ConnectionCount <code>int</code> the new Connection count
     */
    public synchronized void setConnectionCount(int connectionCount) { 
    	this.getResourcePropertySet().get(RP_CONNECTION_COUNT).clear();
    	this.getResourcePropertySet().get(RP_CONNECTION_COUNT).add(new Integer(connectionCount));
    }
    
    /**
     * Setter method for the DocumentCount Resource Property
     * @param DocumentCount <code>int</code> the new document count
     */
    public void setDocumentCount(int docCount) {
    	this.getResourcePropertySet().get(RP_DOCUMENT_COUNT).clear();
    	this.getResourcePropertySet().get(RP_DOCUMENT_COUNT).add(new Integer(docCount));
    }

    /**
     * Setter method for the Updater Resource Property
     * @param updater <code>int[]</code> the new updater values
     */
    public void setUpdater(int[] updaters) {
    	int size = updaters.length;
    	this.getResourcePropertySet().get(RP_UPDATER).clear();
    	for (int i=0; i<size; i++)
    		this.getResourcePropertySet().get(RP_UPDATER).add(new Integer(updaters[i]));
    }
    
    /**
     * Adds a new value to the Updater Resource Property
     * @param updater <code>int</code> the new updater
     */
    public void addUpdater(int updater) {
    	this.getResourcePropertySet().get(RP_UPDATER).add(new Integer(updater));
    }
    
    /**
     * Deletes a value from the Updater Resource Property
     * @param updater <code>int</code> the updater value to remove
     */
    public void deleteUpdater(int updater) {
    	this.getResourcePropertySet().get(RP_UPDATER).remove(new Integer(updater));
    }

}
