/**
 * RTreePyramidPart.java
 *
 * $Author: tsakas $
 * $Date: 2008/01/30 18:32:28 $
 * $Id: RTreeWrapper.java,v 1.2 2008/01/30 18:32:28 tsakas Exp $
 *
 * <pre>
 *             Copyright (c) : 2006 Fast Search & Transfer ASA
 *                             ALL RIGHTS RESERVED
 * </pre>
 */
package org.gcube.indexmanagement.geo;

import java.io.File;
import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.List;

import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.indexmanagement.common.IndexException;
import org.gcube.indexmanagement.geo.ranking.RankEvaluator;
import org.gcube.indexmanagement.geo.refinement.Refiner;
import org.geotools.index.Data;
import org.geotools.index.LockTimeoutException;
import org.geotools.index.TreeException;
import org.geotools.index.rtree.RTree;
import org.geotools.index.rtree.fs.FileSystemPageStore;

import com.vividsolutions.jts.geom.Envelope;

/**
 * A class extending the RTree with data and functionality useful when building
 * a pyramid of indices.
 */
public class RTreeWrapper extends RTree implements Comparable<RTreeWrapper> {
    
	public enum SupportedRelations { geosearch };
	
	//TODO: when the Information System will need to know about modifiers, this
	//definition will not be enough
	public enum GeoSearchModifiers { colID, lang, inclusion, refiner, ranker, not }; 
	
	static GCUBELog logger = new GCUBELog(RTreeWrapper.class);

    /** The maximum are of the entries handled by this RTreeWrapper/pyramid-part */
    private long maxArea;

    /** The GeoIndexType of the geo Index */
    private GeoIndexType indexTypeObject;

    /** The absolute name of the index file on the disk */
    private String indexFile;
    
    /** The random access file containing the raw data */
    private RandomAccessFile rawData;
    
    /**
     * A constructor which builds an index that stores data in the specified
     * PageStore
     * 
     * @param indexFile - the file where to store data.
     * @param indexTypeObject - the geo type object.
     * @throws TreeException - in case opf failure.
     */
    public RTreeWrapper(File indexFile, GeoIndexType indexTypeObject, RandomAccessFile rawData) throws TreeException {
        super(indexFile.exists() ? new FileSystemPageStore(indexFile)
                : new FileSystemPageStore(indexFile, DataWrapper.createDefinition()));
        this.indexTypeObject = indexTypeObject;
        this.indexFile = indexFile.getAbsolutePath();
        this.rawData = rawData;
    }

    /**
     * A constructor which builds an index that stores data in the specified
     * PageStore, and which handles envelopes with an area size lower than the
     * specified maxArea.
     * 
     * @param indexFile - the file where to store data.
     * @param indexTypeObject - the geo object type.
     * @param maxArea
     *            <code>long</code> - the maximum area of objects to be
     *            handled by this index
     * @throws TreeException - in case of failure
     */
    public RTreeWrapper(File indexFile, GeoIndexType indexTypeObject, long maxArea, RandomAccessFile rawData) throws TreeException {
        this(indexFile, indexTypeObject, rawData);
        this.maxArea = maxArea;
    }

    /**
     * Inserts an Entry into the wrapped RTree
     * 
     * @param geoData -
     *            A DataWrapper containing the coordinates, id and data of the
     *            entry to be added
     * @throws IndexException
     *             unable to insert the entry into the RTree
     */
    public void insert(DataWrapper geoData) throws IndexException {
        try {
            insert(geoData.getMbr(), geoData.getData());
        } catch (Exception e) {
            throw new IndexException(e);
        }
    }

    /**
     * Sets the maximum area of objects to be handled by this index
     * 
     * @param maxArea
     *            <code>long</code> - the maximum area of objects to be
     *            handled by this index
     */
    public void setMaxArea(long area) {
        maxArea = area;
    }

    /**
     * Gets the maximum area of objects to be handled by this index
     * 
     * @return <code>long</code> - the maximum area of objects to be handled
     *         by this index
     */
    public long getMaxArea() {
        return maxArea;
    }

    /*
     * (non-Javadoc)
     * @see java.lang.Comparable#compareTo(java.lang.Object)
     */
    public int compareTo(RTreeWrapper object) {
        return new Long(maxArea).compareTo(new Long(object.getMaxArea()));
    }

    public String getIndexFile() {
    	return this.indexFile;
    }
    
    /**
     * Performs a query on this index
     * 
     * @param query -
     *            The area to look for results in
     * @param method -
     *            The containment method to use for the query
     * @param refiners -
     *            A list refiners to use to refine the results
     * @param ranker -
     *            A ranker to use to rank/sort the results
     * @return - an object containing the results of the query.
     * @throws TreeException -
     *             an error querying the underlying RTree
     * @throws LockTimeoutException -
     *             an error querying the underlying RTree
     */
    public QueryResults performQuery(Envelope query, InclusionType method,
            Refiner[] refiners, RankEvaluator ranker) throws TreeException,
            LockTimeoutException, Exception {
        List<Data> matches = new ArrayList<Data>();
        List<Data> includesMatches;
        
        //In the current version of geotools Rtree class is NOT
        //thread-safe. So the threads that search on the underlying 
        //Rtree for this object must do it sequentially.
        synchronized(this) {
        	includesMatches = super.search(query);
        }
        
        logger.info("The underlying rtree query: " + query + " returned " + includesMatches.size() + " results.");
        
        //the return from the rtree is the intersect
        if(method == InclusionType.intersect){
            logger.info("Intersect chosen: complete rtree results returned.");
            return new QueryResults(indexTypeObject, includesMatches, refiners, ranker, rawData);
        }
        
        //post processing for anything else than intersect
        boolean hit;
        for (Data data : includesMatches) {
            Envelope entry = new Envelope((Long) data.getValue(0), (Long) data
                    .getValue(1), (Long) data.getValue(2), (Long) data
                    .getValue(3));
            
            switch (method) {
            case contains:
                hit = query.contains(entry);
                break;
            case inside:
                hit = entry.contains(query);
                break;
            default:
                hit = false;
            }

            if (hit) {
                matches.add(data);
            }
        }
        logger.info("Refined rtree results using \"" + method + "\" to " + matches.size() + " results." );
        return new QueryResults(indexTypeObject, matches, refiners, ranker, rawData);
    }
}
