package org.gcube.indexmanagement.geo;

import java.io.RandomAccessFile;
import java.util.ArrayList;
import java.util.List;

import org.gcube.indexmanagement.geo.ranking.RankEvaluator;
import org.gcube.indexmanagement.geo.ranking.Sorter;
import org.gcube.indexmanagement.geo.refinement.Refiner;
import org.geotools.index.Data;

/**
 * A class used to wrap the results of a query in order to incrementally refine,
 * sort and return them upon request
 */
public class QueryResults {

    /** The index of the next result to return */
    private int currentIdx;

    /**
     * The total number of (unrefined) hits ( = the size of the unrefinedResults
     * list)
     */
    private int totHitCount;

    /** The current number of refined hits */
    private int curRefCount;

    /**
     * The current number of hits processed (both those removed and those
     * accepted by the refinement process)
     */
    private int curUnrefCount;

    /** The current number of results removed in the refinement process */
    private int curRemovedCount;

    /** A list of all the initial hits (urefined) */
    private ArrayList<DataWrapper> results;

    /** A list for results which have been refined, but not returned yet */
    private ArrayList<DataWrapper> leftOvers;

    /** The refiners specified for the query */
    private Refiner[] refiners;

    /** The rank evaluator specified for the query */
    private RankEvaluator ranker;
    
    /** The random access file containing the raw data */
    private RandomAccessFile rawData;

    /**
     * A Constructor which wraps a list of unrefined results, and assigns the
     * refiners and sorters to be used.
     * 
     * @param indexTypeObject - type of geo object
     * @param unrefinedResults -
     *            A list of the unrefined results of a query
     * @param refiners -
     *            An array of the refiners specified for the query
     * @param ranker -
     *            The ranker specified for the query

     */
    public QueryResults(GeoIndexType indexTypeObject,
            List<Data> unrefinedResults, Refiner[] refiners,
            RankEvaluator ranker, RandomAccessFile rawData) throws Exception{
        this.refiners = refiners;
        this.ranker = ranker;
        this.rawData = rawData;
        this.results = wrap(indexTypeObject, unrefinedResults);
        this.currentIdx = 0;
        this.totHitCount = unrefinedResults.size();
        this.curRefCount = 0;
        this.curUnrefCount = 0;
        this.curRemovedCount = 0;
        this.leftOvers = new ArrayList<DataWrapper>();        
    }

    /**
     * A method to refine, sort and return a specified number of results
     * 
     * @param numberOfResults -
     *            the number of results to return
     * @return The requested results. Either the number specified, the number
     *         available (if less than specified) or NULL (if no more are
     *         available)
     */
    public synchronized ArrayList<DataWrapper> getNext(int numberOfResults) {
        ArrayList<DataWrapper> returnList = new ArrayList<DataWrapper>();

        if (leftOvers.size() > numberOfResults) {
            returnList.addAll(leftOvers.subList(0, numberOfResults));
            leftOvers = getSlice(leftOvers, numberOfResults, leftOvers.size());
            return returnList;
        } else {
            returnList.addAll(leftOvers);
            leftOvers.clear();
        }

        while (returnList.size() < numberOfResults
                && currentIdx < results.size()) {

            int estEndIdx = Math.min(currentIdx + numberOfResults,
                    results.size());

            if (ranker != null) {
                Sorter.partialQuickSort(results, currentIdx,
                        estEndIdx, true, ranker);
            }
            if (refiners != null) {
                for (Refiner refiner : refiners) {
                    int removed = refiner.refine(results.subList(
                            currentIdx, estEndIdx));
                    estEndIdx -= removed;
                    curRemovedCount += removed;
                }
            }
            returnList.addAll(results.subList(currentIdx, estEndIdx));
            currentIdx = estEndIdx;
        }
        curRefCount = currentIdx;
        curUnrefCount = currentIdx + curRemovedCount;

        if (returnList.size() > numberOfResults) {
            leftOvers = getSlice(returnList, numberOfResults, returnList.size());
            returnList = getSlice(returnList, 0, numberOfResults);
        }

        return returnList.isEmpty() ? null : returnList;
    }

    /**
     * Returns the rest of the available results
     * 
     * @return the available results
     */
    public synchronized ArrayList<DataWrapper> getRest() {
        return getNext(results.size() - currentIdx);
    }

    /**
     * Returns all results (including previously retrieved), without updating
     * the position marker (ie. getNext() will give the same results regardless
     * of whether getAll() has been called).
     * 
     * @return All the refined results
     */
    public synchronized ArrayList<DataWrapper> getAll() {
        if(currentIdx < results.size()){
            if (ranker != null) {
                Sorter.partialQuickSort(results, currentIdx,
                        results.size(), true, ranker);
            }
    
            if (refiners != null) {
                for (Refiner refiner : refiners) {
                    refiner.refine(results.subList(currentIdx,
                            results.size()));
                }
            }
            leftOvers.addAll(results.subList(currentIdx, results
                    .size()));

            currentIdx = results.size();
            curRefCount = results.size();
            curUnrefCount = totHitCount;
            curRemovedCount = curUnrefCount -  curRefCount;          
        }
        
        
        return results;
    }

    /**
     * A method to get the total number of (unrefined) hits
     * 
     * @return the total number of (unrefined) hits
     */
    public synchronized int getTotalHitCount() {
        return totHitCount;
    }

    /**
     * A method to get the current number of refined hits
     * 
     * @return the current number of refined hits
     */
    public synchronized int getCurrentRefinedCount() {
        return curRefCount;
    }

    /**
     * A method to get the current number of hits processed
     * 
     * @return the current number of hits processed (both those removed and
     *         those accepted during the refinement process)
     */
    public synchronized int getCurrentUnrefinedCount() {
        return curUnrefCount;
    }

    /**
     * A method used to get a copy of a portion of a list.
     * 
     * @param <T> -
     *            The Type of the List to copy a portion of
     * @param list -
     *            The list to copy a portion of
     * @param start -
     *            The starting index of the portion to be copied (inclusive)
     * @param end -
     *            The end index of the portion to be copied (exclusive)
     * @return - A copy of the specified portion of the specified List
     */
    private <T> ArrayList<T> getSlice(List<T> list, int start, int end) {
        return new ArrayList<T>(list.subList(start, end));
    }

    /**
     * Creates an ArrayList of DataWrapper objects wrapping corresponding
     * objects in the specified List of Data object
     * 
     * @param list -
     *            A list of Data objects to be wrapped by DataWrapper objects
     * @return - an ArrayList of DataWrapper objects wrapping corresponding
     *         objects in the specified List of Data object
     */
    private ArrayList<DataWrapper> wrap(GeoIndexType indexTypeObject,
            List<Data> list) throws Exception{
        ArrayList<DataWrapper> wrappedList = new ArrayList<DataWrapper>();
        for (Data data : list) {
            wrappedList.add(DataWrapper.getInstance(indexTypeObject, data, rawData));
        }
        return wrappedList;
    }

}
