package org.gcube.indexmanagement.geo.ranking;

import java.util.Comparator;

import org.gcube.indexmanagement.geo.DataWrapper;
import org.gcube.indexmanagement.geo.GeoIndexType;
import org.gcube.indexmanagement.geo.InclusionType;
import org.gcube.indexmanagement.geo.shape.Polygon;
import org.gcube.indexmanagement.geo.shape.Rectangle;

import com.vividsolutions.jts.geom.Envelope;

/** The base class for all RankEvaluators */
public abstract class RankEvaluator implements Comparator<DataWrapper> {

    /** The query polygon */
    protected Polygon polygon;

    /** The query Minimal Bounding Rectangle */
    protected Envelope query;

    /** The index type of the Index using this RankEvaluator */
    protected GeoIndexType indexType = null;

    /** The containment method used in the query */
    protected InclusionType containmentMethod;

    /** An indicator of whether the sorting should be reversed */
    protected boolean reverseSort;

    /**
     * {@inheritDoc}
     */
    final public int compare(DataWrapper data1, DataWrapper data2) {
        if (!reverseSort) {
            return new Double(rank(data2)).compareTo(new Double(rank(data1)));
        } else
            return new Double(rank(data1)).compareTo(new Double(rank(data2)));
    }

    /**
     * A method called by the GeoIndexResource after creating an instance of a
     * RankEvaluator
     * 
     * @param polygon -
     *            A polygon representation of the original query of which this
     *            RankEvaluator will rank the results
     * @param containmentMethod -
     *            The containment method used in the original query
     * @param reverseSort -
     *            An indicator of whether the rank evaluator should sort the
     *            results in reverse
     * @param indexType -
     *            The geo index type of the GeoIndex using this RankEvaluator
     * @param args -
     *            Any additional plugin specific initialization arguments.
     * @throws RankEvaluatorInitializationException -
     *             An error occurex while initializing the RankEvaluator
     */
    final public void init(Polygon polygon, InclusionType containmentMethod,
            Boolean reverseSort, GeoIndexType indexType, String args[])
            throws RankEvaluatorInitializationException {
        this.polygon = polygon;
        Rectangle rect = polygon.getBoundingBox();
        this.query = new Envelope(rect.getMinX(), rect.getMaxX(), rect
                .getMinY(), rect.getMaxY());
        this.containmentMethod = containmentMethod;
        if (reverseSort == null) {
            reverseSort = false;
        }
        this.reverseSort = reverseSort;

        this.indexType = indexType;
        if (args == null) {
            args = new String[] {};
        }
        initialize(args);
    }

    /**
     * The plugin specific initialization method. Initializes an instance of a
     * RankEvaluator plugin using an array of Strings. All other initialization
     * methods have been completed when this method is called, so it can make
     * use of any of the standard RankEvaluator instance variables.
     * 
     * @param args -
     *            A plugin specific array of Strings used to initialize a
     *            RankEvaluator plugin.
     * @throws RankEvaluatorInitializationException -
     *             An error occurex while initializing the RankEvaluator
     */
    abstract protected void initialize(String args[])
            throws RankEvaluatorInitializationException;

    /**
     * A method used to check if this RankEvaluator plugin is compatible with
     * indices/entries with a specific GeoIndexType
     * 
     * @param indexType -
     *            The GeoIndexType of which to check the compatability with this
     *            plugin
     * @return TRUE if this RankEvaluator plugin is compatible with
     *         indices/entries with the specified GeoIndexType
     */
    abstract public boolean isIndexTypeCompatible(GeoIndexType indexType);

    /**
     * A method used to rank a GeoIndex entry. A value between 0 and 1 is
     * encouraged, but not obligatory.
     * 
     * @param data -
     *            The GeoIndex entry to rank.
     * @return the rank of the specified GeoIndex entry.
     */
    abstract public double rank(DataWrapper data);

}
