package org.gcube.indexmanagement.geo;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map.Entry;
import java.util.Vector;

import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.indexmanagement.common.IndexType;
import org.gcube.indexmanagement.gcqlwrapper.GcqlProcessor;
import org.gcube.indexmanagement.gcqlwrapper.GcqlQueryContainer;
import org.gcube.indexmanagement.geo.shape.Point;
import org.gcube.indexmanagement.geo.shape.Polygon;
import org.gcube.indexmanagement.geo.shape.PolygonProcessing;
import org.gcube.indexmanagement.resourceregistry.RRadaptor;

import search.library.util.cql.query.tree.GCQLAndNode;
import search.library.util.cql.query.tree.GCQLNode;
import search.library.util.cql.query.tree.GCQLNotNode;
import search.library.util.cql.query.tree.GCQLOrNode;
import search.library.util.cql.query.tree.GCQLProjectNode;
import search.library.util.cql.query.tree.GCQLQueryTreeManager;
import search.library.util.cql.query.tree.GCQLTermNode;
import search.library.util.cql.query.tree.Modifier;
import search.library.util.cql.query.tree.ModifierSet;

public class GeoGcqlProcessor extends GcqlProcessor{
	
	private static final String POLYGON_REFINER = "PolygonalRefiner";

	static GCUBELog logger = new GCUBELog(GeoGcqlProcessor.class);
	
	private LinkedHashMap<String, String> projectedFields = new LinkedHashMap<String, String>();
	private Integer numberOfDecimals = 4;
	private HashMap<String, ArrayList<String>> colLangPairs = new HashMap<String, ArrayList<String>>();
	
	public Integer getNumberOfDecimals() {
		return numberOfDecimals;
	}

	public void setNumberOfDecimals(Integer numberOfDecimals) {
		this.numberOfDecimals = numberOfDecimals;
	}
	
	public static void main(String[] args) {
		
		ArrayList<String> presentable = new ArrayList<String>();
		ArrayList<String> searchable = new ArrayList<String>();
		
		//Note that the geosearch relation must have at least 4 points (8 coords) as arguments
		String gCQLQuery = "(((geo geosearch/colID=\"C\"/lang=\"en\"/inclusion=\"inside\" \"10 10 10 20 20 20 20 10\") or (geo geosearch/colID=\"C\"/lang=\"en\"/inclusion=\"1\" \"0 0 0 8 4 8 4 0\")) and ((geo geosearch/colID=\"C\"/lang=\"en\"/inclusion=\"1\" \"-2 -6 -2 4 8 4 8 -6\") or (geo geosearch/colID=\"A\"/lang=\"en\"/inclusion=\"1\" \"-2 -6 -2 4 8 4 8 -6\")))";
		try {
			GcqlQueryContainer results = (GcqlQueryContainer)(new GeoGcqlProcessor().processQuery(presentable, searchable, gCQLQuery, new RRadaptor("dummy")));
			
			System.out.println("result : " + results);
			
		} catch(Exception e) {
			e.printStackTrace();
		}
	}
	
	public void setCurrentColLangPairs(HashMap<String, HashMap<String, Vector<RTreeWrapper>>> index) {
		//get an iterator over the collections 
		Iterator<Entry<String, HashMap<String, Vector<RTreeWrapper>>>> colIter = index.entrySet().iterator();
		while(colIter.hasNext()) {
			Entry<String, HashMap<String, Vector<RTreeWrapper>>> current = colIter.next();
			String colID = current.getKey();
			ArrayList<String> langs =  new ArrayList<String>();
			
			//get an iterator over languages
			Iterator<String> langIter = current.getValue().keySet().iterator();
			while(langIter.hasNext())
			{
				langs.add(langIter.next());
			}
			
			//add the languages for this collections ID
			colLangPairs.put(colID, langs);			
		}
	}

	public GcqlQueryContainer processQuery(ArrayList<String> presentableFields, ArrayList<String> searchableFields, String gCQLQuery, RRadaptor adaptor) throws Exception {
		//store the presentable and searchable fields
		logger.trace("presentables: " + Arrays.toString(presentableFields.toArray(new String[presentableFields.size()])));
		logger.trace("searchables: " + Arrays.toString(searchableFields.toArray(new String[searchableFields.size()])));
		this.presentableFields = presentableFields;
		this.searchableFields =  searchableFields;
		//store the Resource Registry adaptor
		this.adaptor = adaptor;
		
		//use the gCQL parser to get a tree from the String
		GCQLNode head = GCQLQueryTreeManager.parseGCQLString(gCQLQuery);
		//the processNode will run recursively, will retrieve 
		//the projected fields and will generate a lucene query
		GeoGcqlQueryContainer cont = processNode(head, false);
		return cont;
	}
	
	private GeoGcqlQueryContainer processNode(GCQLNode node, boolean not) throws Exception{
		
		//cases for the possible node types
		if(node instanceof GCQLProjectNode)
			return processNode((GCQLProjectNode)node, not);
		if(node instanceof GCQLAndNode)
			return processNode((GCQLAndNode)node, not);
		if(node instanceof GCQLNotNode)
			return processNode((GCQLNotNode)node, not);
		if(node instanceof GCQLOrNode)
			return processNode((GCQLOrNode)node, not);
		if(node instanceof GCQLTermNode)
			return processNode((GCQLTermNode)node, not);
		
		throw new Exception("This node class is not supported: " + node.getClass().toString());
	}
	
	private GeoGcqlQueryContainer processNode(GCQLProjectNode node, boolean not) throws Exception{
		//add all the projections in the projected fields
		Vector<ModifierSet> projections = node.getProjectIndexes();
		for(ModifierSet projection : projections)
		{
			//check if this projection is the wildcard
			if(projection.getBase().equals(IndexType.WILDCARD)) {
				projectedFields.clear();
				return processNode(node.subtree, false);
			}
			
			//get the field label for this field id
			String fieldLabel = adaptor.getFieldNameById(projection.getBase());
			String projField = findPresentable(fieldLabel);
			if(projField == null)
			{
				logger.error(fieldLabel + ", " + projection.getBase() + " is not a presentable field");
				continue;
			}
			projectedFields.put(projection.getBase(), projField);
		}
		//return the lucene query of the subtree
		GeoGcqlQueryContainer result = processNode(node.subtree, false);
		result.setProjectedFields(projectedFields);
		return result;
	}
	
	private GeoGcqlQueryContainer processNode(GCQLTermNode node, boolean not) throws Exception{
		
		logger.debug("Term node - not: " + not);
		
		GeoGcqlQueryContainer result = new GeoGcqlQueryContainer(null);
		boolean found = false;
		//find the relation
		for(RTreeWrapper.SupportedRelations relation : RTreeWrapper.SupportedRelations.values())
		{
			if(node.getRelation().getBase().equalsIgnoreCase(relation.toString()))
			{
				//geo resources currently support only one relation
				switch(relation) {
				case geosearch:
					//we will hold all the refiners here
					ArrayList<RefinementRequest> refiners = new ArrayList<RefinementRequest>();
					//the ranker
					RankingRequest ranker = null;
					//the inclusion type
					InclusionType inclusion = null;
					//the vertices of the search polygon
					ArrayList<Point> vertices = new ArrayList<Point>();
					
					//first get the search polygon described in the term
					String[] coords = splitTerms(node.getTerm());
					
					//we need at least 8 points
					if(coords.length < 8)
					{
						logger.error("the term of geosearch must have at least 8 doubles, while this argument was provided: " + node.getTerm());
						throw new Exception("the term of geosearch must have at least 8 doubles, while this argument was provided: " + node.getTerm());
					}
					
					//store them in the vertices array
					try {
						for(int i=0; i<coords.length; i+=2)
						{
							Point p = new Point(new Double(Double.parseDouble(coords[i]) * Math.pow(10d, getNumberOfDecimals())).longValue(), 
									new Double(Double.parseDouble(coords[i+1]) * Math.pow(10d, getNumberOfDecimals())).longValue());
							vertices.add(p);
						}
					} catch (Exception e) {
						logger.error("Bad syntax for term of the geosearch relation: " + node.getTerm(), e);
						throw new Exception("Bad syntax for term of the geosearch relation: " + node.getTerm(), e);
					}	
					//create the Polygon
					Polygon polygon = new Polygon(vertices);
					//store the colID and lang
					String colID = null;
					String lang = null;
					
					//examine all the modifiers
					for(Modifier modifier : node.getRelation().getModifiers())
					{
						//find out which is the current modifier
						for(RTreeWrapper.GeoSearchModifiers geoMod : RTreeWrapper.GeoSearchModifiers.values())
						{
							if(modifier.getType().equalsIgnoreCase(geoMod.toString()))
							{
								String[] args = null;
								String id = null;
								ArrayList<String> arguments = null;
								boolean rev = false;
								//found
								switch(geoMod) {
								case inclusion:
									//only one inclusion must be specified
									if(inclusion != null)
									{
										logger.warn("only one inclusion must be specified in a geosearch relation");
										break;
									}
									args = splitTerms(modifier.getValue());
									if(args.length > 1)
									{
										logger.error("The inclusion modifier must have 0 | 1 | 2, while the argument provided was: " + modifier.getValue()); 
										throw new Exception("The inclusion modifier must have 0 | 1 | 2, while the argument provided was: " + modifier.getValue());
									}
									String inc = "";
									
									try{
										inc = args[0];
									}catch (NumberFormatException e) {
										logger.error("The inclusion modifier must have 0 | 1 | 2, while the argument provided was: " + modifier.getValue(), e);
										throw new Exception("The inclusion modifier must have 0 | 1 | 2, while the argument provided was: " + modifier.getValue(), e);
									}
									if (inc.equalsIgnoreCase("intersect"))
										inclusion = InclusionType.intersect;
									else if (inc.equalsIgnoreCase("contains"))
										inclusion = InclusionType.contains;
									else if (inc.equalsIgnoreCase("inside"))
										inclusion = InclusionType.inside;
										
									break;
								case colID:
									//only one colID must be specified
									if(colID != null)
									{
										logger.warn("only one colID must be specified in a geosearch relation");
										break;
									}
									args = splitTerms(modifier.getValue());
									if(args.length > 1)
									{
										logger.error("The colID modifier must be a single string, while the argument provided was: " + modifier.getValue());
										throw new Exception("The colID modifier must be a single string, while the argument provided was: " + modifier.getValue());
									}
									colID = args[0];
																		
									break;
								case lang:
									//only one lang must be specified
									if(lang != null)
									{
										logger.warn("only one lang must be specified in a geosearch relation");
										break;
									}
									args = splitTerms(modifier.getValue());
									if(args.length > 1)
									{
										logger.error("The lang modifier must be a single string, while the argument provided was: " + modifier.getValue());
										throw new Exception("The lang modifier must be a single string, while the argument provided was: " + modifier.getValue());
									}
									lang = args[0];
									
									break;
								case not:
									
									//revert the not
									not = !not;
									
									break;
								case ranker:
									//only one ranker must be specified
									if(ranker != null)
									{
										logger.warn("only one ranker modifier must be specified in a geosearch relation");
										break;
									}
									
									args = splitTerms(modifier.getValue());
									
									//we need at least 1 arg(the id)
									if(args.length < 1)
									{
										logger.error("invalid ranker modifier: " + modifier.getValue());
										break;
									}
									
									id = null;
									arguments = new ArrayList<String>();
									rev = false;
									//try to get all the arguments
									try {
										id = args[0];
										rev = Boolean.parseBoolean(args[1]);
										for(int i=2; i<args.length; i++)
											arguments.add(args[i]);
									} catch (Exception e) {
										logger.error("invalid ranker modifier: " + modifier.getValue(), e);
										throw new Exception("invalid ranker modifier: " + modifier.getValue(), e);
									}
									
									//create a new ranker
									ranker = new RankingRequest(id, arguments.toArray(new String[arguments.size()]), rev); 
									
									break;
								case refiner:
									args = splitTerms(modifier.getValue());
									
									//we need at least 1 arg(the id)
									if(args.length < 1)
									{
										logger.error("invalid refiner modifier: " + modifier.getValue());
										break;
									}
									
									id = null;
									arguments = new ArrayList<String>();
									//try to get all the arguments
									try {
										id = args[0];
										for(int i=1; i<args.length; i++)
											arguments.add(args[i]);
									} catch (Exception e) {
										logger.error("invalid refiner modifier: " + modifier.getValue(), e);
										throw new Exception("invalid refiner modifier: " + modifier.getValue(), e);
									}
									
									//create a new ranker - InclusionType will be added in the end
									refiners.add(new RefinementRequest(polygon, id, arguments.toArray(new String[arguments.size()]), null, false));
									
									break;
								default:
									logger.error("Unsupported modifier: " + modifier.getValue());
								}
								break;
							}
						}						
					}
					
					//now add the inclusion type to all the refiners
					if(inclusion == null) {
						logger.warn("the modifier for inclusion is missing in geosearch relation");
						//default will be intersect
						inclusion = InclusionType.intersect;
					} 
					for(RefinementRequest req : refiners) {
						req.setInclusion(inclusion);
					}
					
					//create a query for a collection
					if(colID != null && lang != null)
					{
						GeoGcqlCollectionQuery colQuery = new GeoGcqlCollectionQuery(inclusion, polygon, refiners, ranker, not);
						ArrayList<GeoGcqlCollectionQuery> queries = new ArrayList<GeoGcqlCollectionQuery>();
						queries.add(colQuery);
						HashMap<String, ArrayList<GeoGcqlCollectionQuery>> langMap = new HashMap<String, ArrayList<GeoGcqlCollectionQuery>>();
						langMap.put(lang, queries);
						result.queries.put(colID, langMap);
						logger.debug("ColID: " + colID + " - Lang: " + lang + " - Query: " + colQuery.toString());
						
					} else if(colID != null && lang == null) {
						ArrayList<String> langs = colLangPairs.get(colID);
						//if this collection is not part of the collections indexed
						if(langs == null)
						{
							logger.error("This collection is not indexed in this geo Index resource: " + colID);
							throw new Exception("This collection is not indexed in this geo Index resource: " + colID);
						}
						
						//place a query for all the langs of this collection
						HashMap<String, ArrayList<GeoGcqlCollectionQuery>> langMap = new HashMap<String, ArrayList<GeoGcqlCollectionQuery>>();
						for(String l : langs) {
							GeoGcqlCollectionQuery colQuery = new GeoGcqlCollectionQuery(inclusion, polygon, refiners, ranker, not);
							ArrayList<GeoGcqlCollectionQuery> queries = new ArrayList<GeoGcqlCollectionQuery>();
							queries.add(colQuery);
							langMap.put(l, queries);
							logger.debug("ColID: " + colID + " - Lang: " + l + " - Query: " + colQuery.toString());
						}
						
						//store the map of queries for this collection ID 
						result.queries.put(colID, langMap);
						
					} else if(colID == null && lang == null) {
						
						//for all the collection IDs
						Iterator<Entry<String, ArrayList<String>>> colIter = colLangPairs.entrySet().iterator();
						while(colIter.hasNext()) {
							Entry<String, ArrayList<String>> current = colIter.next();
							String id = current.getKey();
							ArrayList<String> langs = current.getValue();
							
							//place a query for all the langs of this collection
							HashMap<String, ArrayList<GeoGcqlCollectionQuery>> langMap = new HashMap<String, ArrayList<GeoGcqlCollectionQuery>>();
							for(String l : langs) {
								GeoGcqlCollectionQuery colQuery = new GeoGcqlCollectionQuery(inclusion, polygon, refiners, ranker, not);
								ArrayList<GeoGcqlCollectionQuery> queries = new ArrayList<GeoGcqlCollectionQuery>();
								queries.add(colQuery);
								langMap.put(l, queries);
								logger.debug("ColID: " + id + " - Lang: " + l + " - Query: " + colQuery.toString());
							}
							
							//store the map of queries for this collection ID 
							result.queries.put(id, langMap);
						}
						
					} else {
						logger.error("While a language is specified, a collection is not");
						throw new Exception("While a language is specified, a collection is not");
					}
					
					found = true;
					
					break;
				
				default:
					logger.error("Bug! should not reach this point, since this relation is not supported: " + relation);
					throw new Exception("Bug! should not reach this point, since this relation is not supported: " + relation);
				}
				break;
			}
		}
		
		if(!found)
		{
			logger.error("This relation is not supported: " + node.getRelation().getBase());
			throw new Exception("This relation is not supported: " + node.getRelation().getBase());
		} 
		return result;
	}
	
	private GeoGcqlQueryContainer processNode(GCQLAndNode node, boolean not) throws Exception{
		//retrieve the query info from the left subtree
		GeoGcqlQueryContainer cont1 = processNode(node.left, not);
		//retrieve the query info from the right subtree
		GeoGcqlQueryContainer cont2 = processNode(node.right, not);
		
		logger.debug("And node - not: " + not);
		logNumberOfConditions(cont1);
		logNumberOfConditions(cont2);
		
		//the query info that will be returned 
		GeoGcqlQueryContainer result =  new GeoGcqlQueryContainer(null);
		
		if(!not) {
			
			//in case not flag is NOT raised we have an AND node 
			//so we must keep queries for collections 
			//that are referenced in both subtrees. For each matching 
			//collection query we need to merge the refinement criteria
			mergeQueries(cont1, cont2, result);
		} else {
			//in case not flag is raised, AND becomes the OR of NOT nodes 
			//this is an OR node so we must keep all queries, and for 
			//queries that refer to the same collection will be 
			//concatenated in the list for that collection
			concatQueries(cont1, cont2, result);
			
		}
		
		return result;
	}
	
	private GeoGcqlQueryContainer processNode(GCQLOrNode node, boolean not) throws Exception{
		//retrieve the query info from the left subtree
		GeoGcqlQueryContainer cont1 = processNode(node.left, not);
		//retrieve the query info from the right subtree
		GeoGcqlQueryContainer cont2 = processNode(node.right, not);
		
		logger.debug("Or node - not: " + not);
		logNumberOfConditions(cont1);
		logNumberOfConditions(cont2);
		
		//the query info that will be returned 
		GeoGcqlQueryContainer result =  new GeoGcqlQueryContainer(null);
		
		if(!not) {
			//in case not flag is NOT raised we have an OR node 
			//so we must keep all queries, and for 
			//queries that refer to the same collection will be 
			//concatenated in the list for that collection
			concatQueries(cont1, cont2, result);
		} else {
			//in case not flag is raised, OR becomes the AND of NOT nodes 
			//so we must keep queries for collections 
			//that are referenced in both subtrees. For each matching 
			//collection query we need to merge the refinement criteria
			mergeQueries(cont1, cont2, result);
			
		}
		
		return result;
	}
	
	private void concatQueries(GeoGcqlQueryContainer cont1,
			GeoGcqlQueryContainer cont2, GeoGcqlQueryContainer result) {
		//the result will contain the queries of the first subtree
		result.queries = cont1.queries;
		//we will loop through the queries of the second subtree, add the
		//ones that refer to a collection that is not met until now, and 
		//concatenate to the related list otherwise
		Iterator<Entry<String, HashMap<String, ArrayList<GeoGcqlCollectionQuery>>>> innerColIter
			= cont2.queries.entrySet().iterator();
		//for all the collections IDs
		while(innerColIter.hasNext()) {
			
			//if the result queries map does not contain a value for this colID 
			//create one
			Entry<String, HashMap<String, ArrayList<GeoGcqlCollectionQuery>>> current 
				= innerColIter.next();
			String colID = current.getKey();
			HashMap<String, ArrayList<GeoGcqlCollectionQuery>> outerLangMap = 
				result.queries.get(colID);
			if(outerLangMap == null)
			{
				outerLangMap = new HashMap<String, ArrayList<GeoGcqlCollectionQuery>>();
				result.queries.put(colID, outerLangMap);
			}
			
			//for all the languages of this collection ID
			Iterator<Entry<String, ArrayList<GeoGcqlCollectionQuery>>> innerLangIter
				= current.getValue().entrySet().iterator();
			while(innerLangIter.hasNext()) {
				
				//get the language and arraylist of queries
				Entry<String, ArrayList<GeoGcqlCollectionQuery>> currentQueries 
					= innerLangIter.next();
				String lang = currentQueries.getKey();
				ArrayList<GeoGcqlCollectionQuery> queryList = currentQueries.getValue();
				if(queryList == null)
					continue;
				
				//if the outer subtree has specified a query list for this collection ID and language
				ArrayList<GeoGcqlCollectionQuery> outerQueryList = outerLangMap.get(lang);
				if(outerQueryList == null)
				{
					outerQueryList = new ArrayList<GeoGcqlCollectionQuery>();
					outerLangMap.put(lang, outerQueryList);
				}
				outerQueryList.addAll(queryList);				
			}
		}
	}

	private GeoGcqlQueryContainer processNode(GCQLNotNode node, boolean not) throws Exception{
		
		//retrieve the query info from the left subtree
		GeoGcqlQueryContainer cont1 = processNode(node.left, not);
		
		//retrieve the query info from the right subtree
		//revert the not flag for the right tree
		GeoGcqlQueryContainer cont2 = processNode(node.right, !not);
		
		logger.debug("Not node - not: " + not);
		logNumberOfConditions(cont1);
		logNumberOfConditions(cont2);
		
		//the query info that will be returned 
		GeoGcqlQueryContainer result =  new GeoGcqlQueryContainer(null);
		
		if(!not) {
			//this is an NOT(AND-NOT) node and not flag is NOT raised 
			//so we must keep queries for collections 
			//that are referenced in both subtrees. For each matching 
			//collection query we need to merge the refinement criteria
			mergeQueries(cont1, cont2, result);
		} else {
			//in case not flag is raised, AND-NOT becomes the OR of NOT nodes 
			//this is an OR node so we must keep all queries, and for 
			//queries that refer to the same collection will be 
			//concatenated in the list for that collection
			concatQueries(cont1, cont2, result);
		}
		
		return result;
	}
	
	private void mergeQueries(GeoGcqlQueryContainer cont1, GeoGcqlQueryContainer cont2,
			GeoGcqlQueryContainer result) {
		Iterator<Entry<String, HashMap<String, ArrayList<GeoGcqlCollectionQuery>>>> outerColIter 
			= cont1.queries.entrySet().iterator();
		//for all the collection IDs 
		while(outerColIter.hasNext()) {
			
			Entry<String, HashMap<String, ArrayList<GeoGcqlCollectionQuery>>> outerColEntry 
				= outerColIter.next();
			String colID = outerColEntry.getKey();
			Iterator<Entry<String, ArrayList<GeoGcqlCollectionQuery>>> outerLangIter 
				= outerColEntry.getValue().entrySet().iterator();
			
			//for all the languages
			while(outerLangIter.hasNext()) {
				Entry<String, ArrayList<GeoGcqlCollectionQuery>> outerLangEntry 
					= outerLangIter.next();
				String lang = outerLangEntry.getKey();
				//get the set of queries for the first subtree, for this colID and language
				ArrayList<GeoGcqlCollectionQuery> outerQuerySet = outerLangEntry.getValue();				
				//if the set doesn't contain any query, go to the next language
				if(outerQuerySet == null || outerQuerySet.size() == 0)
					continue;
				
				//examine the queries(connected with OR) for this lang and collection ID in the other subtree.
				//We will merge the two sets of queries, by creating N*M new queries(connected with OR), where 
				//N and M are the number of queries in the two sets. 
				HashMap<String, ArrayList<GeoGcqlCollectionQuery>> inner = cont2.queries.get(colID);
				if(inner != null)
				{
					//get the set of queries for the second subtree, for this colID and language
					ArrayList<GeoGcqlCollectionQuery> innerQuerySet = inner.get(lang);
					//if the set doesn't contain any query, go to the next language
					if(innerQuerySet == null || innerQuerySet.size() == 0)
						continue;
					
					//create the set for the new queries
					ArrayList<GeoGcqlCollectionQuery> newQuerySet = new ArrayList<GeoGcqlCollectionQuery>();
					
					//for all the queries of the first set
					for(GeoGcqlCollectionQuery outerQuery : outerQuerySet)
					{
						//for all the queries of the secong set
						for(GeoGcqlCollectionQuery innerQuery : innerQuerySet)
						{
							//create an new collection query from the two current queries
							GeoGcqlCollectionQuery newQuery = mergeCollectionQueries(outerQuery, innerQuery);
							if(newQuery != null)
							{
								//add the new query to the set
								newQuerySet.add(newQuery);
							}
						}
					}
					
					//if we created at least one new query then add it to the new query container
					if(newQuerySet.size() > 0)
					{
						HashMap<String, ArrayList<GeoGcqlCollectionQuery>> langQueries = result.queries.get(colID);
						if(langQueries == null)
						{
							langQueries = new HashMap<String, ArrayList<GeoGcqlCollectionQuery>>();  
						}
						//this (colID, language) pair is visited for the first and last time so there
						//will be no old content in the HashMap
						langQueries.put(lang, newQuerySet);
						result.queries.put(colID, langQueries);
					}
				}
			}
		}
	}

	/**
	 * merges two queries
	 * @param outerQuery - the query from the left subtree
	 * @param innerQuery - the query from the right subtree
	 * @param not - indicates if the two queries and connected 
	 * with AND or NOT(AND-NOT)(refers to the right query)
	 * @return the merged query or null in case the two queries 
	 * are contradicting
	 */
	private GeoGcqlCollectionQuery mergeCollectionQueries(
			GeoGcqlCollectionQuery outerQuery,
			GeoGcqlCollectionQuery innerQuery) {
		
		//TODO: in this method we examine only a number of all the possible cases, 
		//in order to find out if the two queries are contradicting, or if we can
		//limit the searchPolygon(without just adding an extra refiner). This 
		//preprocessing may be very important in very complex queries. All the 
		//possible cases must be examined in the best way, in the future	
		
		boolean not = innerQuery.isNot();
		
		//first examine the AND case
		if(!not) {
			//case we have 2 contains
			if(outerQuery.getInclusion().equals(InclusionType.contains) 
					&& innerQuery.getInclusion().equals(InclusionType.contains)) {
				//get the intersection of the two search polygons
				Polygon newPoly = PolygonProcessing.intersection(outerQuery.getSearchPolygon(), 
						innerQuery.getSearchPolygon());
				//if the intersection is null then the queries are contradicting, return null
				if(newPoly == null)
					return null;
				
				//concatenate the refinement requests
				ArrayList<RefinementRequest> newRefReq = outerQuery.getRefineRequests();
				if(newRefReq == null)
					newRefReq = new ArrayList<RefinementRequest>();
				if(innerQuery.getRefineRequests() != null)
					newRefReq.addAll(innerQuery.getRefineRequests());
				
				//find out which will be the ranker used
				RankingRequest newRankReq = outerQuery.getRankRequest();
				if(newRankReq == null) {
					newRankReq = innerQuery.getRankRequest();//it is ok to be null
				} else {
					//if both are not null give a warning
					if(innerQuery.getRankRequest() != null) {
						//TODO: maybe we could add rules for which rankers are preferable
						logger.warn("Two rankers for the same collection query were found. The first one will be used");
					}
				}
				
				//Note that the not flag is only an indicator for the leaf queries. 
				//Merged queries can only have false for this flag
				return new GeoGcqlCollectionQuery(InclusionType.contains, newPoly, newRefReq, newRankReq, false);
					
			}
			
			//case we have 2 inside
			if(outerQuery.getInclusion().equals(InclusionType.inside) 
					&& innerQuery.getInclusion().equals(InclusionType.inside)) {
				//get the intersection of the two search polygons
				Polygon newPoly = PolygonProcessing.intersection(outerQuery.getSearchPolygon(), 
						innerQuery.getSearchPolygon());
				//if the intersection is null then we will have to add a new refiner, since the union does not
				//form one polygon
				ArrayList<RefinementRequest> newRefReq = new ArrayList<RefinementRequest>();
				if(newPoly == null)
				{
					newPoly = outerQuery.getSearchPolygon();
					newRefReq.add(new RefinementRequest(innerQuery.getSearchPolygon(), POLYGON_REFINER, null, InclusionType.inside, not));
				//get the union of the 2 polygons 
				} else {
					newPoly = PolygonProcessing.union(outerQuery.getSearchPolygon(), 
							innerQuery.getSearchPolygon());
				}
					
				//add all the refinement requests
				if(outerQuery.getRefineRequests() != null)
					newRefReq.addAll(outerQuery.getRefineRequests());
				if(innerQuery.getRefineRequests() != null)
					newRefReq.addAll(innerQuery.getRefineRequests());
				
				//find out which will be the ranker used
				RankingRequest newRankReq = outerQuery.getRankRequest();
				if(newRankReq == null) {
					newRankReq = innerQuery.getRankRequest();//it is ok to be null
				} else {
					//if both are not null give a warning
					if(innerQuery.getRankRequest() != null) {
						//TODO: maybe we could add rules for which rankers are preferable
						logger.warn("Two rankers for the same collection query were found. The first one will be used");
					}
				}
				
				//Note that the not flag is only an indicator for the leaf queries. 
				//Merged queries can only have false for this flag
				return new GeoGcqlCollectionQuery(InclusionType.inside, newPoly, newRefReq, newRankReq, false);
			}
			
			//case we have a contains and an inside
			if((outerQuery.getInclusion().equals(InclusionType.contains) 
					&& innerQuery.getInclusion().equals(InclusionType.inside))
					||
					(outerQuery.getInclusion().equals(InclusionType.inside) 
							&& innerQuery.getInclusion().equals(InclusionType.contains))) {
				//we will just check if the queries are contradicting
				GeoGcqlCollectionQuery insideQuery = null;
				GeoGcqlCollectionQuery containsQuery = null;
				if(innerQuery.getInclusion().equals(InclusionType.inside)) {
					insideQuery = innerQuery;
					containsQuery = outerQuery;
				} else {
					insideQuery = outerQuery;
					containsQuery = innerQuery;
				}
				//inside must be contained in contains
				if(!PolygonProcessing.isContained(insideQuery.getSearchPolygon(), containsQuery.getSearchPolygon()))
				{
					return null;
				}
			}
			
			//case we have a contains and an intersect
			if((outerQuery.getInclusion().equals(InclusionType.contains) 
					&& innerQuery.getInclusion().equals(InclusionType.intersect))
					||
					(outerQuery.getInclusion().equals(InclusionType.intersect) 
							&& innerQuery.getInclusion().equals(InclusionType.contains))) {
				
				//we will just check if the queries are contradicting
				//intersect must overlap with contains
				if(!PolygonProcessing.overlaps(innerQuery.getSearchPolygon(), outerQuery.getSearchPolygon()))
				{
					return null;
				}				
			}
				
			//if we haven't returned a merged or a null query, and in all the other cases except for those examined above,
			//we will just add a new refiner for the second query
			return newQueryWithAddedRefiner(innerQuery, outerQuery);
			
		//AND-NOT case	
		} else {
		
			//case we have 2 contains
			if(outerQuery.getInclusion().equals(InclusionType.contains) 
					&& innerQuery.getInclusion().equals(InclusionType.contains)) {
				//not refers to the inner query. If the not-contains(inner) query
				//contains the contains-query return null
				if(PolygonProcessing.isContained(outerQuery.getSearchPolygon(), innerQuery.getSearchPolygon()))
				{
					return null;
				}
			}
			
			//case we have 2 inside
			if(outerQuery.getInclusion().equals(InclusionType.inside) 
					&& innerQuery.getInclusion().equals(InclusionType.inside)) {
				//not refers to the inner query. If the not-inside(inner) query
				//is contained in the inside-query return null
				if(PolygonProcessing.isContained(innerQuery.getSearchPolygon(), outerQuery.getSearchPolygon()))
				{
					return null;
				}
			}
			
			//case we have 2 intersect
			if(outerQuery.getInclusion().equals(InclusionType.intersect) 
					&& innerQuery.getInclusion().equals(InclusionType.intersect)) {
				//not refers to the inner query. If the not-intersect(inner) query
				//contains the intersect-query return null
				if(PolygonProcessing.isContained(outerQuery.getSearchPolygon(), innerQuery.getSearchPolygon()))
				{
					return null;
				}
			}
			
			return newQueryWithAddedRefiner(innerQuery, outerQuery);
		}
	}
	
	private GeoGcqlCollectionQuery newQueryWithAddedRefiner(GeoGcqlCollectionQuery innerQuery, 
			GeoGcqlCollectionQuery outerQuery) {
		
		boolean not = innerQuery.isNot();
		
		ArrayList<RefinementRequest> newRefReq = new ArrayList<RefinementRequest>();
		newRefReq.add(new RefinementRequest(innerQuery.getSearchPolygon(), POLYGON_REFINER, null, innerQuery.getInclusion(), not));
		if(outerQuery.getRefineRequests() != null)
			newRefReq.addAll(outerQuery.getRefineRequests());
		
		//in case we have an AND-NOT flag we must revert the refiners of the inner query
		if(not && innerQuery.getRefineRequests()!=null) {
			for(RefinementRequest req : innerQuery.getRefineRequests())
				req.revertNot();			
		}
		if(innerQuery.getRefineRequests() != null)
			newRefReq.addAll(innerQuery.getRefineRequests());
		
		//find out which will be the ranker used
		RankingRequest newRankReq = outerQuery.getRankRequest();
		if(newRankReq == null) {
			newRankReq = innerQuery.getRankRequest();//it is ok to be null
		} else {
			//if both are not null give a warning
			if(innerQuery.getRankRequest() != null) {
				//TODO: maybe we could add rules for which rankers are preferable
				logger.warn("Two rankers for the same collection query were found. The first one will be used");
			}
		}
		
		//Note that the not flag is only an indicator for the leaf queries. 
		//Merged queries can only have false for this flag
		return new GeoGcqlCollectionQuery(outerQuery.getInclusion(), outerQuery.getSearchPolygon(), newRefReq, newRankReq, false);
	}
	
	private void logNumberOfConditions(GeoGcqlQueryContainer queryContainer) {
		for(Entry<String, HashMap<String, ArrayList<GeoGcqlCollectionQuery>>> currentColEntry : queryContainer.queries.entrySet()) {
			logger.debug(" Collection: " + currentColEntry.getKey());
			for(Entry<String, ArrayList<GeoGcqlCollectionQuery>> langEntries : currentColEntry.getValue().entrySet()) {
				logger.debug(" Language: " + langEntries.getKey());
				logger.debug("Number of queries: " + langEntries.getValue().size());
			}
		}
	}
}
