package org.gcube.indexmanagement.common;

/**
 * Represent the characteristics of a single field in a forward index RowSet.
 */
public class ForwardIndexField {

    /** The name of the field */
    public String name;
    
    /** The size of the field */
    public int size;
    
    /** The data type of the field */
    public DataType dataType;
    
    /** The format of the data type (if applicable) **/
    public String dataTypeFormat;
    
    /**
     * An enumeration of accepted data types, along with their respective Class
     * objects and default byte size
     */
    public enum DataType {
        /** 
         * The INT data type
         */
        INT(Integer.class, 4), 
        /** 
         * The DATE data type.
         */
        DATE (Long.class, 8), 
        /** 
         * The FLOAT data type.
         */
        FLOAT(Float.class, 4), 
        /** 
         * The STRING data type.
         */
        STRING(String.class, 40);
        
        /** 
         *The Class of this DataType instance 
         */
        private final Class typeClass;
        
        /** The default size */
        private final int defaultSize;
        
        /** Private constructor, used to create the accepted data types */
        private DataType(Class typeClass, int defaultSize) {
            this.typeClass = typeClass;
            this.defaultSize = defaultSize;
        }
        
        /** A case insensitive version of valueOf(String).
         * @param name - the string value to gte the type for
         * @return DataType of the input string name.
         * @throws IndexException - in case of failure.
         */
        public static DataType getByName(String name) throws Exception {
            for (DataType type : DataType.values()) {
                if (name.equalsIgnoreCase(type.toString())) {
                    return type;
                }
            }
            throw new Exception(
                    "specified DataTypeName does not match any DataTypes: \""
                            + name + "\"");
        }

        /**
         * Getter method for the Class object of this data type
         * 
         * @return the Class object of this data type
         */
        public Class getTypeClass() {
            return typeClass;
        }

        /**
         * Getter method for the default byte size of this data type
         * 
         * @return the default byte size of this data type
         */
        public int getDefaultSize() {
            return defaultSize;
        }

    }

    /**
     * Empty constructor. The field attributes must be set in subsequent calls.
     */
    public ForwardIndexField() {
    }

    /**
     * Constructor setting all field attributes.
     * 
     * @param name -
     *            The name of the field to be created
     * @param dataType -
     *            The data type of the field to be created
     */
    public ForwardIndexField(String name, DataType dataType) {
        this.name = name;
        this.dataType = dataType;
        this.size = dataType.getDefaultSize();
    }

    /**
     * Constructor setting all field attributes. Sets the data type to
     * DataType.STRING with the specified byte size.
     * 
     * @param name -
     *            The name of the field to be created
     * @param stringDataTypeSize -
     *            The byte size of the field to be created
     */
    public ForwardIndexField(String name, int stringDataTypeSize) {
        this.name = name;
        this.dataType = DataType.STRING;
        this.size = stringDataTypeSize;
    }

    /**
     * Returns the field name
     * 
     * @return the field name
     */
    public String getName() {
    	return name;
    }
    
    /**
     * Returns the data type in use
     * 
     * @return the data type
     */
    public DataType getDataType() {
    	return dataType;
    }
    
    /**
     * Returns the format of the data type in use
     * 
     * @return the data type format
     */
    public String getDataTypeFormat() {
    	return dataTypeFormat;
    }
    
    /**
     * {@inheritDoc}
     */
    public String toString() {
        return name + "( dataType="
                + dataType + " class=" + dataType.getTypeClass().getName()
                + " size=" + size + " )";
    }
}
