package elasticsearchindex;

import static org.elasticsearch.node.NodeBuilder.nodeBuilder;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import org.elasticsearch.ElasticSearchException;
import org.elasticsearch.action.admin.cluster.state.ClusterStateResponse;
import org.elasticsearch.action.admin.indices.create.CreateIndexResponse;
import org.elasticsearch.action.admin.indices.mapping.put.PutMappingResponse;
import org.elasticsearch.action.bulk.BulkRequestBuilder;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.index.IndexRequestBuilder;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.Client;
import org.elasticsearch.client.IndicesAdminClient;
import org.elasticsearch.cluster.metadata.MappingMetaData;
import org.elasticsearch.common.collect.ImmutableMap;
import org.elasticsearch.common.settings.ImmutableSettings;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.node.Node;
import org.elasticsearch.search.SearchHit;
import org.gcube.common.core.contexts.GHNContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import elasticsearchindex.components.FullTextIndexType;
import elasticsearchindex.components.IndexException;
import elasticsearchindex.components.IndexField;
import elasticsearchindex.components.IndexType;
import elasticsearchindex.components.LanguageIdPlugin;
import elasticsearchindex.helpers.ElasticSearchHelper;
import elasticsearchindex.helpers.QueryParser;
import elasticsearchindex.helpers.RowsetParser;
import gr.uoa.di.madgik.commons.server.ConnectionManagerConfig;
import gr.uoa.di.madgik.commons.server.PortRange;
import gr.uoa.di.madgik.commons.server.TCPConnectionManager;
import gr.uoa.di.madgik.grs.buffer.GRS2BufferException;
import gr.uoa.di.madgik.grs.buffer.IBuffer.Status;
import gr.uoa.di.madgik.grs.events.KeyValueEvent;
import gr.uoa.di.madgik.grs.proxy.tcp.TCPConnectionHandler;
import gr.uoa.di.madgik.grs.proxy.tcp.TCPStoreConnectionHandler;
import gr.uoa.di.madgik.grs.reader.ForwardReader;
import gr.uoa.di.madgik.grs.reader.GRS2ReaderException;
import gr.uoa.di.madgik.grs.record.GRS2RecordDefinitionException;
import gr.uoa.di.madgik.grs.record.GenericRecord;
import gr.uoa.di.madgik.grs.record.Record;
import gr.uoa.di.madgik.grs.writer.GRS2WriterException;
import gr.uoa.di.madgik.grs.writer.RecordWriter;

public class FullTextNode {
	private static final Logger logger = LoggerFactory.getLogger(FullTextNode.class);

	private static int DEFAULT_NUM_OF_REPLICAS = 0;
	private static int DEFAULT_NUM_OF_SHARDS = 1;
	private static int MAX_FRAGMENT_CNT = 5;
	private static int MAX_FRAGMENT_SIZE = 150;

	private static final long RSTIMEOUT = 5;
	private static String ALL_INDEXES = "allIndexes";
	public static String META_INDEX = "meta-index";

	private Client indexClient;
	private Node indexNode;
	private Set<String> indexTypes = new HashSet<String>();
	private FTNodeCache cache;

	private String clusterName;
	private String indexName;
	private Integer noOfReplicas;
	private Integer noOfShards;
	private String scope;
	private int maxFragmentCnt;
	private int maxFragmentSize;

	private HashMap<String, FullTextIndexType> colForField = new HashMap<String, FullTextIndexType>();

	// Constructors

	public FullTextNode() {
		logger.info("Initializing FullTextNode");
		this.cache = new FTNodeCache();
		this.initialize();
	}

	public FullTextNode(String clusterName, String indexName, Integer noOfReplicas, Integer noOfShards, String scope, int maxFragmentCnt, int maxFragmentSize) {
		this();
		this.clusterName = clusterName;
		this.indexName = indexName;
		this.noOfReplicas = noOfReplicas;
		this.noOfShards = noOfShards;
		this.scope = scope;
		this.maxFragmentCnt = maxFragmentCnt;
		this.maxFragmentSize = maxFragmentSize;
	}

	public FullTextNode(String clusterName, String indexName, String scope) {
		this();
		this.clusterName = clusterName;
		this.indexName = indexName;
		this.noOfReplicas = DEFAULT_NUM_OF_REPLICAS;
		this.noOfShards = DEFAULT_NUM_OF_SHARDS;
		this.scope = scope;
		this.maxFragmentCnt = MAX_FRAGMENT_CNT;
		this.maxFragmentSize = MAX_FRAGMENT_SIZE;
	}

	// End of Constructors

	public String getClusterName() {
		return clusterName;
	}

	public String getIndexName() {
		return indexName;
	}

	public Integer getNoOfReplicas() {
		return noOfReplicas;
	}

	public Integer getNoOfShards() {
		return noOfShards;
	}

	public String getScope() {
		return scope;
	}

	public int getMaxFragmentCnt() {
		return maxFragmentCnt;
	}

	public int getMaxFragmentSize() {
		return maxFragmentSize;
	}
	
	public Client getIndexClient()
	{
		return indexClient;
	}

	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("FullTextNode [indexNode=").append(indexNode).append(", cache=").append(cache).append(", clusterName=").append(clusterName)
				.append(", indexName=").append(indexName).append(", noOfReplicas=").append(noOfReplicas).append(", noOfShards=").append(noOfShards)
				.append(", scope=").append(scope).append(", maxFragmentCnt=").append(maxFragmentCnt).append(", maxFragmentSize=").append(maxFragmentSize)
				.append("]");
		return builder.toString();
	}

	public synchronized void addIndexType(String indexTypeStr) throws Exception {
		FullTextIndexType indexType = QueryParser.retrieveIndexType(indexTypeStr, this.scope, this.cache);
		addIndexType(indexTypeStr, indexType);
	}

	public synchronized void addIndexType(String indexTypeStr, FullTextIndexType indexType) throws Exception {
		logger.info("Calling addIndexType");
		if (indexType == null) {
			logger.warn("IndexType is null");
			throw new Exception("Trying to null as IndexType. Check how you got it");
		}
		
		// in case indexType is not in cache
		indexType = QueryParser.addFullTextIndexTypeIntoCache(indexTypeStr, this.scope, indexType, this.cache);

		// create mapping if not exists
		IndicesAdminClient iac = this.indexClient.admin().indices();

		logger.info("Checking if index exists");
		if (iac.prepareExists(this.indexName).execute().actionGet().exists()) {
			logger.info("Index already exists");
		} else {
			CreateIndexResponse cir = iac.prepareCreate(this.indexName).execute().actionGet();
			logger.info("Create Index Response : " + cir);
		}

		logger.info("Index Type");
		logger.info("-----------------------------------------------");
		logger.info(indexType.toString());
		logger.info("-----------------------------------------------");
		
		Map<String, Object> mapping = new HashMap<String, Object>();
		List<String> presentables = new ArrayList<String>();

		for (IndexField idxTypeField : indexType.getFields()) {
			String store, index;

			store = idxTypeField.store ? "yes" : "no";

			if (idxTypeField.index)
				index = idxTypeField.tokenize ? "analyzed" : "not_analyzed";
			else
				index = "no";

			if (idxTypeField.name.equalsIgnoreCase(IndexType.COLLECTION_FIELD))
				index = "not_analyzed";

			Map<String, Object> fieldMap = new HashMap<String, Object>();
			fieldMap.put("type", "string");
			fieldMap.put("store", store);
			fieldMap.put("index", index);

			mapping.put(idxTypeField.name, fieldMap);

			if (idxTypeField.returned && !idxTypeField.name.equalsIgnoreCase(IndexType.COLLECTION_FIELD))
				presentables.add(idxTypeField.name);
		}

		this.cache.presentableFieldsPerIndexType.put(indexTypeStr, presentables);

		Map<String, Object> propertyMap = new HashMap<String, Object>();
		propertyMap.put("properties", mapping);

		// /logger.info("propertyMap : " + propertyMap);
		Map<String, Object> mappingMap = new HashMap<String, Object>();
		mappingMap.put(indexTypeStr, propertyMap);

		String json = ElasticSearchHelper.createJSONObject(mappingMap).string();
		logger.info("json : " + json);

		PutMappingResponse pmr = iac.preparePutMapping().setIndices(this.indexName).setType(indexTypeStr).setSource(json).execute().actionGet();
		logger.info("Update Settings Response : " + pmr.toString());

		this.indexTypes.add(indexTypeStr);
	}

	public void createOrJoinCluster() {
		logger.info("creating or joining cluster");
		logger.info("cluster.name : " + clusterName);
		logger.info("index.number_of_replicas : " + noOfReplicas);
		logger.info("index.number_of_shards : " + noOfShards);

		Settings settings = ImmutableSettings.settingsBuilder().put("cluster.name", clusterName).put("index.number_of_replicas", String.valueOf(noOfReplicas))
				.put("index.number_of_shards", String.valueOf(noOfShards))
				// .put("discovery.zen.ping.multicast.ping.enabled", false)
				// .put("discovery.zen.ping.multicast.enabled",false)
				// .put("discovery.zen.ping.unicast.enabled",true)
				// .put("discovery.zen.ping.unicast.hosts ",
				// "jazzman.di.uoa.gr")
				.build();

		this.indexNode = nodeBuilder().settings(settings).node();
		this.indexClient = this.indexNode.client();
	}

	/***
	 * 
	 * @param knownNodes
	 *            Map of Address -> Port
	 * @throws IOException
	 */
	public void joinCluster(Map<String, Integer> knownNodes) throws IOException {
		logger.info("joining cluster of known node : " + knownNodes);
		logger.info("cluster.name : " + clusterName);
		logger.info("index.number_of_replicas : " + noOfReplicas);
		logger.info("index.number_of_shards : " + noOfShards);

		String hosts = ElasticSearchHelper.createKnownHostsString(knownNodes);

		logger.info("hosts string : " + hosts);

		Settings settings = ImmutableSettings.settingsBuilder().put("cluster.name", clusterName).put("index.number_of_replicas", String.valueOf(noOfReplicas))
				.put("index.number_of_shards", String.valueOf(noOfShards)).put("discovery.zen.ping.multicast.ping.enabled", false)
				.put("discovery.zen.ping.multicast.enabled", false).put("discovery.zen.ping.unicast.enabled", true)
				.put("discovery.zen.ping.unicast.hosts", hosts).build();

		this.indexNode = nodeBuilder().settings(settings).node();
		this.indexClient = this.indexNode.client();
	}

	/**
	 * 
	 * @param queryString
	 * @return
	 * @throws IndexException
	 * @throws GRS2WriterException
	 */
	public String query(String queryString) throws IndexException, GRS2WriterException {
		return query(queryString, -1);
	}

	/**
	 * Executes the query of gRS2 locator of maximum maxHits records (if >0)
	 * 
	 * @param queryString
	 * @param maxHits
	 * @return gRS2 locator of the results
	 * @throws IndexException
	 * @throws GRS2WriterException
	 */
	public String query(String queryString, int maxHits) throws GRS2WriterException, IndexException {
		logger.info("queryString received : " + queryString);

		final List<String> collIDs = QueryParser.getCollectionsIDFromQuery(queryString);
		logger.info("collectionID of query : " + collIDs);

		final Set<String> indexTypes = QueryParser.getIndexTypesByCollectionIDs(this.cache.indexTypesByCollIDs, collIDs, this.indexClient, this.indexName);
		logger.info("indexTypes for collectionIDs : " + indexTypes);

		final List<String> presentables = QueryParser.createPresentableForIndexTypes(this.cache.presentableFieldsPerIndexType, indexTypes);
		logger.info("presentables for index types : " + presentables);

		final List<String> projections = QueryParser.getProjectionsQueryFromQueryString(queryString);
		logger.info("projections of queryString : " + projections);

		final String query = QueryParser.getLuceneQueryFromQueryString(queryString);
		logger.info("query part of queryString : " + query);

		final SearchHit[] hits = projections.contains(IndexType.SNIPPET) ? ElasticSearchHelper.queryElasticSearch(this.indexClient, this.indexName, query,
				maxHits, presentables, this.maxFragmentSize, this.maxFragmentCnt) : ElasticSearchHelper.queryElasticSearch(this.indexClient, this.indexName,
				query, maxHits);
		logger.info("Number of hits returned by index : " + hits.length);

		final RecordWriter<GenericRecord> rsWriter = QueryParser.initRSWriterForSearchHits(presentables, projections);
		// send an event for the total number of results

		logger.info("emitting key value event with key : " + IndexType.RESULTSNOFINAL_EVENT + " and value : " + hits.length);

		rsWriter.emit(new KeyValueEvent(IndexType.RESULTSNOFINAL_EVENT, String.valueOf(hits.length)));

		final int max_fragmet_cnt = this.maxFragmentCnt;
		Runnable writerRun = new Runnable() {
			public void run() {
				try {
					for (SearchHit hit : hits) {
						// while the reader hasn't stopped reading
						if (!QueryParser.writeSearchHitInResultSet(hit, rsWriter, projections, presentables, max_fragmet_cnt, RSTIMEOUT))
							break;
					}
					if (rsWriter.getStatus() != Status.Dispose)
						rsWriter.close();
				} catch (Exception e) {
					logger.error("Error during search.", e);
					try {
						if (rsWriter.getStatus() != Status.Dispose)
							rsWriter.close();
					} catch (Exception ex) {
						logger.error("Error while closing RS writer.", ex);
					}
				}
			}
		};
		new Thread(writerRun).start();

		logger.info("results locator : " + rsWriter.getLocator());

		return rsWriter.getLocator().toString();
	}

	public String queryStream(String queryString) throws IndexException, GRS2WriterException {
		return queryStream(queryString, -1);
	}

	/**
	 * Important: The returned results are not sorted!
	 * 
	 * @param queryString
	 * @return
	 * @throws IndexException
	 * @throws GRS2WriterException
	 */
	public String queryStream(String queryString, final int maxHits) throws IndexException, GRS2WriterException {
		logger.info("queryString received : " + queryString);

		final List<String> collIDs = QueryParser.getCollectionsIDFromQuery(queryString);
		logger.info("collectionID of query : " + collIDs);

		final Set<String> indexTypes = QueryParser.getIndexTypesByCollectionIDs(this.cache.indexTypesByCollIDs, collIDs, this.indexClient, this.indexName);
		logger.info("indexTypes for collectionIDs : " + indexTypes);

		final List<String> presentables = QueryParser.createPresentableForIndexTypes(this.cache.presentableFieldsPerIndexType, indexTypes);
		logger.info("presentables for index types : " + presentables);

		final List<String> projections = QueryParser.getProjectionsQueryFromQueryString(queryString);
		logger.info("projections of queryString : " + projections);

		final String query = QueryParser.getLuceneQueryFromQueryString(queryString);
		logger.info("query part of queryString : " + query);

		long numberOfHits = ElasticSearchHelper.queryCountElasticSearch(this.indexClient, this.indexName, query);

		if (maxHits > 0 && numberOfHits > maxHits)
			numberOfHits = maxHits;

		logger.info("Number of hits returned by index : " + numberOfHits);

		final RecordWriter<GenericRecord> rsWriter = QueryParser.initRSWriterForSearchHits(presentables, projections);
		// send an event for the total number of results

		logger.info("emitting key value event with key : " + IndexType.RESULTSNOFINAL_EVENT + " and value : " + numberOfHits);

		rsWriter.emit(new KeyValueEvent(IndexType.RESULTSNOFINAL_EVENT, String.valueOf(numberOfHits)));

		final Client client = this.indexClient;
		final String indexName = this.indexName;
		final int max_fragment_size = this.maxFragmentSize;
		final int max_fragment_cnt = this.maxFragmentCnt;

		Runnable writerRun = new Runnable() {

			public void run() {
				try {
					SearchResponse scrollResp = projections.contains(IndexType.SNIPPET) ? ElasticSearchHelper.queryElasticSearchScroll(client, indexName,
							query, maxHits, presentables, max_fragment_size, max_fragment_cnt) : ElasticSearchHelper.queryElasticSearchScroll(client,
							indexName, query, maxHits);

					int hits = 0;

					while (true) {
						scrollResp = ElasticSearchHelper.getNextSearchResponse(client, scrollResp);
						logger.info("hits from scroll : " + scrollResp.hits().hits().length);

						for (SearchHit hit : scrollResp.hits()) {
							// while the reader hasn't stopped reading
							if (!QueryParser.writeSearchHitInResultSet(hit, rsWriter, projections, presentables, max_fragment_cnt, RSTIMEOUT))
								break;
						}
						if (rsWriter.getStatus() != Status.Dispose)
							rsWriter.close();

						if (scrollResp.hits().hits().length == 0) {
							break;
						}
						hits++;
						if (hits > maxHits)
							break;
					}
				} catch (Exception e) {
					logger.error("Error during search.", e);
					try {
						if (rsWriter.getStatus() != Status.Dispose)
							rsWriter.close();
					} catch (Exception ex) {
						logger.error("Error while closing RS writer.", ex);
					}
				}
			}
		};
		new Thread(writerRun).start();

		logger.info("results locator : " + rsWriter.getLocator());

		return rsWriter.getLocator().toString();
	}

	/**
	 * Deletes the documents with IDs within the list <a>docIDs</a>
	 * 
	 * @param docIDs
	 */
	public void deleteDocuments(List<String> docIDs) {
		ElasticSearchHelper.deleteDocuments(this.indexClient, this.indexName, docIDs);
	}

	/**
	 * Commits the changes to the index by sending a flush and a refresh request
	 */
	public void commitMeta() {
		ElasticSearchHelper.commit(this.indexClient, META_INDEX);
	}
	
	public void commit() {
		ElasticSearchHelper.commit(this.indexClient, this.indexName);
	}

	public void clearIndex(String indexTypeName) {
		ElasticSearchHelper.clearIndex(this.indexClient, this.indexName, indexTypeName);
	}

	/**
	 * Closes the index
	 */
	public void close() {
		this.indexClient.close();
		this.indexNode.close();
	}

	private void initialize() {
		TCPConnectionManager.Init(new ConnectionManagerConfig(GHNContext.getContext().getHostname(), new ArrayList<PortRange>(), true));
		logger.info("Added hostname: "+GHNContext.getContext().getHostname());
		TCPConnectionManager.RegisterEntry(new TCPConnectionHandler());
		TCPConnectionManager.RegisterEntry(new TCPStoreConnectionHandler());
	}

	/**
	 * Feeds the index with a rowset.
	 * 
	 * @param rowset
	 * @return <a>True</a> on success or <a>False</a> on failure
	 */
	public boolean feedRowset(String rowset) {
		BulkRequestBuilder bulkRequest = this.indexClient.prepareBulk();
		HashMap<String, FullTextIndexType> colForIdTypes = new HashMap<String, FullTextIndexType>();
		boolean status = feedRowset(bulkRequest, rowset, colForIdTypes);

		if (status) {
			long before = System.currentTimeMillis();
			BulkResponse bulkResponse = bulkRequest.execute().actionGet();
			long after = System.currentTimeMillis();
			logger.info("Time for commiting the bulk requests : " + (after - before) / 1000.0 + " secs");
			logger.info("bulkResponse : " + bulkResponse.tookInMillis() / 1000.0 + " secs");
			logger.info("bulkResponse : " + bulkResponse);
			if (bulkResponse.hasFailures()) {
				logger.info("failures have happened");
			}
			this.commit();

		} else {
			logger.info("feedRowset failed");
		}

		return status;
	}

	public boolean feedLocator(String resultSetLocation) throws GRS2ReaderException, URISyntaxException, GRS2RecordDefinitionException, GRS2BufferException {
		return feedLocator(resultSetLocation, colForField);
	}

	/**
	 * Feeds the index with rowsets that are read from the the given locator
	 * 
	 * @param resultSetLocation
	 * @param colForField
	 * @return
	 * @throws GRS2ReaderException
	 * @throws URISyntaxException
	 * @throws GRS2RecordDefinitionException
	 * @throws GRS2BufferException
	 */
	public boolean feedLocator(String resultSetLocation, HashMap<String, FullTextIndexType> colForField) throws GRS2ReaderException, URISyntaxException,
			GRS2RecordDefinitionException, GRS2BufferException {
		// UpdaterService

		long beforeFeed, afterFeed;

		logger.info("Initializing reader at resultset : " + resultSetLocation);
		ForwardReader<Record> reader = new ForwardReader<Record>(new URI(resultSetLocation));
		reader.setIteratorTimeout(RSTIMEOUT);
		reader.setIteratorTimeUnit(TimeUnit.MINUTES);

		int rowSetCount = 0;
		boolean success = true;

		beforeFeed = System.currentTimeMillis();

		BulkRequestBuilder bulkRequest = this.indexClient.prepareBulk();

		// HashMap<String, FullTextIndexType> colForField = new HashMap<String,
		// FullTextIndexType>();

		try {
			logger.info("Initializing resultset reader iterator");
			Iterator<Record> it = reader.iterator();
			long before, after;

			while (it.hasNext()) {
				logger.info("Getting result : " + rowSetCount);
				// System.out.println("Getting result : " + rowSetCount);

				before = System.currentTimeMillis();
				Record result = it.next();
				after = System.currentTimeMillis();
				logger.info("Time for getting record from Result Set : " + (after - before) / 1000.0 + " secs");
				// logger.info("Result : " + i + " : " + result);

				before = System.currentTimeMillis();
				String rowset = RowsetParser.getRowsetFromResult(result);
				after = System.currentTimeMillis();
				logger.info("Time for getting rowset from record : " + (after - before) / 1000.0 + " secs");
				// logger.info("Result rowset : " + rowset);
				success = feedRowset(bulkRequest, rowset, colForField);

				if (success == false) {
					logger.info("feed rowset failed");
					break;
				}

				logger.info("Result " + rowSetCount + " inserted");
				rowSetCount++;
			}
		} catch (Exception e) {
			logger.info("Exception will feeding", e);
		}

		reader.close();

		if (success) {
			long before = System.currentTimeMillis();
			BulkResponse bulkResponse = bulkRequest.execute().actionGet();
			long after = System.currentTimeMillis();
			logger.info("Time for commiting the bulk requests : " + (after - before) / 1000.0 + " secs");
			logger.info("bulkResponse : " + bulkResponse.tookInMillis() / 1000.0 + " secs");
			logger.info("bulkResponse : " + bulkResponse);
			if (bulkResponse.hasFailures()) {
				logger.info("failures have happened");
			}
			this.commit();
			updateManagerProperties(colForField);
		}
		logger.info("Total number of records feeded : " + rowSetCount);

		afterFeed = System.currentTimeMillis();
		logger.info("Total insert time : " + (afterFeed - beforeFeed) / 1000.0);
		System.out.println("Total insert time : " + (afterFeed - beforeFeed) / 1000.0);

		return success;
	}

	// ////////////////////////////////////
	// / private methods
	// ////////////////////////////////////

	/**
	 * Adds a rowset in the bulk of a bulkRequest
	 * 
	 * @param bulkRequest
	 * @param rowset
	 * @param colForIdTypes
	 * @return <a>True</a> on success or <a>False</a> on failure
	 */
	private boolean feedRowset(BulkRequestBuilder bulkRequest, String rowset, HashMap<String, FullTextIndexType> colForIdTypes) {
		long before, after;

		before = System.currentTimeMillis();
		String rsIdxTypeID = RowsetParser.getIdxTypeNameRowset(this.indexName, rowset);
		after = System.currentTimeMillis();
		logger.info("Time for getting rsIdxTypeID from rowset : " + (after - before) / 1000.0 + " secs");
		logger.info("Result IndexTypeID : " + rsIdxTypeID);

		before = System.currentTimeMillis();
		String lang = RowsetParser.getLangRowset(this.indexName, rowset);
		after = System.currentTimeMillis();
		logger.info("Time for getting lang from rowset : " + (after - before) / 1000.0 + " secs");

		logger.info("Result lang : " + lang);
		// if no language is detected set the unknown
		if (lang == null || lang.equals(""))
			lang = LanguageIdPlugin.LANG_UNKNOWN;

		before = System.currentTimeMillis();
		String colID = RowsetParser.getColIDRowset(this.indexName, rowset);
		after = System.currentTimeMillis();
		logger.info("Time for colID lang from rowset : " + (after - before) / 1000.0 + " secs");

		logger.info("Result colID : " + colID);
		// if no collection ID is detected we have a problem and the
		// update must be cancelled
		if (colID == null || colID.equals("")) {
			logger.error("No collection ID given in ROWSET: " + rowset);
			return false;
		}

		try {
			addIndexType(rsIdxTypeID);
		} catch (Exception e) {
			logger.warn("Add index type exception", e);
			System.out.println("Add index type exception");
			e.printStackTrace();
			return false;
		}
		// check if the same indexTypeID is specified
		if (rsIdxTypeID == null || !this.indexTypes.contains(rsIdxTypeID)) {
			logger.error("IndexType : " + rsIdxTypeID + " not in indexTypes : " + indexTypes);
			return false;
		}

		before = System.currentTimeMillis();
		String alteredRowset = RowsetParser.preprocessRowset(rowset, lang, colID, this.indexName, rsIdxTypeID, this.scope, this.cache);
		after = System.currentTimeMillis();
		logger.info("Time for preprocessRowset : " + (after - before) / 1000.0 + " secs");
		// logger.info("Result alteredRowset : " + alteredRowset);

		if (alteredRowset == null) {
			logger.error("could not preprocess rowset: " + rowset);
			return false;
		}
		before = System.currentTimeMillis();

		FullTextIndexType idxType = this.cache.cachedIndexTypes.get(QueryParser.createIndexTypekey(rsIdxTypeID, this.scope));
		logger.info("index type for name : " + idxType);
		logger.info("all indexTypes in cache : " + this.cache.cachedIndexTypes.keySet());

		ElasticSearchHelper.insertRowSet(bulkRequest, this.indexClient, this.indexName, idxType, this.indexTypes, alteredRowset);
		after = System.currentTimeMillis();
		logger.info("Time for insertRowSet : " + (after - before) / 1000.0 + " secs");

		colForIdTypes.put(colID + IndexType.SEPERATOR_FIELD_INFO + lang, idxType);

		return true;
	}

	/**
	 * Used to refresh the indexType and the presentable fields per index type.
	 * Usually called after invalidation
	 * 
	 * @param indexTypeStr
	 */
	private synchronized void bindIndexType(String indexTypeStr) {

		FullTextIndexType indexType = QueryParser.retrieveIndexType(indexTypeStr, this.scope, this.cache);

		logger.info("Index Type");
		logger.info("-----------------------------------------------");
		logger.info(indexType.toString());
		logger.info("-----------------------------------------------");
		List<String> presentables = new ArrayList<String>();

		for (IndexField idxTypeField : indexType.getFields()) {
			if (idxTypeField.returned && !idxTypeField.name.equalsIgnoreCase(IndexType.COLLECTION_FIELD))
				presentables.add(idxTypeField.name);
		}

		this.indexTypes.add(indexTypeStr);
		this.cache.presentableFieldsPerIndexType.put(indexTypeStr, presentables);
	}

	/**
	 * Refreshes the index types
	 */
	@SuppressWarnings("unused")
	private void getIndexTypesOfIndex() {
		ClusterStateResponse clusterResponse = this.indexClient.admin().cluster().prepareState().execute().actionGet();
		logger.info("clusterResponse : " + clusterResponse);

		ImmutableMap<String, MappingMetaData> mappings = clusterResponse.state().getMetaData().index(indexName).getMappings();

		logger.info("index types found in index : " + mappings.keySet());

		for (String indexType : mappings.keySet()) {
			logger.info("adding index type : " + indexType);
			try {
				this.bindIndexType(indexType);
				logger.info("adding index type : " + indexType + " succeded");
			} catch (Exception e) {
				logger.info("adding index type : " + indexType + " failed");
			}
		}
	}

	@SuppressWarnings("unchecked")
	public boolean updateManagerProperties(HashMap<String, FullTextIndexType> colForIdTypes) {

		// Retrieving current collectionIds and fields
		Map<String, Object> result = null;
		String id = null;
		long version = 0;
		try {
			SearchResponse response = this.indexClient.prepareSearch(META_INDEX).setQuery(QueryBuilders.matchAllQuery()).execute().actionGet();
			for (SearchHit hit : response.getHits().getHits()) {
				result = hit.getSource();
				id = hit.getId();
				version = hit.getVersion();
			}

		} catch (Exception e) {
			logger.warn("No meta-index yet", e);
		}
		ArrayList<String> collectionIdsToBeAdded;
		ArrayList<String> fieldsToBeAdded;
		if (result != null) {
			collectionIdsToBeAdded = (ArrayList<String>) result.get("collectionIDs");
			fieldsToBeAdded = (ArrayList<String>) result.get("fields");
		} else {
			collectionIdsToBeAdded = new ArrayList<String>();
			fieldsToBeAdded = new ArrayList<String>();
		}

		Set<String> keys = colForIdTypes.keySet();
		for (String colIDandLang : keys) {
			FullTextIndexType index = colForIdTypes.get(colIDandLang);
			IndexField[] fields = index.getFields();
			// add to the Manager properties, the fields retrieved per
			// collection and language
			for (IndexField field : fields) {
				if (!RowsetParser.addToFieldInfo(fieldsToBeAdded, field.name, colIDandLang, index)) {
					return false;
				}
			}

			// add info for the special field allIndexes supported by FT, for
			// this lang and colID
			fieldsToBeAdded.add(colIDandLang + IndexType.SEPERATOR_FIELD_INFO + IndexType.SEARCHABLE_TAG + IndexType.SEPERATOR_FIELD_INFO + ALL_INDEXES);

			collectionIdsToBeAdded.add(colIDandLang);

		}

		// remove duplicates
		HashSet<String> hs1 = new HashSet<String>();
		HashSet<String> hs2 = new HashSet<String>();
		hs1.addAll(collectionIdsToBeAdded);
		hs2.addAll(fieldsToBeAdded);
		collectionIdsToBeAdded.clear();
		fieldsToBeAdded.clear();
		collectionIdsToBeAdded.addAll(hs1);
		fieldsToBeAdded.addAll(hs2);

		try {
			Map<String, Object> document = new HashMap<String, Object>();
			document.put("collectionIDs", collectionIdsToBeAdded.toArray(new String[collectionIdsToBeAdded.size()]));
			document.put("fields", fieldsToBeAdded.toArray(new String[fieldsToBeAdded.size()]));
			IndexRequestBuilder indexRequest = indexClient.prepareIndex(META_INDEX, META_INDEX, id)
			.setSource(ElasticSearchHelper.createJSONObject(document).string());
			if(result!=null)
				indexRequest.setVersion(++version);
			indexRequest.execute();
			logger.info("Inserted colIDs and fields to " + META_INDEX);
			commitMeta();
		} catch (ElasticSearchException e) {
			logger.error("Caught an exception: ", e);
		} catch (IOException e) {
			logger.error("Caught an exception: ", e);
		}

		return true;
	}
	
	
	public boolean addMetaIndex()
	{
		Map<String, Object> result = null;
		// check if there is a meta-index
		try {
			SearchResponse response = this.indexClient.prepareSearch(META_INDEX).setQuery(QueryBuilders.matchAllQuery()).execute().actionGet();
			for (SearchHit hit : response.getHits().getHits()) {
				result = hit.getSource();
			}

		} catch (Exception e) {
			logger.warn("No meta-index yet", e);
		}
		// if not add it
		if(result==null)
		{
			try {
				Map<String, Object> document = new HashMap<String, Object>();
				document.put("collectionIDs", new String[0]);
				document.put("fields", new String[0] );
				IndexRequestBuilder indexRequest = indexClient.prepareIndex(META_INDEX, META_INDEX).setSource(ElasticSearchHelper.createJSONObject(document).string());
				indexRequest.execute();
			} catch (IOException e) {
				logger.error("Failed to add meta index",e);
				return false;
			}	
		}
		return true;
	}

}
