package org.gcube.elasticsearch;

import static org.elasticsearch.node.NodeBuilder.nodeBuilder;
import gr.uoa.di.madgik.commons.server.ConnectionManagerConfig;
import gr.uoa.di.madgik.commons.server.PortRange;
import gr.uoa.di.madgik.commons.server.TCPConnectionManager;
import gr.uoa.di.madgik.grs.buffer.GRS2BufferException;
import gr.uoa.di.madgik.grs.proxy.tcp.TCPConnectionHandler;
import gr.uoa.di.madgik.grs.proxy.tcp.TCPStoreConnectionHandler;
import gr.uoa.di.madgik.grs.reader.ForwardReader;
import gr.uoa.di.madgik.grs.reader.GRS2ReaderException;
import gr.uoa.di.madgik.grs.record.GRS2RecordDefinitionException;
import gr.uoa.di.madgik.grs.record.Record;
import gr.uoa.di.madgik.grs.writer.GRS2WriterException;
import gr.uoa.di.madgik.rr.ResourceRegistry;
import gr.uoa.di.madgik.rr.ResourceRegistryException;

import java.io.IOException;
import java.io.Serializable;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import org.carrot2.elasticsearch.ClusteringAction;
import org.carrot2.elasticsearch.ClusteringAction.ClusteringActionRequestBuilder;
import org.carrot2.elasticsearch.ClusteringAction.ClusteringActionResponse;
import org.carrot2.elasticsearch.DocumentGroup;
import org.carrot2.elasticsearch.ListAlgorithmsAction;
import org.carrot2.elasticsearch.ListAlgorithmsAction.ListAlgorithmsActionResponse;
import org.carrot2.elasticsearch.LogicalField;
import org.elasticsearch.action.WriteConsistencyLevel;
import org.elasticsearch.action.admin.cluster.node.info.NodeInfo;
import org.elasticsearch.action.admin.cluster.node.info.NodesInfoRequest;
import org.elasticsearch.action.admin.cluster.state.ClusterStateResponse;
import org.elasticsearch.action.admin.indices.alias.IndicesAliasesResponse;
import org.elasticsearch.action.admin.indices.create.CreateIndexRequestBuilder;
import org.elasticsearch.action.admin.indices.create.CreateIndexResponse;
import org.elasticsearch.action.admin.indices.mapping.delete.DeleteMappingRequest;
import org.elasticsearch.action.admin.indices.mapping.put.PutMappingResponse;
import org.elasticsearch.action.bulk.BulkItemResponse;
import org.elasticsearch.action.bulk.BulkRequestBuilder;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.get.MultiGetItemResponse;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.search.SearchRequestBuilder;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.action.search.SearchType;
import org.elasticsearch.client.Client;
import org.elasticsearch.client.IndicesAdminClient;
import org.elasticsearch.cluster.metadata.AliasMetaData;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.cluster.metadata.MappingMetaData;
import org.elasticsearch.common.collect.ImmutableOpenMap;
import org.elasticsearch.common.settings.ImmutableSettings;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.transport.InetSocketTransportAddress;
import org.elasticsearch.common.transport.TransportAddress;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.XContentFactory;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.indices.IndexMissingException;
import org.elasticsearch.node.Node;
import org.elasticsearch.search.SearchHit;
import org.gcube.elasticsearch.entities.ClusterResponse;
import org.gcube.elasticsearch.helpers.ElasticSearchHelper;
import org.gcube.elasticsearch.helpers.QueryParser;
import org.gcube.elasticsearch.helpers.RowsetParser;
import org.gcube.indexmanagement.common.FullTextIndexType;
import org.gcube.indexmanagement.common.IndexException;
import org.gcube.indexmanagement.common.IndexField;
import org.gcube.indexmanagement.common.IndexType;
import org.gcube.indexmanagement.resourceregistry.RRadaptor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;

public class FullTextNode implements Serializable {
	private static final long serialVersionUID = 1L;

	private static final Logger logger = LoggerFactory.getLogger(FullTextNode.class);

	private static final int DEFAULT_MAX_TERMS = 50;
	
	private static Integer BULKREQUEST_SIZE = 5000;
	private static Integer DEFAULT_NUM_OF_REPLICAS = 0;
	private static Integer DEFAULT_NUM_OF_SHARDS = 1;
	private static Integer MAX_FRAGMENT_CNT = 5;
	private static Integer MAX_FRAGMENT_SIZE = 150;
	private static String DEFAULT_DATADIR = ".";
	
	private static final long RSTIMEOUT = 5;
	private static String ALL_INDEXES = "allIndexes";
	public static String META_INDEX = "meta-index";
	public static String TEMP_PREFIX = "temp";
	public static String INDEX_PREFIX = "idx";
	public static String ACTIVE_INDEX = "active_index";
	private static int INDEX_REFRESH_INTERVAL = 60;
	public static String DEFAULT_INDEXNAME = "default_index";
	public static String DEFAULT_INDEXSTORE = "file system";
	public static String DEFAULT_ANALYZER = "simple";
	public static String KEYWORD_ANALYZER = "keyword"; //it is used when no analysis is required, should we use not_analysed?
	

	private Client indexClient;
	private Node indexNode;
	private Set<String> indexTypes = new HashSet<String>();
	private FTNodeCache cache;

	private String clusterName;
	private String defaultIndexName;
	private Integer noOfReplicas;
	private Integer noOfShards;
	private Integer maxResults;
	private String scope;
	private Integer maxFragmentCnt;
	private Integer maxFragmentSize;
	private String dataDir = DEFAULT_DATADIR;
	
	private final ExecutorService executorService =  Executors.newCachedThreadPool();// Executors.newCachedThreadPool();
	
	transient private RRadaptor rradaptor;
	
	private String hostname;

	private HashMap<String, FullTextIndexType> colForField = new HashMap<String, FullTextIndexType>();
	// Constructors

	public FullTextNode(String hostname) throws ResourceRegistryException, InterruptedException {
		this(hostname, true);
		logger.info("Initializing FullTextNode");
		this.cache = new FTNodeCache();
		this.initialize(true);
	}
	
	public FullTextNode(String hostname, Boolean useRRAdaptor) throws ResourceRegistryException, InterruptedException {
		logger.info("Initializing FullTextNode");
		this.cache = new FTNodeCache();
		this.hostname = hostname;
		this.initialize(useRRAdaptor);
	}

	public FullTextNode(String hostname, String clusterName, String indexName, Integer noOfReplicas, Integer noOfShards, String scope, Integer maxFragmentCnt, Integer maxFragmentSize, Boolean useRRAdaptor) throws ResourceRegistryException, InterruptedException {
		this(hostname, useRRAdaptor);
		this.clusterName = clusterName;
		this.defaultIndexName = indexName;
		this.noOfReplicas = noOfReplicas;
		this.noOfShards = noOfShards;
		this.scope = scope;
		this.maxFragmentCnt = maxFragmentCnt;
		this.maxFragmentSize = maxFragmentSize;
	}
	
	public FullTextNode(String hostname, String clusterName, String indexName, Integer noOfReplicas, Integer noOfShards, String scope, Integer maxFragmentCnt, Integer maxFragmentSize) throws ResourceRegistryException, InterruptedException {
		this(hostname);
		this.clusterName = clusterName;
		this.defaultIndexName = indexName;
		this.noOfReplicas = noOfReplicas;
		this.noOfShards = noOfShards;
		this.scope = scope;
		this.maxFragmentCnt = maxFragmentCnt;
		this.maxFragmentSize = maxFragmentSize;
	}

	public FullTextNode(String hostname, String clusterName, String indexName, String scope, Boolean useRRAdaptor) throws ResourceRegistryException, InterruptedException {
		this(hostname, useRRAdaptor);
		this.clusterName = clusterName;
		this.defaultIndexName = indexName;
		this.noOfReplicas = DEFAULT_NUM_OF_REPLICAS;
		this.noOfShards = DEFAULT_NUM_OF_SHARDS;
		this.scope = scope;
		this.maxFragmentCnt = MAX_FRAGMENT_CNT;
		this.maxFragmentSize = MAX_FRAGMENT_SIZE;
	}
	
	public FullTextNode(String hostname, String clusterName, String indexName, String scope) throws ResourceRegistryException, InterruptedException {
		this(hostname);
		this.clusterName = clusterName;
		this.defaultIndexName = indexName;
		this.noOfReplicas = DEFAULT_NUM_OF_REPLICAS;
		this.noOfShards = DEFAULT_NUM_OF_SHARDS;
		this.scope = scope;
		this.maxFragmentCnt = MAX_FRAGMENT_CNT;
		this.maxFragmentSize = MAX_FRAGMENT_SIZE;
	}
	
	public FullTextNode(String hostname, String dataDir, String clusterName, String indexName, Integer noOfReplicas, Integer noOfShards, String scope, Integer maxFragmentCnt, Integer maxFragmentSize, Boolean useRRAdaptor) throws ResourceRegistryException, InterruptedException{
		this(hostname, clusterName, indexName, noOfReplicas, noOfShards, scope, maxFragmentCnt, maxFragmentSize, useRRAdaptor);
		this.dataDir = dataDir;
	}
	
	
	public FullTextNode(String hostname, String dataDir, String clusterName, String indexName, Integer noOfReplicas, Integer noOfShards, String scope, Integer maxFragmentCnt, Integer maxFragmentSize) throws ResourceRegistryException, InterruptedException {
		this(hostname, clusterName, indexName, noOfReplicas, noOfShards, scope, maxFragmentCnt, maxFragmentSize);
		this.dataDir = dataDir;
	}

	public FullTextNode(String hostname, String dataDir, String clusterName, String indexName, String scope, Boolean useRRAdaptor) throws ResourceRegistryException, InterruptedException {
		this(hostname, clusterName, indexName, scope, useRRAdaptor);
		this.dataDir = dataDir;
	}

	public FullTextNode(String hostname, String dataDir, String clusterName, String indexName, String scope) throws ResourceRegistryException, InterruptedException {
		this(hostname, clusterName, indexName, scope);
		this.dataDir = dataDir;
	}

	// End of Constructors

	public String getClusterName() {
		return this.clusterName;
	}

	public String getIndexName() {
		return this.defaultIndexName;
	}

	public Integer getNoOfReplicas() {
		return this.noOfReplicas;
	}

	public Integer getNoOfShards() {
		return this.noOfShards;
	}

	public String getScope() {
		return this.scope;
	}
	
	public String getHostname() {
		return this.hostname;
	}

	public Integer getMaxFragmentCnt() {
		return this.maxFragmentCnt;
	}

	public Integer getMaxFragmentSize() {
		return this.maxFragmentSize;
	}

	public Client getIndexClient() {
		return this.indexClient;
	}
	
	public void setMaxResults(Integer maxResults){
		this.maxResults = maxResults;
	}
	

	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("FullTextNode [indexNode=").append(this.indexNode).append(", cache=").append(this.cache).append(", clusterName=").append(this.clusterName)
				.append(", indexName=").append(this.defaultIndexName).append(", noOfReplicas=").append(this.noOfReplicas).append(", noOfShards=").append(this.noOfShards)
				.append(", scope=").append(this.scope).append(", maxFragmentCnt=").append(this.maxFragmentCnt).append(", maxFragmentSize=").append(this.maxFragmentSize)
				.append("]");
		return builder.toString();
	}
	
	
	/////////////////

	public synchronized void addIndexType(String indexTypeStr, String indexName) throws Exception {
		FullTextIndexType indexType = QueryParser.retrieveIndexType(indexTypeStr, this.scope, this.cache);
		this.addIndexType(indexTypeStr, indexType, indexName);
	}

	public synchronized void addIndexType(String indexTypeStr, FullTextIndexType indexType, String indexName) throws Exception {
		logger.info("Calling addIndexType");
		if (indexType == null) {
			logger.warn("IndexType is null");
			throw new Exception("Trying to null as IndexType. Check how you got it");
		}

		// in case indexType is not in cache
		QueryParser.addFullTextIndexTypeIntoCache(indexTypeStr, this.scope, indexType, this.cache);
		
		//TODO: change to support index -> index_type
		if (this.indexTypes.contains(indexName + "-" + indexTypeStr)){
			logger.info("IndexType has already been added.");
			return;
		}

		// create mapping if not exists
		IndicesAdminClient iac = this.indexClient.admin().indices();

		logger.info("Checking if index exists");
		if (iac.prepareExists(indexName).get().isExists()) {
			logger.info("Index already exists");
		} else {
			CreateIndexRequestBuilder createIndexRequest = iac.prepareCreate(indexName)
					.setSettings(this.getIndexCreateSetting());
			logger.info("Create Index Request : " + createIndexRequest.request());
			
			CreateIndexResponse cir = createIndexRequest.get();
			logger.info("Create Index Response : " + cir);
		}


		logger.trace("Index Type");
		logger.trace("-----------------------------------------------");
		logger.trace(indexType.toString());
		logger.trace("-----------------------------------------------");

		Map<String, Object> mapping = new HashMap<String, Object>();
		List<String> presentables = new ArrayList<String>();
		List<String> searchables = new ArrayList<String>();
		List<String> highlightables = new ArrayList<String>();

		for (IndexField idxTypeField : indexType.getFields()) {
			@SuppressWarnings("unused")
			String store, index, type, highlightable;

			store = idxTypeField.store ? "yes" : "no";

			if (idxTypeField.index)
				index = idxTypeField.tokenize ? "analyzed" : "not_analyzed";
			else
				index = "no";

			if (idxTypeField.name.equalsIgnoreCase(IndexType.COLLECTION_FIELD) || idxTypeField.name.equalsIgnoreCase(IndexType.DOCID_FIELD))
				index = "analyzed";

			if (idxTypeField.type != null && idxTypeField.type.trim().length() > 0)
				type = idxTypeField.type;
			else
				type = "string";
			
			//TODO: consider using multifield for simple and raw fields
			//probably this means that the queries should be changed
			Map<String, Object> fieldMap = new HashMap<String, Object>();
			fieldMap.put("type", type);
			//fieldMap.put("store", store);
			fieldMap.put("index", index);
			
			//XXX: pick the right highlighter
			//fieldMap.put("index_options" , "offsets");
			//fieldMap.put("term_vector", "with_positions_offsets");
			
			fieldMap.put("boost", Float.valueOf(idxTypeField.boost));
			fieldMap.put("copy_to", idxTypeField.name + "_raw");
			if (idxTypeField.name.equalsIgnoreCase(IndexType.COLLECTION_FIELD) || idxTypeField.name.equalsIgnoreCase(IndexType.DOCID_FIELD))
				fieldMap.put("analyzer", KEYWORD_ANALYZER);
			else
				fieldMap.put("analyzer", DEFAULT_ANALYZER);
			

			mapping.put(idxTypeField.name, fieldMap);
			
			//can be used for facets etc
			Map<String, Object> rawFieldMap = new HashMap<String, Object>();
			rawFieldMap.put("type", type);
			//rawFieldMap.put("store", "yes");
			rawFieldMap.put("index", "not_analyzed");
			rawFieldMap.put("boost", Float.valueOf(idxTypeField.boost));
			mapping.put(idxTypeField.name + "_raw", rawFieldMap);
			

			if (idxTypeField.highlightable && idxTypeField.returned){
				highlightables.add(idxTypeField.name);
			}
			
			if (idxTypeField.returned /*&& !idxTypeField.name.equalsIgnoreCase(IndexType.COLLECTION_FIELD)*/){
				//not explicitly removed from highlighatbles
				presentables.add(idxTypeField.name);
			}
			if (idxTypeField.index)
				searchables.add(idxTypeField.name);
		}
		
		//Add objectid field by default
		Map<String, Object> fieldMap = new HashMap<String, Object>();
		fieldMap.put("type", "string");
		//fieldMap.put("store", "yes");
		fieldMap.put("index", "analyzed");
		fieldMap.put("analyzer", KEYWORD_ANALYZER);
		mapping.put(IndexType.DOCID_FIELD, fieldMap);
		
		
		fieldMap = new HashMap<String, Object>();
		fieldMap.put("type", "string");
		//fieldMap.put("store", "yes");
		fieldMap.put("index", "analyzed");
		fieldMap.put("analyzer", KEYWORD_ANALYZER);
		mapping.put(ElasticSearchHelper.SECURITY_FIELD, fieldMap);
		

		this.cache.presentableFieldsPerIndexType.put(indexTypeStr, presentables);
		this.cache.searchableFieldsPerIndexType.put(indexTypeStr, searchables);
		this.cache.highlightableFieldsPerIndexType.put(indexTypeStr, highlightables);

		
		logger.info("1. in addIndexType cache presentables   : " + this.cache.presentableFieldsPerIndexType);
		logger.info("1. in addIndexType cache searchables    : " + this.cache.searchableFieldsPerIndexType);
		logger.info("1. in addIndexType cache highlightables : " + this.cache.highlightableFieldsPerIndexType);
		
		Map<String, Object> propertyMap = new HashMap<String, Object>();
		propertyMap.put("properties", mapping);

		// /logger.info("propertyMap : " + propertyMap);
		Map<String, Object> mappingMap = new HashMap<String, Object>();
		mappingMap.put(indexTypeStr, propertyMap);

		String json = ElasticSearchHelper.createJSONObject(mappingMap).string();
		logger.info("json : " + json);

		PutMappingResponse pmr = iac.preparePutMapping()
				.setIndices(indexName)
				.setType(indexTypeStr)
				.setSource(json)
				.get();
		logger.info("Update Settings Response : " + pmr.toString());

		this.indexTypes.add(indexTypeStr + "-" + indexName);
		
	}

	public void createOrJoinCluster() {
		logger.info("creating or joining cluster");
		logger.info("cluster.name : " + this.clusterName);
		logger.info("index.number_of_replicas : " + this.noOfReplicas);
		logger.info("index.number_of_shards : " + this.noOfShards);
		logger.info("path.data : " + this.dataDir);

		Settings settings = getIndexCreateSetting();

		this.indexNode = nodeBuilder()
				.client(false)
				.clusterName(this.clusterName)
				.settings(settings).node();
		this.indexClient = this.indexNode.client();
		
	}

	public String getESTransportAddress() {
		logger.info("getting transport address for node : " + this.hostname);
		String transportAddress = null;
		
		Iterator<NodeInfo> it = this.indexClient.admin()
				.cluster()
				.nodesInfo(new NodesInfoRequest())
				.actionGet()
				.iterator();
		while (it.hasNext()){
			NodeInfo ni = it.next();
			
			TransportAddress address = ni.getTransport().getAddress().publishAddress();
			InetSocketTransportAddress sockAddress = null;
			if (address instanceof InetSocketTransportAddress)
				sockAddress = (InetSocketTransportAddress) address;
			else
				continue;
			
			String hostname = sockAddress.address().getHostName();
			String hoststring = sockAddress.address().getHostString();
			Integer port = Integer.valueOf(sockAddress.address().getPort());
			
			logger.info("checking hostname : " + hostname + " hosttring : " + hoststring + " port : " + port);
			
			if (hostname.equalsIgnoreCase(this.hostname)){
				transportAddress = hostname + ":" + port;
				break;
			}
		}
		
		logger.info("found transport address : " + transportAddress);
		
		return transportAddress;
	}
	
	/***
	 * 
	 * @param knownNodes
	 *            list of hostname:port
	 * @throws IOException
	 */
	
	public void joinCluster(List<String> hosts) throws IOException {
		String hostStr = Joiner.on(",").join(hosts);
		logger.info("joining cluster of known node : " + hosts);
		logger.info("cluster.name                  : " + this.clusterName);
		logger.info("index.number_of_replicas      :    " + this.noOfReplicas);
		logger.info("index.number_of_shards        : " + this.noOfShards);
		logger.info("path.data                     : " + this.dataDir);
		logger.info("hostStr                       : " + hostStr);

		Settings settings = ImmutableSettings.settingsBuilder()
//				.put("path.data", this.dataDir)
				.put(getIndexCreateSetting())
				.put("discovery.zen.ping.multicast.ping.enabled", false)
				.put("discovery.zen.ping.multicast.enabled", false)
				.put("discovery.zen.ping.unicast.enabled", true)
				.put("discovery.zen.ping.unicast.hosts", hostStr)
				.build();
		
		this.indexNode = nodeBuilder()
				.client(false)
				.clusterName(this.clusterName)
				.settings(settings)
				.node();
		this.indexClient = this.indexNode.client();
	}

	/**
	 * 
	 * @param queryString
	 * @return gRS2 locator of the query results
	 * @throws IndexException
	 * @throws GRS2WriterException
	 */
	public String query(String queryString, Set<String> securityIdentifiers) throws IndexException, GRS2WriterException {
		return this.query(queryString, 0, -1, securityIdentifiers);
	}
	
	public String query(String queryString, Set<String> securityIdentifiers, Boolean useRR) throws IndexException, GRS2WriterException {
		return this.query(queryString, 0, -1, securityIdentifiers, useRR);
	}

	/**
	 * Executes the query of gRS2 locator of maximum maxHits records (if >0)
	 * 
	 * @param queryString
	 * @param maxHits
	 * @return gRS2 locator of the results
	 * @throws IndexException
	 * @throws GRS2WriterException
	 */
	public String query(final String queryString, Integer from, Integer maxHits, Set<String> securityIdentifiers) throws GRS2WriterException, IndexException {
		return this.query(queryString, from, maxHits, securityIdentifiers, Boolean.TRUE);
	}
	
	
	
	
	public String query(final String queryString, Integer from, Integer maxHits, Set<String> securityIdentifiers, Boolean useRR) throws GRS2WriterException, IndexException {
		logger.info("calling query with : " + queryString + ". RR : " + (this.rradaptor != null) + " useRR : " + useRR);
		if (useRR)
			return FullTextNodeHelpers.query(this.indexClient, queryString, from, maxHits, securityIdentifiers, this.cache, this.maxResults, this.maxFragmentCnt, this.maxFragmentSize, this.rradaptor, ACTIVE_INDEX, executorService);
		else
			return FullTextNodeHelpers.query(this.indexClient, queryString, from, maxHits, securityIdentifiers, this.cache, this.maxResults, this.maxFragmentCnt, this.maxFragmentSize, null, ACTIVE_INDEX, executorService);
	}

	
	/////
	public Map<String, Integer> frequentTerms(String queryString, Set<String> securityIdentifiers) throws IndexException, GRS2WriterException {
		return this.frequentTerms(queryString, DEFAULT_MAX_TERMS, securityIdentifiers);
	}
	
	public Map<String, Integer> frequentTerms(String queryString, Set<String> securityIdentifiers, Boolean useRR) throws IndexException, GRS2WriterException {
		return this.frequentTerms(queryString, DEFAULT_MAX_TERMS, securityIdentifiers, useRR);
	}

	public Map<String, Integer> frequentTerms(final String queryString, Integer maxTerms, Set<String> securityIdentifiers) throws GRS2WriterException, IndexException {
		return this.frequentTerms(queryString, maxTerms, securityIdentifiers, Boolean.TRUE);
	}
	
	public Map<String, Integer> frequentTerms(final String queryString, Integer maxTerms, Set<String> securityIdentifiers, Boolean useRR) throws GRS2WriterException, IndexException {
		logger.info("calling query with : " + queryString + ". RR : " + (this.rradaptor != null) + " useRR : " + useRR);
		if (useRR)
			return FullTextNodeHelpers.frequentTerms(this.indexClient, queryString, maxTerms, securityIdentifiers, this.cache, this.rradaptor, ACTIVE_INDEX);
		else
			return FullTextNodeHelpers.frequentTerms(this.indexClient, queryString, maxTerms, securityIdentifiers, this.cache, null, ACTIVE_INDEX);
	}
	/////
	
	
	public String queryStream(final String queryString, Set<String> securityIdentifiers) throws IndexException, GRS2WriterException {
		return queryStream(queryString, -1, securityIdentifiers);
	}

	/**
	 * Important: The returned results are not sorted!
	 * 
	 * @param queryString
	 * @return gRS2 locator of the results
	 * @throws IndexException
	 * @throws GRS2WriterException
	 */
	public String queryStream(final String queryString, final int maxHits, Set<String> securityIdentifiers) throws IndexException, GRS2WriterException {
		return FullTextNodeHelpers.queryStream(this.indexClient, queryString, maxHits, this.cache, securityIdentifiers, this.maxResults, this.maxFragmentCnt, this.maxFragmentSize, this.rradaptor, ACTIVE_INDEX, executorService);
		
	}

	public boolean activateIndex(String indexName) {
		String idxName = null;
		if (indexName != null)
			idxName = indexName;
		else
			idxName = this.defaultIndexName;
		
		IndicesAliasesResponse iar = this.indexClient.admin().indices().prepareAliases()
				.addAlias(idxName, ACTIVE_INDEX)
				.get();
		
		logger.info("Alias for  : " + idxName + " does not exist. Creating now");
		if (!iar.isAcknowledged()){
			logger.warn("Alias for  : " + idxName + " creation failed");
			return false;
		} else {
			logger.info("Alias for  : " + idxName + " created");
			return true;
		}
	}
	
	public boolean deactivateIndex(String indexName) {
		IndicesAliasesResponse iar = this.indexClient.admin().indices().prepareAliases()
				.removeAlias(indexName, ACTIVE_INDEX)
				.get();
		
		if (!iar.isAcknowledged()){
			logger.warn("Couldn't remove : " + indexName + " from alias");
			return false;
		} else {
			logger.info(indexName + " removed from alias");
			return true;
		}
	}
	
	/**
	 * Deletes the index and the metaIndex
	 * */
	public boolean deleteIndex() {
		boolean indexRet = ElasticSearchHelper.delete(this.indexClient, ACTIVE_INDEX);
		return indexRet;
	}
	
	public boolean deleteMeta() {
		boolean metaRet = ElasticSearchHelper.delete(this.indexClient, META_INDEX);
		return metaRet;
	}
	
	public boolean deleteAll() {
		boolean indexRet = ElasticSearchHelper.deleteAllIndices(this.indexClient);
		return indexRet;
	}
	
	public Set<String> getCollectionsOfIndex(String indexName) {
		Set<String> collectionsOfIndex = ElasticSearchHelper.getAllCollectionsOfIndex(this.indexClient, indexName);
		
		logger.info("collectionsOfIndex : " + indexName + " count : " + collectionsOfIndex);
		
		return collectionsOfIndex;
	}
	
	
	public Long getCollectionDocuments(String collectionID) {
		Long count = ElasticSearchHelper.collectionDocumentsCountElasticSearch(this.indexClient, ACTIVE_INDEX, collectionID);
		logger.info("collection : " + collectionID + " count : " + count);
		return count;
	}
	
	
	public Set<String> getIndicesOfCollection(String collection) {
		Set<String> indicesOfCollection = ElasticSearchHelper.indicesOfCollection(this.indexClient, ACTIVE_INDEX , collection);
		
		logger.info("indicesOfCollection : " + collection + " count : " + indicesOfCollection);
		
		return indicesOfCollection;
	}
	
	/**
	 * Deletes the index and the metaIndex
	 * */
	public boolean deleteIndex(String indexName) {
		this.deactivateIndex(indexName);
		Set<String> collectionsOfIndex = ElasticSearchHelper.getAllCollectionsOfIndex(this.indexClient, indexName);
		
		//ElasticSearchHelper.deleteCollections(this.indexClient, indexName, collectionsOfIndex);
		
		boolean indexRet = ElasticSearchHelper.delete(this.indexClient, indexName);
		//this.commit(indexName);
		
		boolean doDelete = false;
		IndicesAdminClient iac = this.indexClient.admin().indices();
		logger.info("Checking if alias exists");
		if (!iac.prepareExists(ACTIVE_INDEX).get().isExists()) {
			logger.info("alias does not exist");
			doDelete = false;
		} else {
			logger.info("alias exists");
			doDelete = true;
		}
		
		if (indexRet) {
			if (doDelete){
				for (String collID : collectionsOfIndex){
					if (!ElasticSearchHelper.collectionExists(this.indexClient, ACTIVE_INDEX, collID)){
						logger.info("collection : " + collID + " does not exist in the rest indices");
						Boolean updateResult = updateManagerPropertiesRemoveCollID(collID);
						
						if (!updateResult){
							logger.warn("meta index is not updated");
							return false;
						}
						
					} else {
						logger.info("collection : " + collID + " exists in the rest indices");
					}
				}
			} else { //alias does not exist. delete the collection from meta
				logger.info("alias does not exist so we will delete the collections from meta");
				
				for (String collID : collectionsOfIndex){
					Boolean updateResult = updateManagerPropertiesRemoveCollID(collID);
					
					if (!updateResult){
						logger.warn("meta index is not updated");
						return false;
					}
				}
			}
			
			this.commitMeta();
		}

		return indexRet;
	}
	
	
	/**
	 * Deletes the documents with IDs within the list <a>docIDs</a>
	 * 
	 * @param docIDs
	 */
	public void deleteDocuments(List<String> docIDs) {
		ElasticSearchHelper.deleteDocuments(this.indexClient, ACTIVE_INDEX, docIDs);
	}
	
	/**
	 * Deletes the documents of the collection with ID <a>collID</a>
	 * 
	 * @param collID
	 */
	public Boolean deleteCollection(String collID) {
		Boolean result = ElasticSearchHelper.deleteCollection(this.indexClient, ACTIVE_INDEX, collID);
		if (result) {
			Boolean updateResult = updateManagerPropertiesRemoveCollID(collID);
			
			if (!updateResult){
				logger.warn("meta index is not updated");
				return false;
			}
			
			this.commitMeta();
		}
		return result;
	}

	/**
	 * Commits the changes to the index by sending a flush and a refresh request
	 */
	public void commitMeta() {
		ElasticSearchHelper.commit(this.indexClient, META_INDEX);
	}

	public void commit() {
		ElasticSearchHelper.commit(this.indexClient, ACTIVE_INDEX);
	}
	
	public void commit(String indexName) {
		ElasticSearchHelper.commit(this.indexClient, indexName);
	}

	public void clearIndex(String indexTypeName) {
		ElasticSearchHelper.clearIndex(this.indexClient, ACTIVE_INDEX, indexTypeName);
	}

	/**
	 * Closes the index
	 */
	public void close() {
		this.indexClient.close();
		this.indexNode.close();
	}
	
	private void initialize(Boolean useRRAdaptor) throws ResourceRegistryException, InterruptedException {
		this.initialize();
		if (useRRAdaptor){
			logger.info("Initializing ResourceRegistry");
			try {
				ResourceRegistry.startBridging();
				TimeUnit.SECONDS.sleep(1);
				while(!ResourceRegistry.isInitialBridgingComplete()) TimeUnit.SECONDS.sleep(10);
			} catch (ResourceRegistryException e) {
				logger.error("Resource Registry could not be initialized", e);
				throw e;
			} catch (InterruptedException e) {
				logger.error("Resource Registry could not be initialized", e);
				throw e;
			}
			this.rradaptor = new RRadaptor();
			logger.info("Initializing ResourceRegistry is DONE");
		} else {
			logger.info("ResourceRegistry will NOT be initialized as configured");
		}
	}

	private void initialize() {
		TCPConnectionManager.Init(new ConnectionManagerConfig(this.hostname, Arrays.asList(new PortRange(4000, 4100)), true));
		TCPConnectionManager.RegisterEntry(new TCPConnectionHandler());
		TCPConnectionManager.RegisterEntry(new TCPStoreConnectionHandler());
	}

	/**
	 * Feeds the index with a rowset.
	 * 
	 * @param rowset
	 * @return <a>True</a> on success or <a>False</a> on failure
	 */
	public boolean feedRowset(String rowset, String indexName, Set<String> securityIdentifiers) {
		BulkRequestBuilder bulkRequest = this.indexClient.prepareBulk();
		HashMap<String, FullTextIndexType> colForIdTypes = new HashMap<String, FullTextIndexType>();
		boolean status = feedRowset(bulkRequest, rowset, colForIdTypes, indexName, securityIdentifiers);

		if (status) {
			long before = System.currentTimeMillis();
			BulkResponse bulkResponse = bulkRequest
					.setConsistencyLevel(WriteConsistencyLevel.ONE)
					.get();
			long after = System.currentTimeMillis();
			logger.info("Time for commiting the bulk requests : " + (after - before) / 1000.0 + " secs");
			logger.info("bulkResponse : " + bulkResponse.getTookInMillis() / 1000.0 + " secs");
			logger.info("bulkResponse : " + bulkResponse);
			if (bulkResponse.hasFailures()) {
				logger.info("failures have happened");
			}
			this.commit(indexName);

		} else {
			logger.info("feedRowset failed");
		}

		return status;
	}

	
	public boolean feedLocator(String resultSetLocation, String indexName, Set<String> securityIdentifiers) throws GRS2ReaderException, URISyntaxException, GRS2RecordDefinitionException, GRS2BufferException {
		return feedLocator(resultSetLocation, this.colForField, indexName, securityIdentifiers);
	}
	
	public boolean feedLocator(String resultSetLocation, Set<String> securityIdentifiers) throws GRS2ReaderException, URISyntaxException, GRS2RecordDefinitionException, GRS2BufferException {
		return feedLocator(resultSetLocation, this.colForField, this.defaultIndexName, securityIdentifiers);
	}

	public Boolean feedLocator(String resultSetLocation, HashMap<String, FullTextIndexType> colForField, Set<String> securityIdentifiers) throws GRS2ReaderException, URISyntaxException,
	GRS2RecordDefinitionException, GRS2BufferException {
		return feedLocator(resultSetLocation, colForField, this.defaultIndexName, securityIdentifiers);
	}
	/**
	 * Feeds the index with rowsets that are read from the the given locator
	 * 
	 * @param resultSetLocation
	 * @param colForField
	 * @return true if feed was successful, otherwise false
	 * @throws GRS2ReaderException
	 * @throws URISyntaxException
	 * @throws GRS2RecordDefinitionException
	 * @throws GRS2BufferException
	 */
	public Boolean feedLocator(String resultSetLocation, HashMap<String, FullTextIndexType> colForField, String indexName, Set<String> securityIdentifiers) throws GRS2ReaderException, URISyntaxException,
			GRS2RecordDefinitionException, GRS2BufferException {
		// UpdaterService

		long beforeFeed, afterFeed;

		logger.info("Initializing reader at resultset : " + resultSetLocation);
		ForwardReader<Record> reader = new ForwardReader<Record>(new URI(resultSetLocation));
		reader.setIteratorTimeout(RSTIMEOUT);
		reader.setIteratorTimeUnit(TimeUnit.MINUTES);

		int rowSetCount = 0;
		boolean success = true;

		beforeFeed = System.currentTimeMillis();


		// HashMap<String, FullTextIndexType> colForField = new HashMap<String,
		// FullTextIndexType>();

		BulkRequestBuilder bulkRequest = this.indexClient.prepareBulk();
		try {
			logger.info("Initializing resultset reader iterator");
			Iterator<Record> it = reader.iterator();
			long before, after;

			while (it.hasNext()) {
				
				
				logger.trace("Getting result : " + rowSetCount);
				// System.out.println("Getting result : " + rowSetCount);

				before = System.currentTimeMillis();
				Record result = it.next();
				after = System.currentTimeMillis();
				logger.trace("Time for getting record from Result Set : " + (after - before) / 1000.0 + " secs");

				before = System.currentTimeMillis();
				String rowset = RowsetParser.getRowsetFromResult(result);
				after = System.currentTimeMillis();
				logger.trace("Time for getting rowset from record : " + (after - before) / 1000.0 + " secs");
				// logger.info("Result rowset : " + rowset);
				success = feedRowset(bulkRequest, rowset, colForField, indexName, securityIdentifiers);

				if (success == false) {
					logger.warn("feed rowset failed : " + rowset);
					break;
				}

				logger.info("Result " + rowSetCount + " inserted");
				rowSetCount++;
				
				
				if (rowSetCount % BULKREQUEST_SIZE == 0) {
					logger.info("BulkRequest reached  " + BULKREQUEST_SIZE + " records and will be executed");
					if (doBulk(bulkRequest)){
						bulkRequest = this.indexClient.prepareBulk();
						//TODO: should we commit here?? but not update manager properties
					} else {
						bulkRequest = null;
						success = false;
						break;
					}
				}
			}
		} catch (Exception e) {
			logger.info("Exception will feeding", e);
			success = false;
		}

		reader.close();
		
		if (success){
			logger.info("BulkRequest will insert the last  " + rowSetCount % BULKREQUEST_SIZE + " records");
			
			if (rowSetCount > 0){ // did we add any records in total?
				if (rowSetCount % BULKREQUEST_SIZE == 0 ){ // case that the last request is empty
					logger.info("no records to add in the final bulk request");
				} else { // last request has some records to be added
					if (doBulk(bulkRequest)){
						logger.info("last bulk request succedded. will commit now");
						this.commit(indexName);
						updateManagerProperties(colForField);
						this.commitMeta();
						logger.info("Total number of records fed : " + rowSetCount);
					} else {
						success = false;
					}
				}
			} else { // if no records added at all then do nothing
				logger.warn("no records to add");
			}
		}
		
		afterFeed = System.currentTimeMillis();
		logger.info("Total insert time : " + (afterFeed - beforeFeed) / 1000.0);
		//System.out.println("Total insert time : " + (afterFeed - beforeFeed) / 1000.0);
		
		return success;
	}
	
	
	private static Boolean doBulk(BulkRequestBuilder bulkRequest) {
			long beforeBulk = System.currentTimeMillis();
			BulkResponse bulkResponse = bulkRequest.setConsistencyLevel(WriteConsistencyLevel.ONE).get();
			long afterBulk = System.currentTimeMillis();
			logger.info("Time for commiting the bulk requests : " + (afterBulk - beforeBulk) / 1000.0 + " secs");
			logger.info("bulkResponse : " + bulkResponse.getTookInMillis() / 1000.0 + " secs");
			logger.info("bulkResponse : " + bulkResponse);
			if (bulkResponse.hasFailures()) {
				logger.warn("failures have happened. Message : " + bulkResponse.buildFailureMessage());
				
				logger.warn("Details ");
				for (BulkItemResponse brItem : bulkResponse.getItems()) {
					logger.warn("id : " + brItem.getId() + " " + brItem.getIndex() + " " + brItem.getType() + ", Failure Message : " + brItem.getFailureMessage());
				}
				logger.warn("failures have happened. Message : " + bulkResponse.buildFailureMessage());
				logger.warn("Error. Feeding failed");
				return false;
			} else {
				logger.info("bulk request finished without errors");
				return true;
				
			}
	}

	// ////////////////////////////////////
	// / private methods
	// ////////////////////////////////////

	/**
	 * Adds a rowset in the bulk of a bulkRequest
	 * 
	 * @param bulkRequest
	 * @param rowset
	 * @param colForIdTypes
	 * @return <a>True</a> on success or <a>False</a> on failure
	 */
	private boolean feedRowset(BulkRequestBuilder bulkRequest, String rowset, HashMap<String, FullTextIndexType> colForIdTypes, String indexName, Set<String> securityIdentifiers) {
		long before, after;

		before = System.currentTimeMillis();
		String rsIdxTypeID = RowsetParser.getIdxTypeNameRowset(rowset);
		after = System.currentTimeMillis();
		logger.trace("Time for getting rsIdxTypeID from rowset : " + (after - before) / 1000.0 + " secs");
		logger.trace("Result IndexTypeID : " + rsIdxTypeID);

		before = System.currentTimeMillis();
		String lang = RowsetParser.getLangRowset(rowset);
		after = System.currentTimeMillis();
		logger.trace("Time for getting lang from rowset : " + (after - before) / 1000.0 + " secs");

		logger.trace("Result lang : " + lang);
		// if no language is detected set the unknown
		if (lang == null || lang.equals(""))
			//lang = //LanguageIdPlugin.LANG_UNKNOWN;//IndexType.LANG_UNKNOWN;
			lang = IndexType.LANG_UNKNOWN;

		before = System.currentTimeMillis();
		String colID = RowsetParser.getColIDRowset(rowset);
		after = System.currentTimeMillis();
		logger.trace("Time for colID lang from rowset : " + (after - before) / 1000.0 + " secs");

		logger.trace("Result colID : " + colID);
		// if no collection ID is detected we have a problem and the
		// update must be cancelled
		if (colID == null || colID.equals("")) {
			logger.trace("No collection ID given in ROWSET: " + rowset);
			return false;
		}

		indexName = createIndexName(indexName, colID, false);
		
		try {
			addIndexType(rsIdxTypeID, indexName);
		} catch (Exception e) {
			logger.error("Add index type exception", e);
//			System.out.println("Add index type exception");
//			e.printStackTrace();
			return false;
		}
		// check if the same indexTypeID is specified
		if (rsIdxTypeID == null || !this.indexTypes.contains(rsIdxTypeID + "-" + indexName)) {
			logger.error("IndexType : " + rsIdxTypeID + "-" + indexName + " not in indexTypes : " + this.indexTypes);
			return false;
		}

		//before = System.currentTimeMillis();
		//String alteredRowset = RowsetParser.preprocessRowset(rowset, lang, colID, indexName, rsIdxTypeID, this.scope, this.cache);
		//after = System.currentTimeMillis();
		
		logger.trace("Time for preprocessRowset : " + (after - before) / 1000.0 + " secs");
		// logger.info("Result alteredRowset : " + alteredRowset);

//		if (alteredRowset == null) {
//			logger.error("could not preprocess rowset: " + rowset);
//			return false;
//		}
		before = System.currentTimeMillis();

		FullTextIndexType idxType = this.cache.cachedIndexTypes.get(QueryParser.createIndexTypekey(rsIdxTypeID, this.scope));
		logger.trace("index type for name : " + idxType.getIndexTypeName());
		logger.trace("all indexTypes in cache : " + this.cache.cachedIndexTypes.keySet());

		Boolean insertResult = ElasticSearchHelper.insertRowSet(bulkRequest, this.indexClient, indexName, idxType, this.indexTypes, rowset, securityIdentifiers);
		if (!insertResult) {
			logger.error("error in inserting the rowset " + rowset);
			return false;
		}
		after = System.currentTimeMillis();
		logger.trace("Time for insertRowSet : " + (after - before) / 1000.0 + " secs");

		colForIdTypes.put(colID + IndexType.SEPERATOR_FIELD_INFO + lang, idxType);

		return true;
	}

	
	
	private static String createIndexName(String indexName, String collectionID, Boolean useCollectionID) {
		if (useCollectionID)
			return collectionID + "_" + indexName;
		else
			return indexName;
	}
	
	/**
	 * Used to refresh the indexType and the presentable fields per index type.
	 * Usually called after invalidation
	 * 
	 * @param indexTypeStr
	 */
	private synchronized void bindIndexType(String indexTypeStr, String indexName) {
		logger.info("Calling bindIndexType");

		FullTextIndexType indexType = QueryParser.retrieveIndexType(indexTypeStr, this.scope, this.cache);
		logger.info("Index Type");
		logger.info("-----------------------------------------------");
		logger.info(indexType.toString());
		logger.info("-----------------------------------------------");
		List<String> presentables = new ArrayList<String>();
		List<String> searchables = new ArrayList<String>();
		List<String> highlightables = new ArrayList<String>();

		for (IndexField idxTypeField : indexType.getFields()) {
			if (idxTypeField.highlightable && idxTypeField.returned){
				highlightables.add(idxTypeField.name);
			}
			
			if (idxTypeField.returned /*&& !idxTypeField.name.equalsIgnoreCase(IndexType.COLLECTION_FIELD)*/)
				presentables.add(idxTypeField.name);
			if (idxTypeField.index)
				searchables.add(idxTypeField.name);
		}

		this.indexTypes.add(indexTypeStr + "-" + indexName);
		this.cache.presentableFieldsPerIndexType.put(indexTypeStr, presentables);
		this.cache.searchableFieldsPerIndexType.put(indexTypeStr, searchables);
		this.cache.highlightableFieldsPerIndexType.put(indexTypeStr, highlightables);
		
		logger.info("1. in bindIndexType cache presentables   : " + this.cache.presentableFieldsPerIndexType);
		logger.info("1. in bindIndexType cache searchables    : " + this.cache.searchableFieldsPerIndexType);
		logger.info("1. in bindIndexType cache highlightables : " + this.cache.highlightableFieldsPerIndexType);
	}

	/**
	 * Refreshes the index types
	 */
	public void refreshIndexTypesOfIndex() {
		ClusterStateResponse clusterResponse = this.indexClient.admin().cluster().prepareState().get();
		logger.info("clusterResponse : " + clusterResponse);
		
		ImmutableOpenMap<String, ImmutableOpenMap<String, AliasMetaData>> aliases = clusterResponse.getState().getMetaData().aliases();
		logger.info("aliases : " + aliases);
		
		ImmutableOpenMap<String, AliasMetaData> alias = aliases.get(ACTIVE_INDEX);
		logger.info("alias : " + alias);
		
		if (alias != null && alias.size() > 0){
			logger.info("alias is not null");
			
			for (String indexName : Sets.newHashSet(alias.keysIt())){
				
				logger.info("indexName for alias : " + indexName);
				
				IndexMetaData indexMetaData = clusterResponse.getState().getMetaData().index(indexName);
				logger.info("indexMetaData : " + indexMetaData);
				
				ImmutableOpenMap<String, MappingMetaData> mappings = indexMetaData.mappings();
				logger.info("mappings : " + mappings);
				
				Set<String> indexTypes = new HashSet<String>();
				if (mappings != null && mappings.size() > 0){
					logger.info("mappings not null. creating index types from the keys...");
					indexTypes = Sets.newHashSet(mappings.keysIt());
					logger.info("mappings not null. creating index types from the keys...OK");
				}
				
				logger.info("index types found in index : " + indexTypes);
	
				for (String indexType : indexTypes) {
					logger.info("adding index type : " + indexType);
					try {
						this.bindIndexType(indexType, indexName);
						logger.info("adding index type : " + indexType + " succeded");
					} catch (Exception e) {
						logger.info("adding index type : " + indexType + " failed");
					}
				}
			}
		}
		
	}

	@SuppressWarnings("unchecked")
	public boolean updateManagerProperties(HashMap<String, FullTextIndexType> colForIdTypes) {

		// Retrieving current collectionIds and fields
		Map<String, Object> result = null;
		String id = null;
		long version = 0;
		try {
			SearchResponse response = this.indexClient.prepareSearch(META_INDEX)
					.setQuery(QueryBuilders.matchAllQuery())
					.get();
			for (SearchHit hit : response.getHits().getHits()) {
				result = hit.getSource();
				id = hit.getId();
				version = hit.getVersion();
			}

		} catch (Exception e) {
			logger.warn("No meta-index yet", e);
		}
		
		
		List<String> collectionIdsToBeAdded = new ArrayList<String>();
		List<String> fieldsToBeAdded = new ArrayList<String>();
		
		if (result != null &&  (List<String>) result.get("collectionIDs") != null)
			collectionIdsToBeAdded = (List<String>) result.get("collectionIDs");
		if (result != null && (List<String>) result.get("fields") != null)
			fieldsToBeAdded = (List<String>) result.get("fields");

		Set<String> keys = colForIdTypes.keySet();
		for (String colIDandLang : keys) {
			FullTextIndexType index = colForIdTypes.get(colIDandLang);
			IndexField[] fields = index.getFields();
			// add to the Manager properties, the fields retrieved per
			// collection and language
			for (IndexField field : fields) {
				if (!RowsetParser.addToFieldInfo(fieldsToBeAdded, field.name, colIDandLang, index)) {
					return false;
				}
			}

			// add info for the special field allIndexes supported by FT, for
			// this lang and colID
			fieldsToBeAdded.add(colIDandLang + IndexType.SEPERATOR_FIELD_INFO + IndexType.SEARCHABLE_TAG + IndexType.SEPERATOR_FIELD_INFO + ALL_INDEXES);
			// removing lang from colIDandLang to add just the collection part to the collection ids
			collectionIdsToBeAdded.add(colIDandLang.split(":")[0]);

		}

		// remove duplicates
		HashSet<String> hs1 = new HashSet<String>();
		HashSet<String> hs2 = new HashSet<String>();
		hs1.addAll(collectionIdsToBeAdded);
		hs2.addAll(fieldsToBeAdded);
//		collectionIdsToBeAdded.clear();
//		fieldsToBeAdded.clear();
//		collectionIdsToBeAdded.addAll(hs1);
//		fieldsToBeAdded.addAll(hs2);

		try {
			Map<String, Object> document = new HashMap<String, Object>();
			document.put("collectionIDs", hs1);
			document.put("fields", hs2);
			
			if (result != null)
				version = version + 1;
			
			IndexResponse response = this.indexClient.prepareIndex(META_INDEX, META_INDEX, id)
		        .setSource(ElasticSearchHelper.createJSONObject(document).string())
		        .setConsistencyLevel(WriteConsistencyLevel.ONE)
		        .setVersion(version)
		        .get();

			logger.info("Add records to meta-index response id : " + response.getId());
			logger.info("Inserted colIDs and fields to " + META_INDEX);
			logger.info("committing meta-index");
			commitMeta();
			logger.info("Added records to meta-index");
		} catch (IOException e) {
			logger.error("Caught an exception: ", e);
		}

		return true;
	}
	
	
	
	@SuppressWarnings("unchecked")
	public boolean updateManagerPropertiesRemoveCollID(String collectionID) {

		logger.info("removing collection : " + collectionID + " from meta index");
		// Retrieving current collectionIds and fields
		Map<String, Object> result = null;
		String id = null;
		long version = 0;
		try {
			SearchResponse response = this.indexClient.prepareSearch(META_INDEX)
					.setQuery(QueryBuilders.matchAllQuery())
					.get();
			
			for (SearchHit hit : response.getHits().getHits()) {
				result = hit.getSource();
				id = hit.getId();
				version = hit.getVersion();
			}

		} catch (Exception e) {
			logger.warn("No meta-index yet", e);
		}
		

		List<String> collectionIdsToBeAdded = new ArrayList<String>();
		List<String> fieldsToBeAdded = new ArrayList<String>();
		if (result != null && (List<String>) result.get("collectionIDs") != null)
			collectionIdsToBeAdded = (List<String>) result.get("collectionIDs");
		if (result != null && (List<String>) result.get("fields") != null)
			fieldsToBeAdded = (List<String>) result.get("fields");
		
		
		if (collectionIdsToBeAdded.contains(collectionID)){
			collectionIdsToBeAdded.remove(collectionID);
		}
		
		List<String> fieldsTobeRemoved = new ArrayList<String>();
		for (String field : fieldsToBeAdded) {
			if (field.startsWith(collectionID + ":"))
				fieldsTobeRemoved.add(field);
		}
		
		fieldsToBeAdded.removeAll(fieldsTobeRemoved);


		// remove duplicates
		HashSet<String> hs1 = new HashSet<String>();
		HashSet<String> hs2 = new HashSet<String>();
		hs1.addAll(collectionIdsToBeAdded);
		hs2.addAll(fieldsToBeAdded);

		try {
			Map<String, Object> document = new HashMap<String, Object>();
			document.put("collectionIDs", hs1);
			document.put("fields", hs2);
			
			if (result != null)
				version = version + 1;
			
			logger.info("meta will contain : " + document + " now");
			
			IndexResponse response = this.indexClient.prepareIndex(META_INDEX, META_INDEX, id)
		        .setSource(ElasticSearchHelper.createJSONObject(document).string())
		        .setConsistencyLevel(WriteConsistencyLevel.ONE)
		        .setVersion(version)
		        .get();

			logger.info("Add records to meta-index response id : " + response.getId());
			logger.info("Inserted colIDs and fields to " + META_INDEX);
			logger.info("committing meta-index");
			commitMeta();
			logger.info("Added records to meta-index");
		} catch (IOException e) {
			logger.error("Caught an exception: ", e);
		}

		return true;
	}
	
	

	public boolean addMetaIndex() {
		int counter = 0;
		Map<String, Object> result = null;
		try {
			Thread.sleep(2000);
		} catch (InterruptedException e) {
			logger.error("Failed to sleep", e);
		}
		// check if there is a meta-index
		while (true) {
			try {
				SearchResponse response = this.indexClient.prepareSearch(META_INDEX)
						.setQuery(QueryBuilders.matchAllQuery())
						.get();
				
				for (SearchHit hit : response.getHits().getHits()) {
					result = hit.getSource();
				}
				break;
			} catch (IndexMissingException e) {
				logger.warn("Index missing, proceeding to creation");
				break;
			} catch (Exception e) {
				logger.warn("Not initialized yet, retrying");
				counter++;
				try {
					Thread.sleep(1000);
				} catch (InterruptedException e1) {
					logger.error("Failed to sleep", e);
				}
				if (counter == 10)
					return false;
			}
		}
		// if not add it
		if (result == null) {
			try {
				Map<String, Object> document = new HashMap<String, Object>();
				document.put("collectionIDs", new String[0]);
				document.put("fields", new String[0]);
				
				IndicesAdminClient iac = this.indexClient.admin().indices();
				logger.info("creating meta-index response");
				CreateIndexResponse cir = iac.prepareCreate(META_INDEX).setSettings(this.getIndexCreateSetting()).get();
				logger.info("created meta-index response : " + cir.toString());
				
				IndexResponse response = this.indexClient.prepareIndex(META_INDEX, META_INDEX)
			        .setSource(ElasticSearchHelper.createJSONObject(document).string())
			        .setConsistencyLevel(WriteConsistencyLevel.ONE)
			        .execute()
			        .actionGet();

				logger.info("Add empty records to meta-index response id : " + response.getId());
				logger.info("committing meta-index");
				commitMeta();
				logger.info("Added empty meta-index");
			} catch (IOException e) {
				logger.error("Failed to add meta index", e);
				return false;
			}
		}
		return true;
	}

	
	
	public void invalidateCache() {
		this.cache.invalidate();
	}

	public boolean rebuildMetaIndex(List<String> collectionIds, List<String> fields) {
		DeleteMappingRequest deleteMapping = new DeleteMappingRequest(META_INDEX).types(META_INDEX);
		this.indexClient.admin().indices().deleteMapping(deleteMapping).actionGet();
		try {
			Map<String, Object> document = new HashMap<String, Object>();
			document.put("collectionIDs", collectionIds);
			document.put("fields", fields);
			
			this.indexClient.prepareIndex(META_INDEX, META_INDEX)
					.setSource(ElasticSearchHelper.createJSONObject(document).string())
					.setConsistencyLevel(WriteConsistencyLevel.ONE)
					.get();
			
			logger.info("Rebuilt meta-index");
		} catch (IOException e) {
			logger.error("Failed to rebuild meta index", e);
			return false;
		}
		return true;
	}
	
	
	
	public Map<String, List<String>> getCollectionsAndFieldsOfIndex(){
		SearchResponse response = this.indexClient.prepareSearch(META_INDEX).setQuery(QueryBuilders.matchAllQuery()).execute().actionGet();
		try {
			@SuppressWarnings("unchecked")
			List<String> collections = (List<String>) response.getHits().getHits()[0].getSource().get("collectionIDs");
			@SuppressWarnings("unchecked")
			List<String> fields = (List<String>) response.getHits().getHits()[0].getSource().get("fields");
			
			Map<String, List<String>> result = new HashMap<>();
			result.put("collections", collections);
			result.put("fields", fields);
			
			return result;
		} catch (Exception e) {
			logger.warn("Error while getting collections of index", e);
			return null;
		}
	}
	
	public List<String> getCollectionOfIndex(){
		SearchResponse response = this.indexClient.prepareSearch(META_INDEX).setQuery(QueryBuilders.matchAllQuery()).execute().actionGet();
		try {
			@SuppressWarnings("unchecked")
			List<String> collections = (List<String>) response.getHits().getHits()[0].getSource().get("collectionIDs");
			return collections;
		} catch (Exception e) {
			logger.warn("Error while getting collections of index", e);
			return null;
		}
	}
	
	public List<String> getFieldsOfIndex(){
		SearchResponse response = this.indexClient.prepareSearch(META_INDEX).setQuery(QueryBuilders.matchAllQuery()).execute().actionGet();
		try {
			@SuppressWarnings("unchecked")
			List<String> fields = (List<String>) response.getHits().getHits()[0].getSource().get("fields");
			return fields;
		} catch (Exception e) {
			logger.warn("Error while getting fields of index", e);
			return null;
		}
	}
	
	private Settings getIndexCreateSetting() {
		Settings settings = ImmutableSettings.builder()
				.put("index.number_of_replicas", String.valueOf(this.noOfReplicas))
				.put("index.number_of_shards", String.valueOf(this.noOfShards))
				.put("index.store", DEFAULT_INDEXSTORE)
				.put("index.refresh_interval", INDEX_REFRESH_INTERVAL)
				.put("path.data", this.dataDir)
			.build();
		
		return settings;
	}
	
	
	/////// Attachments in index
	
	static final String FILE_INDEX_NAME = "file-index";
	static final String FILE_INDEX_TYPE = "file-index-type";
	
	
	public Boolean createFileIndex(){
		return createFileIndex(this.indexClient, FILE_INDEX_NAME, FILE_INDEX_TYPE, this.getIndexCreateSetting());
	}
	
	
	public static Boolean createFileIndex(Client indexClient, String indexName, String indexType, Settings settings) {
		
		
		IndicesAdminClient iac = indexClient.admin().indices();

		logger.info("Checking if index exists");
		if (iac.prepareExists(indexName).get().isExists()) {
			logger.info("Index already exists");
			return true;
		} else {
			CreateIndexRequestBuilder createIndexRequest = iac.prepareCreate(indexName)
					.setSettings(settings);
			logger.info("Create Index Request : " + createIndexRequest.request());
			
			CreateIndexResponse cir = createIndexRequest.get();
			logger.info("Create Index Response : " + cir);
			XContentBuilder fileIndexMapping = getFileIndexMapping(indexType);
			
			
			
			PutMappingResponse pmr = iac.preparePutMapping().setIndices(indexName).setType(indexType).setSource(fileIndexMapping).get();
			return pmr.isAcknowledged();
		}
		
	}
	
	
	//files
	
	
	public void addFile(String base64) throws IOException{
		addFile(this.indexClient, base64, FILE_INDEX_NAME, FILE_INDEX_TYPE);
	}
	
	public static void addFile(Client indexClient, String base64, String indexName, String indexType) throws IOException {
		Map<String, String> doc = new HashMap<>();
		doc.put("file", base64);
		
		XContentBuilder json = ElasticSearchHelper.createJSONObject(doc);
		logger.info("inserting file");
		logger.info(json.string());
		
		indexClient.prepareIndex(indexName, indexType)
			.setSource(json)
			.get();
	}
	
	
	private static XContentBuilder getFileIndexMapping(String type){
	    XContentBuilder builder = null;
	    try {
	        builder = XContentFactory.jsonBuilder().startObject().startObject(type)
	        		.startObject("properties")
	        			.startObject("file")
	                    	.field("type", "attachment")
		                    .startObject("fields")
		                    	.startObject("file")
		                    		.field("index", "analyzed")
		                    		//XXX:TODO pick the right highlighter
		                    		//.field("index_options" , "offsets")
		                    		//.field("term_vector", "with_positions_offsets")
		                    		.field("store", "true")
		                    	.endObject()
		                    	.startObject("date")
		                    	 	.field("index", "analyzed")
		                    	 	.field("store", "true")
		                    	.endObject()
		                    	.startObject("author")
		                    		.field("index", "analyzed")
		                    		.field("store", "true")
		                    	.endObject()
		                    	.startObject("content")
		                    		.field("store", "true")
		                    	.endObject()
		                    	.startObject("title")
		                    		.field("index", "analyzed")
		                    		.field("store", "true")
		                    	.endObject()
		                    	
		                    .endObject()
		                  .endObject()
		               .endObject();
	        }
	        catch (IOException e) {
	            e.printStackTrace();
	        }
	    return builder;
	}

	
	// Clustering 
	
	
	
	public List<ClusterResponse> cluster(String query, String queryHint, Integer numberOfClusters, String urlField, List<String> titleFields, List<String> contentFields, List<String> languageFields, Set<String> sids, String algorithm, Integer searchHits){
		
		return FullTextNodeHelpers.clustering(this.indexClient, query, queryHint, numberOfClusters, urlField, titleFields, contentFields, languageFields, this.rradaptor, sids, algorithm, searchHits);
		
	}

	
	
	public void listAlgorithms() {
		ListAlgorithmsActionResponse ral = ListAlgorithmsAction.INSTANCE.newRequestBuilder(this.indexClient).get();
		logger.info("algorithms : ",  ral.getAlgorithms());
		System.out.println(ral.getAlgorithms());
		
		
		SearchRequestBuilder srb = this.indexClient.prepareSearch(ACTIVE_INDEX)
				//.setSearchType(SearchType.DFS_QUERY_THEN_FETCH)
				.setSize(10000)
				.setFrom(0)
				.setQuery(QueryBuilders.matchQuery("_all", "soccer")).addFields("ObjectID", "title", "content");
		
		
		System.out.println(srb.toString());
		
		Integer clusterCount = 2;
		
		ClusteringActionRequestBuilder ca = ClusteringAction.INSTANCE.newRequestBuilder(this.indexClient).
				setAlgorithm(ral.getAlgorithms().get(1)).
				setQueryHint("soccer").
				setSearchRequest(srb.request()).
				addFieldMapping("ObjectID", LogicalField.URL).
				addFieldMapping("title", LogicalField.TITLE).
				addFieldMapping("content", LogicalField.CONTENT).addAttribute("LingoClusteringAlgorithm.desiredClusterCountBase", clusterCount);
			
		
		
		System.out.println(ca.toString());
		
		ClusteringActionResponse car = ca.get();
		
		SearchHit[] hits = car.getSearchResponse().getHits().getHits();
		
		DocumentGroup[] groups = car.getDocumentGroups();
		for (DocumentGroup group : groups)
			System.out.println(group.getLabel());
		
		for (SearchHit hit : hits){
			System.out.println(hit.getSourceAsString());
		}
		
	}
}
