package org.gcube.elasticsearch;

import static org.elasticsearch.node.NodeBuilder.nodeBuilder;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import org.elasticsearch.ElasticSearchException;
import org.elasticsearch.action.WriteConsistencyLevel;
import org.elasticsearch.action.admin.cluster.state.ClusterStateResponse;
import org.elasticsearch.action.admin.indices.create.CreateIndexResponse;
import org.elasticsearch.action.admin.indices.mapping.delete.DeleteMappingRequest;
import org.elasticsearch.action.admin.indices.mapping.put.PutMappingResponse;
import org.elasticsearch.action.bulk.BulkItemResponse;
import org.elasticsearch.action.bulk.BulkRequestBuilder;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.index.IndexRequestBuilder;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.client.Client;
import org.elasticsearch.client.IndicesAdminClient;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.cluster.metadata.MappingMetaData;
import org.elasticsearch.common.collect.ImmutableMap;
import org.elasticsearch.common.settings.ImmutableSettings;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.index.query.QueryBuilders;
import org.elasticsearch.indices.IndexMissingException;
import org.elasticsearch.node.Node;
import org.elasticsearch.search.SearchHit;
import org.gcube.common.core.contexts.GHNContext;
import org.gcube.elasticsearch.helpers.ElasticSearchHelper;
import org.gcube.elasticsearch.helpers.QueryParser;
import org.gcube.elasticsearch.helpers.RowsetParser;
import org.gcube.indexmanagement.common.FullTextIndexType;
import org.gcube.indexmanagement.common.IndexException;
import org.gcube.indexmanagement.common.IndexField;
import org.gcube.indexmanagement.common.IndexType;
import org.gcube.indexmanagement.common.linguistics.languageidplugin.LanguageIdPlugin;
import org.gcube.indexmanagement.lucenewrapper.LuceneGcqlQueryContainer;
import org.gcube.indexmanagement.resourceregistry.RRadaptor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import gr.uoa.di.madgik.commons.server.ConnectionManagerConfig;
import gr.uoa.di.madgik.commons.server.PortRange;
import gr.uoa.di.madgik.commons.server.TCPConnectionManager;
import gr.uoa.di.madgik.grs.buffer.GRS2BufferException;
import gr.uoa.di.madgik.grs.buffer.IBuffer.Status;
import gr.uoa.di.madgik.grs.events.KeyValueEvent;
import gr.uoa.di.madgik.grs.proxy.tcp.TCPConnectionHandler;
import gr.uoa.di.madgik.grs.proxy.tcp.TCPStoreConnectionHandler;
import gr.uoa.di.madgik.grs.reader.ForwardReader;
import gr.uoa.di.madgik.grs.reader.GRS2ReaderException;
import gr.uoa.di.madgik.grs.record.GRS2RecordDefinitionException;
import gr.uoa.di.madgik.grs.record.GenericRecord;
import gr.uoa.di.madgik.grs.record.Record;
import gr.uoa.di.madgik.grs.writer.GRS2WriterException;
import gr.uoa.di.madgik.grs.writer.RecordWriter;
import gr.uoa.di.madgik.rr.ResourceRegistry;
import gr.uoa.di.madgik.rr.ResourceRegistryException;

public class FullTextNode {
	private static final Logger logger = LoggerFactory.getLogger(FullTextNode.class);

	private static int DEFAULT_NUM_OF_REPLICAS = 0;
	private static int DEFAULT_NUM_OF_SHARDS = 1;
	private static int MAX_FRAGMENT_CNT = 5;
	private static int MAX_FRAGMENT_SIZE = 150;
	private static String DEFAULT_DATADIR = ".";
	
	private static final long RSTIMEOUT = 5;
	private static String ALL_INDEXES = "allIndexes";
	public static String META_INDEX = "meta-index";
	public static String DEFAULT_ANALYZER = "simple";
	public static String COLLECTION_ID_ANALYZER = "keyword";
	

	private Client indexClient;
	private Node indexNode;
	private Set<String> indexTypes = new HashSet<String>();
	private FTNodeCache cache;

	private String clusterName;
	private String indexName;
	private Integer noOfReplicas;
	private Integer noOfShards;
	private String scope;
	private int maxFragmentCnt;
	private int maxFragmentSize;
	private String dataDir = DEFAULT_DATADIR;
	private RRadaptor rradaptor;

	private HashMap<String, FullTextIndexType> colForField = new HashMap<String, FullTextIndexType>();

	// Constructors

	public FullTextNode() {
		logger.info("Initializing FullTextNode");
		this.cache = new FTNodeCache();
		this.initialize();
	}

	public FullTextNode(String clusterName, String indexName, Integer noOfReplicas, Integer noOfShards, String scope, int maxFragmentCnt, int maxFragmentSize) {
		this();
		this.clusterName = clusterName;
		this.indexName = indexName;
		this.noOfReplicas = noOfReplicas;
		this.noOfShards = noOfShards;
		this.scope = scope;
		this.maxFragmentCnt = maxFragmentCnt;
		this.maxFragmentSize = maxFragmentSize;
	}

	public FullTextNode(String clusterName, String indexName, String scope) {
		this();
		this.clusterName = clusterName;
		this.indexName = indexName;
		this.noOfReplicas = DEFAULT_NUM_OF_REPLICAS;
		this.noOfShards = DEFAULT_NUM_OF_SHARDS;
		this.scope = scope;
		this.maxFragmentCnt = MAX_FRAGMENT_CNT;
		this.maxFragmentSize = MAX_FRAGMENT_SIZE;
	}
	
	public FullTextNode(String dataDir, String clusterName, String indexName, Integer noOfReplicas, Integer noOfShards, String scope, int maxFragmentCnt, int maxFragmentSize) {
		this(clusterName, indexName, noOfReplicas, noOfShards, scope, maxFragmentCnt, maxFragmentSize);
		this.dataDir = dataDir;
	}

	public FullTextNode(String dataDir, String clusterName, String indexName, String scope) {
		this(clusterName, indexName, scope);
		this.dataDir = dataDir;
	}

	// End of Constructors

	public String getClusterName() {
		return clusterName;
	}

	public String getIndexName() {
		return indexName;
	}

	public Integer getNoOfReplicas() {
		return noOfReplicas;
	}

	public Integer getNoOfShards() {
		return noOfShards;
	}

	public String getScope() {
		return scope;
	}

	public int getMaxFragmentCnt() {
		return maxFragmentCnt;
	}

	public int getMaxFragmentSize() {
		return maxFragmentSize;
	}

	public Client getIndexClient() {
		return indexClient;
	}

	@Override
	public String toString() {
		StringBuilder builder = new StringBuilder();
		builder.append("FullTextNode [indexNode=").append(indexNode).append(", cache=").append(cache).append(", clusterName=").append(clusterName)
				.append(", indexName=").append(indexName).append(", noOfReplicas=").append(noOfReplicas).append(", noOfShards=").append(noOfShards)
				.append(", scope=").append(scope).append(", maxFragmentCnt=").append(maxFragmentCnt).append(", maxFragmentSize=").append(maxFragmentSize)
				.append("]");
		return builder.toString();
	}

	public synchronized void addIndexType(String indexTypeStr) throws Exception {
		FullTextIndexType indexType = QueryParser.retrieveIndexType(indexTypeStr, this.scope, this.cache);
		this.addIndexType(indexTypeStr, indexType);
	}

	public synchronized void addIndexType(String indexTypeStr, FullTextIndexType indexType) throws Exception {
		logger.info("Calling addIndexType");
		if (indexType == null) {
			logger.warn("IndexType is null");
			throw new Exception("Trying to null as IndexType. Check how you got it");
		}

		// in case indexType is not in cache
		indexType = QueryParser.addFullTextIndexTypeIntoCache(indexTypeStr, this.scope, indexType, this.cache);

		// create mapping if not exists
		IndicesAdminClient iac = this.indexClient.admin().indices();

		logger.info("Checking if index exists");
		if (iac.prepareExists(this.indexName).execute().actionGet().isExists()) {
			logger.info("Index already exists");
		} else {
			CreateIndexResponse cir = iac.prepareCreate(this.indexName).execute().actionGet();
			logger.info("Create Index Response : " + cir);
		}

		logger.info("Index Type");
		logger.info("-----------------------------------------------");
		logger.info(indexType.toString());
		logger.info("-----------------------------------------------");

		Map<String, Object> mapping = new HashMap<String, Object>();
		List<String> presentables = new ArrayList<String>();
		List<String> searchables = new ArrayList<String>();

		for (IndexField idxTypeField : indexType.getFields()) {
			String store, index;

			store = idxTypeField.store ? "yes" : "no";

			if (idxTypeField.index)
				index = idxTypeField.tokenize ? "analyzed" : "not_analyzed";
			else
				index = "no";

			if (idxTypeField.name.equalsIgnoreCase(IndexType.COLLECTION_FIELD))
				index = "analyzed";

			Map<String, Object> fieldMap = new HashMap<String, Object>();
			fieldMap.put("type", "string");
			fieldMap.put("store", store);
			fieldMap.put("index", index);
			
			if (idxTypeField.name.equalsIgnoreCase(IndexType.COLLECTION_FIELD))
				fieldMap.put("analyzer", COLLECTION_ID_ANALYZER);
			else
				fieldMap.put("analyzer", DEFAULT_ANALYZER);

			mapping.put(idxTypeField.name, fieldMap);

			if (idxTypeField.returned /*&& !idxTypeField.name.equalsIgnoreCase(IndexType.COLLECTION_FIELD)*/)
				presentables.add(idxTypeField.name);
			if (idxTypeField.index)
				searchables.add(idxTypeField.name);
		}
		
		//Add objectid field by default
		Map<String, Object> fieldMap = new HashMap<String, Object>();
		fieldMap.put("type", "string");
		fieldMap.put("store", "yes");
		fieldMap.put("index", "analyzed");
		fieldMap.put("analyzer", DEFAULT_ANALYZER);
		mapping.put(IndexType.DOCID_FIELD.toLowerCase(), fieldMap);

		this.cache.presentableFieldsPerIndexType.put(indexTypeStr, presentables);
		this.cache.searchableFieldsPerIndexType.put(indexTypeStr, searchables);

		Map<String, Object> propertyMap = new HashMap<String, Object>();
		propertyMap.put("properties", mapping);

		// /logger.info("propertyMap : " + propertyMap);
		Map<String, Object> mappingMap = new HashMap<String, Object>();
		mappingMap.put(indexTypeStr, propertyMap);

		String json = ElasticSearchHelper.createJSONObject(mappingMap).string();
		logger.info("json : " + json);

		PutMappingResponse pmr = iac.preparePutMapping().setIndices(this.indexName).setType(indexTypeStr).setSource(json).execute().actionGet();
		logger.info("Update Settings Response : " + pmr.toString());

		this.indexTypes.add(indexTypeStr);
	}

	public void createOrJoinCluster() {
		logger.info("creating or joining cluster");
		logger.info("cluster.name : " + this.clusterName);
		logger.info("index.number_of_replicas : " + this.noOfReplicas);
		logger.info("index.number_of_shards : " + this.noOfShards);
		logger.info("path.data : " + this.dataDir);

		Settings settings = ImmutableSettings.settingsBuilder()
				.put("cluster.name", this.clusterName)
				.put("index.number_of_replicas", String.valueOf(this.noOfReplicas))
				.put("index.number_of_shards", String.valueOf(this.noOfShards))
				.put("index.refresh_interval", String.valueOf(-1))
				.put("path.data", this.dataDir)
				// .put("discovery.zen.ping.multicast.ping.enabled", false)
				// .put("discovery.zen.ping.multicast.enabled",false)
				// .put("discovery.zen.ping.unicast.enabled",true)
				// .put("discovery.zen.ping.unicast.hosts ",
				// "jazzman.di.uoa.gr")
				.build();

		this.indexNode = nodeBuilder().settings(settings).node();
		this.indexClient = this.indexNode.client();
	}

	/***
	 * 
	 * @param knownNodes
	 *            Map of Address -> Port
	 * @throws IOException
	 */
	public void joinCluster(Map<String, Integer> knownNodes) throws IOException {
		logger.info("joining cluster of known node : " + knownNodes);
		logger.info("cluster.name : " + this.clusterName);
		logger.info("index.number_of_replicas : " + this.noOfReplicas);
		logger.info("index.number_of_shards : " + this.noOfShards);
		logger.info("path.data : " + this.dataDir);

		String hosts = ElasticSearchHelper.createKnownHostsString(knownNodes);

		logger.info("hosts string : " + hosts);

		Settings settings = ImmutableSettings.settingsBuilder().put("cluster.name", this.clusterName).put("index.number_of_replicas", String.valueOf(this.noOfReplicas))
				.put("index.number_of_shards", String.valueOf(this.noOfShards)).put("discovery.zen.ping.multicast.ping.enabled", false)
				.put("path.data", this.dataDir)
				.put("discovery.zen.ping.multicast.enabled", false).put("discovery.zen.ping.unicast.enabled", true)
				.put("discovery.zen.ping.unicast.hosts", hosts).build();

		this.indexNode = nodeBuilder().settings(settings).node();
		this.indexClient = this.indexNode.client();
	}

	/**
	 * 
	 * @param queryString
	 * @return gRS2 locator of the query results
	 * @throws IndexException
	 * @throws GRS2WriterException
	 */
	public String query(String queryString) throws IndexException, GRS2WriterException {
		return query(queryString, -1);
	}

	/**
	 * Executes the query of gRS2 locator of maximum maxHits records (if >0)
	 * 
	 * @param queryString
	 * @param maxHits
	 * @return gRS2 locator of the results
	 * @throws IndexException
	 * @throws GRS2WriterException
	 */
	public String query(String queryString, int maxHits) throws GRS2WriterException, IndexException {

		logger.info("queryString received : " + queryString);

		final List<String> collIDs = QueryParser.getCollectionsIDFromQuery(queryString);
		logger.info("collectionID of query : " + collIDs);

		final Set<String> indexTypes = QueryParser.getIndexTypesByCollectionIDs(this.cache.indexTypesByCollIDs, collIDs, this.indexClient, this.indexName);
		logger.info("indexTypes for collectionIDs : " + indexTypes);

		final List<String> presentables = QueryParser.createPresentableForIndexTypes(this.cache.presentableFieldsPerIndexType, indexTypes);
		logger.info("presentables for index types : " + presentables);

		final List<String> searchables = QueryParser.createSearchablesForIndexTypes(this.cache.searchableFieldsPerIndexType, indexTypes);
		logger.info("searchables for index types : " + searchables);

		logger.info("queryString before convert : " + queryString);
		final LuceneGcqlQueryContainer queryContainer = QueryParser.convertToLuceneQuery(queryString, presentables, searchables, this.rradaptor);

		final String query = QueryParser.getLuceneQueryFromQueryString(queryContainer);
		logger.info("query part of queryString : " + query);

		final List<String> projections = QueryParser.getProjectionsQueryFromQueryString(queryContainer);
		logger.info("projections of queryString : " + projections);

		logger.info("queryString after convert : " + query + " project " + projections);

		final SearchHit[] hits = projections.contains(IndexType.SNIPPET) ? ElasticSearchHelper.queryElasticSearch(this.indexClient, this.indexName, query,
				maxHits, presentables, this.maxFragmentSize, this.maxFragmentCnt) : ElasticSearchHelper.queryElasticSearch(this.indexClient, this.indexName,
				query, maxHits);
		logger.info("Number of hits returned by index : " + hits.length);

		final RecordWriter<GenericRecord> rsWriter = QueryParser.initRSWriterForSearchHits(presentables, projections, this.rradaptor);
		// send an event for the total number of results

		logger.info("emitting key value event with key : " + IndexType.RESULTSNOFINAL_EVENT + " and value : " + hits.length);

		rsWriter.emit(new KeyValueEvent(IndexType.RESULTSNOFINAL_EVENT, String.valueOf(hits.length)));

		final int maxFragmentCnt = this.maxFragmentCnt;
		Runnable writerRun = new Runnable() {
			public void run() {
				try {
					for (SearchHit hit : hits) {
						// while the reader hasn't stopped reading
						if (!QueryParser.writeSearchHitInResultSet(hit, rsWriter, projections, presentables, maxFragmentCnt, RSTIMEOUT))
							break;
					}
					if (rsWriter.getStatus() != Status.Dispose)
						rsWriter.close();
				} catch (Exception e) {
					logger.error("Error during search.", e);
					try {
						if (rsWriter.getStatus() != Status.Dispose)
							rsWriter.close();
					} catch (Exception ex) {
						logger.error("Error while closing RS writer.", ex);
					}
				}
			}
		};
		new Thread(writerRun).start();

		logger.info("results locator : " + rsWriter.getLocator());

		return rsWriter.getLocator().toString();
	}

	public String queryStream(String queryString) throws IndexException, GRS2WriterException {
		return queryStream(queryString, -1);
	}

	/**
	 * Important: The returned results are not sorted!
	 * 
	 * @param queryString
	 * @return gRS2 locator of the results
	 * @throws IndexException
	 * @throws GRS2WriterException
	 */
	public String queryStream(String queryString, final int maxHits) throws IndexException, GRS2WriterException {
		logger.info("queryString received : " + queryString);

		final List<String> collIDs = QueryParser.getCollectionsIDFromQuery(queryString);
		logger.info("collectionID of query : " + collIDs);

		final Set<String> indexTypes = QueryParser.getIndexTypesByCollectionIDs(this.cache.indexTypesByCollIDs, collIDs, this.indexClient, this.indexName);
		logger.info("indexTypes for collectionIDs : " + indexTypes);

		final List<String> presentables = QueryParser.createPresentableForIndexTypes(this.cache.presentableFieldsPerIndexType, indexTypes);
		logger.info("presentables for index types : " + presentables);

		final List<String> searchables = QueryParser.createSearchablesForIndexTypes(this.cache.searchableFieldsPerIndexType, indexTypes);
		logger.info("searchables for index types : " + searchables);

		logger.info("queryString before convert : " + queryString);
		final LuceneGcqlQueryContainer queryContainer = QueryParser.convertToLuceneQuery(queryString, presentables, searchables, this.rradaptor);

		final String query = QueryParser.getLuceneQueryFromQueryString(queryContainer);
		logger.info("query part of queryString : " + query);

		final List<String> projections = QueryParser.getProjectionsQueryFromQueryString(queryContainer);
		logger.info("projections of queryString : " + projections);

		logger.info("queryString after convert : " + query + " project " + projections);

		long numberOfHits = ElasticSearchHelper.queryCountElasticSearch(this.indexClient, this.indexName, query);

		if (maxHits > 0 && numberOfHits > maxHits)
			numberOfHits = maxHits;

		logger.info("Number of hits returned by index : " + numberOfHits);

		final RecordWriter<GenericRecord> rsWriter = QueryParser.initRSWriterForSearchHits(presentables, projections, this.rradaptor);
		// send an event for the total number of results

		logger.info("emitting key value event with key : " + IndexType.RESULTSNOFINAL_EVENT + " and value : " + numberOfHits);

		rsWriter.emit(new KeyValueEvent(IndexType.RESULTSNOFINAL_EVENT, String.valueOf(numberOfHits)));

		final Client client = this.indexClient;
		final String indexName = this.indexName;
		final int maxFragmentSize = this.maxFragmentSize;
		final int maxFragmentCnt = this.maxFragmentCnt;

		Runnable writerRun = new Runnable() {

			public void run() {
				try {
					SearchResponse scrollResp = projections.contains(IndexType.SNIPPET) ? ElasticSearchHelper.queryElasticSearchScroll(client, indexName,
							query, maxHits, presentables, maxFragmentSize, maxFragmentCnt) : ElasticSearchHelper.queryElasticSearchScroll(client,
							indexName, query, maxHits);

					int hits = 0;

					while (true) {
						scrollResp = ElasticSearchHelper.getNextSearchResponse(client, scrollResp);
						logger.info("hits from scroll : " + scrollResp.getHits().getHits().length);

						for (SearchHit hit : scrollResp.getHits()) {
							// while the reader hasn't stopped reading
							if (!QueryParser.writeSearchHitInResultSet(hit, rsWriter, projections, presentables, maxFragmentCnt, RSTIMEOUT))
								break;
						}
						if (rsWriter.getStatus() != Status.Dispose)
							rsWriter.close();

						if (scrollResp.getHits().getHits().length == 0) {
							break;
						}
						hits++;
						if (hits > maxHits)
							break;
					}
				} catch (Exception e) {
					logger.error("Error during search.", e);
					try {
						if (rsWriter.getStatus() != Status.Dispose)
							rsWriter.close();
					} catch (Exception ex) {
						logger.error("Error while closing RS writer.", ex);
					}
				}
			}
		};
		new Thread(writerRun).start();

		logger.info("results locator : " + rsWriter.getLocator());

		return rsWriter.getLocator().toString();
	}

	/**
	 * Deletes the index and the metaIndex
	 * */
	public boolean delete() {
		boolean indexRet = ElasticSearchHelper.delete(this.indexClient, this.indexName);
		boolean metaRet = ElasticSearchHelper.delete(this.indexClient, META_INDEX);
		return indexRet && metaRet;
	}
	
	
	/**
	 * Deletes the documents with IDs within the list <a>docIDs</a>
	 * 
	 * @param docIDs
	 */
	public void deleteDocuments(List<String> docIDs) {
		ElasticSearchHelper.deleteDocuments(this.indexClient, this.indexName, docIDs);
	}
	
	/**
	 * Deletes the documents of the collection with ID <a>collID</a>
	 * 
	 * @param docIDs
	 */
	public Boolean deleteCollection(String collID) {
		Boolean result = ElasticSearchHelper.deleteCollection(this.indexClient, this.indexName, collID);
		if (result) {
			updateManagerPropertiesRemoveCollID(collID);
			this.commitMeta();
		}
		return result;
	}

	/**
	 * Commits the changes to the index by sending a flush and a refresh request
	 */
	public void commitMeta() {
		ElasticSearchHelper.commit(this.indexClient, META_INDEX);
	}

	public void commit() {
		ElasticSearchHelper.commit(this.indexClient, this.indexName);
	}

	public void clearIndex(String indexTypeName) {
		ElasticSearchHelper.clearIndex(this.indexClient, this.indexName, indexTypeName);
	}

	/**
	 * Closes the index
	 */
	public void close() {
		this.indexClient.close();
		this.indexNode.close();
	}

	private void initialize() {
		TCPConnectionManager.Init(new ConnectionManagerConfig(GHNContext.getContext().getHostname(), new ArrayList<PortRange>(), true));
		TCPConnectionManager.RegisterEntry(new TCPConnectionHandler());
		TCPConnectionManager.RegisterEntry(new TCPStoreConnectionHandler());
		logger.info("Initializing ResourceRegistry");
		try {
			ResourceRegistry.startBridging();
			TimeUnit.SECONDS.sleep(1);
			while(!ResourceRegistry.isInitialBridgingComplete()) TimeUnit.SECONDS.sleep(10);
		} catch (ResourceRegistryException e) {
			logger.error("Resource Registry could not be initialized", e);
			return;
		} catch (InterruptedException e) {
			logger.error("Resource Registry could not be initialized", e);
			return;
		}
		this.rradaptor = new RRadaptor(this.scope);
		logger.info("Initializing ResourceRegistry is DONE");
	}

	/**
	 * Feeds the index with a rowset.
	 * 
	 * @param rowset
	 * @return <a>True</a> on success or <a>False</a> on failure
	 */
	public boolean feedRowset(String rowset) {
		BulkRequestBuilder bulkRequest = this.indexClient.prepareBulk();
		HashMap<String, FullTextIndexType> colForIdTypes = new HashMap<String, FullTextIndexType>();
		boolean status = feedRowset(bulkRequest, rowset, colForIdTypes);

		if (status) {
			long before = System.currentTimeMillis();
			BulkResponse bulkResponse = bulkRequest.setConsistencyLevel(WriteConsistencyLevel.ONE).execute().actionGet();
			long after = System.currentTimeMillis();
			logger.info("Time for commiting the bulk requests : " + (after - before) / 1000.0 + " secs");
			logger.info("bulkResponse : " + bulkResponse.getTookInMillis() / 1000.0 + " secs");
			logger.info("bulkResponse : " + bulkResponse);
			if (bulkResponse.hasFailures()) {
				logger.info("failures have happened");
			}
			this.commit();

		} else {
			logger.info("feedRowset failed");
		}

		return status;
	}

	public boolean feedLocator(String resultSetLocation) throws GRS2ReaderException, URISyntaxException, GRS2RecordDefinitionException, GRS2BufferException {
		return feedLocator(resultSetLocation, colForField);
	}

	/**
	 * Feeds the index with rowsets that are read from the the given locator
	 * 
	 * @param resultSetLocation
	 * @param colForField
	 * @return true if feed was successful, otherwise false
	 * @throws GRS2ReaderException
	 * @throws URISyntaxException
	 * @throws GRS2RecordDefinitionException
	 * @throws GRS2BufferException
	 */
	public boolean feedLocator(String resultSetLocation, HashMap<String, FullTextIndexType> colForField) throws GRS2ReaderException, URISyntaxException,
			GRS2RecordDefinitionException, GRS2BufferException {
		// UpdaterService

		long beforeFeed, afterFeed;

		logger.info("Initializing reader at resultset : " + resultSetLocation);
		ForwardReader<Record> reader = new ForwardReader<Record>(new URI(resultSetLocation));
		reader.setIteratorTimeout(RSTIMEOUT);
		reader.setIteratorTimeUnit(TimeUnit.MINUTES);

		int rowSetCount = 0;
		boolean success = true;

		beforeFeed = System.currentTimeMillis();

		BulkRequestBuilder bulkRequest = this.indexClient.prepareBulk();

		// HashMap<String, FullTextIndexType> colForField = new HashMap<String,
		// FullTextIndexType>();

		try {
			logger.info("Initializing resultset reader iterator");
			Iterator<Record> it = reader.iterator();
			long before, after;

			while (it.hasNext()) {
				logger.info("Getting result : " + rowSetCount);
				// System.out.println("Getting result : " + rowSetCount);

				before = System.currentTimeMillis();
				Record result = it.next();
				after = System.currentTimeMillis();
				logger.info("Time for getting record from Result Set : " + (after - before) / 1000.0 + " secs");

				before = System.currentTimeMillis();
				String rowset = RowsetParser.getRowsetFromResult(result);
				after = System.currentTimeMillis();
				logger.info("Time for getting rowset from record : " + (after - before) / 1000.0 + " secs");
				// logger.info("Result rowset : " + rowset);
				success = feedRowset(bulkRequest, rowset, colForField);

				if (success == false) {
					logger.info("feed rowset failed");
					break;
				}

				logger.info("Result " + rowSetCount + " inserted");
				rowSetCount++;
			}
		} catch (Exception e) {
			logger.info("Exception will feeding", e);
		}

		reader.close();

		if (success) {
			long before = System.currentTimeMillis();
			BulkResponse bulkResponse = bulkRequest.setConsistencyLevel(WriteConsistencyLevel.ONE).execute().actionGet();
			long after = System.currentTimeMillis();
			logger.info("Time for commiting the bulk requests : " + (after - before) / 1000.0 + " secs");
			logger.info("bulkResponse : " + bulkResponse.getTookInMillis() / 1000.0 + " secs");
			logger.info("bulkResponse : " + bulkResponse);
			if (bulkResponse.hasFailures()) {
				logger.warn("failures have happened. Message : " + bulkResponse.buildFailureMessage());

				logger.warn("Details ");
				for (BulkItemResponse brItem : bulkResponse.items()) {
					logger.warn("id : " + brItem.getId() + " " + brItem.getIndex() + " " + brItem.getType() + ", Failure Message : " + brItem.getFailureMessage());
				}
				logger.warn("failures have happened. Message : " + bulkResponse.buildFailureMessage());
				logger.warn("Error. Feeding failed");
				return false;
			} else {
				this.commit();
				updateManagerProperties(colForField);
				this.commitMeta();
				logger.info("Total number of records feeded : " + rowSetCount);

				afterFeed = System.currentTimeMillis();
				logger.info("Total insert time : " + (afterFeed - beforeFeed) / 1000.0);
				//System.out.println("Total insert time : " + (afterFeed - beforeFeed) / 1000.0);
			}
		} else {
			logger.warn("Error. Feeding failed");
		}
		return success;
	}

	// ////////////////////////////////////
	// / private methods
	// ////////////////////////////////////

	/**
	 * Adds a rowset in the bulk of a bulkRequest
	 * 
	 * @param bulkRequest
	 * @param rowset
	 * @param colForIdTypes
	 * @return <a>True</a> on success or <a>False</a> on failure
	 */
	private boolean feedRowset(BulkRequestBuilder bulkRequest, String rowset, HashMap<String, FullTextIndexType> colForIdTypes) {
		long before, after;

		before = System.currentTimeMillis();
		String rsIdxTypeID = RowsetParser.getIdxTypeNameRowset(this.indexName, rowset);
		after = System.currentTimeMillis();
		logger.info("Time for getting rsIdxTypeID from rowset : " + (after - before) / 1000.0 + " secs");
		logger.info("Result IndexTypeID : " + rsIdxTypeID);

		before = System.currentTimeMillis();
		String lang = RowsetParser.getLangRowset(this.indexName, rowset);
		after = System.currentTimeMillis();
		logger.info("Time for getting lang from rowset : " + (after - before) / 1000.0 + " secs");

		logger.info("Result lang : " + lang);
		// if no language is detected set the unknown
		if (lang == null || lang.equals(""))
			lang = LanguageIdPlugin.LANG_UNKNOWN;

		before = System.currentTimeMillis();
		String colID = RowsetParser.getColIDRowset(this.indexName, rowset);
		after = System.currentTimeMillis();
		logger.info("Time for colID lang from rowset : " + (after - before) / 1000.0 + " secs");

		logger.info("Result colID : " + colID);
		// if no collection ID is detected we have a problem and the
		// update must be cancelled
		if (colID == null || colID.equals("")) {
			logger.error("No collection ID given in ROWSET: " + rowset);
			return false;
		}

		try {
			addIndexType(rsIdxTypeID);
		} catch (Exception e) {
			logger.error("Add index type exception", e);
//			System.out.println("Add index type exception");
//			e.printStackTrace();
			return false;
		}
		// check if the same indexTypeID is specified
		if (rsIdxTypeID == null || !this.indexTypes.contains(rsIdxTypeID)) {
			logger.error("IndexType : " + rsIdxTypeID + " not in indexTypes : " + indexTypes);
			return false;
		}

		before = System.currentTimeMillis();
		String alteredRowset = RowsetParser.preprocessRowset(rowset, lang, colID, this.indexName, rsIdxTypeID, this.scope, this.cache);
		after = System.currentTimeMillis();
		logger.info("Time for preprocessRowset : " + (after - before) / 1000.0 + " secs");
		// logger.info("Result alteredRowset : " + alteredRowset);

		if (alteredRowset == null) {
			logger.error("could not preprocess rowset: " + rowset);
			return false;
		}
		before = System.currentTimeMillis();

		FullTextIndexType idxType = this.cache.cachedIndexTypes.get(QueryParser.createIndexTypekey(rsIdxTypeID, this.scope));
		logger.info("index type for name : " + idxType);
		logger.info("all indexTypes in cache : " + this.cache.cachedIndexTypes.keySet());

		ElasticSearchHelper.insertRowSet(bulkRequest, this.indexClient, this.indexName, idxType, this.indexTypes, alteredRowset);
		after = System.currentTimeMillis();
		logger.info("Time for insertRowSet : " + (after - before) / 1000.0 + " secs");

		colForIdTypes.put(colID + IndexType.SEPERATOR_FIELD_INFO + lang, idxType);

		return true;
	}

	/**
	 * Used to refresh the indexType and the presentable fields per index type.
	 * Usually called after invalidation
	 * 
	 * @param indexTypeStr
	 */
	private synchronized void bindIndexType(String indexTypeStr) {

		FullTextIndexType indexType = QueryParser.retrieveIndexType(indexTypeStr, this.scope, this.cache);

		logger.info("Index Type");
		logger.info("-----------------------------------------------");
		logger.info(indexType.toString());
		logger.info("-----------------------------------------------");
		List<String> presentables = new ArrayList<String>();
		List<String> searchables = new ArrayList<String>();

		for (IndexField idxTypeField : indexType.getFields()) {
			if (idxTypeField.returned /*&& !idxTypeField.name.equalsIgnoreCase(IndexType.COLLECTION_FIELD)*/)
				presentables.add(idxTypeField.name);
			if (idxTypeField.index)
				searchables.add(idxTypeField.name);
		}

		this.indexTypes.add(indexTypeStr);
		this.cache.presentableFieldsPerIndexType.put(indexTypeStr, presentables);
		this.cache.searchableFieldsPerIndexType.put(indexTypeStr, searchables);
	}

	/**
	 * Refreshes the index types
	 */
	public void refreshIndexTypesOfIndex() {
		ClusterStateResponse clusterResponse = this.indexClient.admin().cluster().prepareState().execute().actionGet();
		logger.info("clusterResponse : " + clusterResponse);
		IndexMetaData indexMetaData = clusterResponse.getState().getMetaData().index(indexName);
		if (indexMetaData != null) {
			ImmutableMap<String, MappingMetaData> mappings = indexMetaData.getMappings();

			logger.info("index types found in index : " + mappings.keySet());

			for (String indexType : mappings.keySet()) {
				logger.info("adding index type : " + indexType);
				try {
					this.bindIndexType(indexType);
					logger.info("adding index type : " + indexType + " succeded");
				} catch (Exception e) {
					logger.info("adding index type : " + indexType + " failed");
				}
			}

		}
	}

	@SuppressWarnings("unchecked")
	public boolean updateManagerProperties(HashMap<String, FullTextIndexType> colForIdTypes) {

		// Retrieving current collectionIds and fields
		Map<String, Object> result = null;
		String id = null;
		long version = 0;
		try {
			SearchResponse response = this.indexClient.prepareSearch(META_INDEX).setQuery(QueryBuilders.matchAllQuery()).execute().actionGet();
			for (SearchHit hit : response.getHits().getHits()) {
				result = hit.getSource();
				id = hit.getId();
				version = hit.getVersion();
			}

		} catch (Exception e) {
			logger.warn("No meta-index yet", e);
		}
		ArrayList<String> collectionIdsToBeAdded;
		ArrayList<String> fieldsToBeAdded;
		if (result != null) {
			collectionIdsToBeAdded = (ArrayList<String>) result.get("collectionIDs");
			fieldsToBeAdded = (ArrayList<String>) result.get("fields");
		} else {
			collectionIdsToBeAdded = new ArrayList<String>();
			fieldsToBeAdded = new ArrayList<String>();
		}

		Set<String> keys = colForIdTypes.keySet();
		for (String colIDandLang : keys) {
			FullTextIndexType index = colForIdTypes.get(colIDandLang);
			IndexField[] fields = index.getFields();
			// add to the Manager properties, the fields retrieved per
			// collection and language
			for (IndexField field : fields) {
				if (!RowsetParser.addToFieldInfo(fieldsToBeAdded, field.name, colIDandLang, index)) {
					return false;
				}
			}

			// add info for the special field allIndexes supported by FT, for
			// this lang and colID
			fieldsToBeAdded.add(colIDandLang + IndexType.SEPERATOR_FIELD_INFO + IndexType.SEARCHABLE_TAG + IndexType.SEPERATOR_FIELD_INFO + ALL_INDEXES);
			// removing lang from colIDandLang to add just the collection part to the collection ids
			collectionIdsToBeAdded.add(colIDandLang.split(":")[0]);

		}

		// remove duplicates
		HashSet<String> hs1 = new HashSet<String>();
		HashSet<String> hs2 = new HashSet<String>();
		hs1.addAll(collectionIdsToBeAdded);
		hs2.addAll(fieldsToBeAdded);
		collectionIdsToBeAdded.clear();
		fieldsToBeAdded.clear();
		collectionIdsToBeAdded.addAll(hs1);
		fieldsToBeAdded.addAll(hs2);

		try {
			Map<String, Object> document = new HashMap<String, Object>();
			document.put("collectionIDs", collectionIdsToBeAdded.toArray(new String[collectionIdsToBeAdded.size()]));
			document.put("fields", fieldsToBeAdded.toArray(new String[fieldsToBeAdded.size()]));
			
			if (result != null)
				version = version + 1;
			
			IndexResponse response = indexClient.prepareIndex(META_INDEX, META_INDEX, id)
		        .setSource(ElasticSearchHelper.createJSONObject(document).string())
		        .setConsistencyLevel(WriteConsistencyLevel.ONE)
		        .setVersion(version)
		        .execute()
		        .actionGet();

			logger.info("Add records to meta-index response id : " + response.getId());
			logger.info("Inserted colIDs and fields to " + META_INDEX);
			logger.info("committing meta-index");
			commitMeta();
			logger.info("Added records to meta-index");
		} catch (ElasticSearchException e) {
			logger.error("Caught an exception: ", e);
		} catch (IOException e) {
			logger.error("Caught an exception: ", e);
		}

		return true;
	}
	
	
	
	@SuppressWarnings("unchecked")
	public boolean updateManagerPropertiesRemoveCollID(String collectionID) {

		// Retrieving current collectionIds and fields
		Map<String, Object> result = null;
		String id = null;
		long version = 0;
		try {
			SearchResponse response = this.indexClient.prepareSearch(META_INDEX).setQuery(QueryBuilders.matchAllQuery()).execute().actionGet();
			for (SearchHit hit : response.getHits().getHits()) {
				result = hit.getSource();
				id = hit.getId();
				version = hit.getVersion();
			}

		} catch (Exception e) {
			logger.warn("No meta-index yet", e);
		}
		ArrayList<String> collectionIdsToBeAdded;
		ArrayList<String> fieldsToBeAdded;
		if (result != null) {
			collectionIdsToBeAdded = (ArrayList<String>) result.get("collectionIDs");
			fieldsToBeAdded = (ArrayList<String>) result.get("fields");
		} else {
			collectionIdsToBeAdded = new ArrayList<String>();
			fieldsToBeAdded = new ArrayList<String>();
		}
		
		
		if (collectionIdsToBeAdded.contains(collectionID)){
			collectionIdsToBeAdded.remove(collectionID);
		}
		
		List<String> fieldsTobeRemoved = new ArrayList<String>();
		for (String field : fieldsToBeAdded) {
			if (field.startsWith(collectionID + ":"))
				fieldsTobeRemoved.add(field);
		}
		
		fieldsToBeAdded.removeAll(fieldsTobeRemoved);


		// remove duplicates
		HashSet<String> hs1 = new HashSet<String>();
		HashSet<String> hs2 = new HashSet<String>();
		hs1.addAll(collectionIdsToBeAdded);
		hs2.addAll(fieldsToBeAdded);
		collectionIdsToBeAdded.clear();
		fieldsToBeAdded.clear();
		collectionIdsToBeAdded.addAll(hs1);
		fieldsToBeAdded.addAll(hs2);

		try {
			Map<String, Object> document = new HashMap<String, Object>();
			document.put("collectionIDs", collectionIdsToBeAdded.toArray(new String[collectionIdsToBeAdded.size()]));
			document.put("fields", fieldsToBeAdded.toArray(new String[fieldsToBeAdded.size()]));
			
			if (result != null)
				version = version + 1;
			
			IndexResponse response = indexClient.prepareIndex(META_INDEX, META_INDEX, id)
		        .setSource(ElasticSearchHelper.createJSONObject(document).string())
		        .setConsistencyLevel(WriteConsistencyLevel.ONE)
		        .setVersion(version)
		        .execute()
		        .actionGet();

			logger.info("Add records to meta-index response id : " + response.getId());
			logger.info("Inserted colIDs and fields to " + META_INDEX);
			logger.info("committing meta-index");
			commitMeta();
			logger.info("Added records to meta-index");
		} catch (ElasticSearchException e) {
			logger.error("Caught an exception: ", e);
		} catch (IOException e) {
			logger.error("Caught an exception: ", e);
		}

		return true;
	}
	
	

	public boolean addMetaIndex() {
		int counter = 0;
		Map<String, Object> result = null;
		try {
			Thread.sleep(2000);
		} catch (InterruptedException e) {
			logger.error("Failed to sleep", e);
		}
		// check if there is a meta-index
		while (true) {
			try {
				SearchResponse response = this.indexClient.prepareSearch(META_INDEX).setQuery(QueryBuilders.matchAllQuery()).execute().actionGet();
				for (SearchHit hit : response.getHits().getHits()) {
					result = hit.getSource();
				}
				break;
			} catch (IndexMissingException e) {
				logger.warn("Index missing, proceeding to creation");
				break;
			} catch (Exception e) {
				logger.warn("Not initialized yet, retrying");
				counter++;
				try {
					Thread.sleep(1000);
				} catch (InterruptedException e1) {
					logger.error("Failed to sleep", e);
				}
				if (counter == 10)
					return false;
			}
		}
		// if not add it
		if (result == null) {
			try {
				Map<String, Object> document = new HashMap<String, Object>();
				document.put("collectionIDs", new String[0]);
				document.put("fields", new String[0]);
				
				IndicesAdminClient iac = this.indexClient.admin().indices();
				logger.info("creating meta-index response");
				CreateIndexResponse cir = iac.prepareCreate(META_INDEX).execute().actionGet();
				logger.info("created meta-index response : " + cir.toString());
				
				IndexResponse response = indexClient.prepareIndex(META_INDEX, META_INDEX)
			        .setSource(ElasticSearchHelper.createJSONObject(document).string())
			        .setConsistencyLevel(WriteConsistencyLevel.ONE)
			        .execute()
			        .actionGet();

				logger.info("Add empty records to meta-index response id : " + response.getId());
				logger.info("committing meta-index");
				commitMeta();
				logger.info("Added empty meta-index");
			} catch (IOException e) {
				logger.error("Failed to add meta index", e);
				return false;
			}
		}
		return true;
	}

	public void invalidateCache() {
		this.cache.invalidate();
	}

	public boolean rebuildMetaIndex(String[] collectionIds, String[] fields) {
		DeleteMappingRequest deleteMapping = new DeleteMappingRequest(META_INDEX).type(META_INDEX);
		indexClient.admin().indices().deleteMapping(deleteMapping).actionGet();
		try {
			Map<String, Object> document = new HashMap<String, Object>();
			document.put("collectionIDs", collectionIds);
			document.put("fields", fields);
			IndexRequestBuilder indexRequest = indexClient.prepareIndex(META_INDEX, META_INDEX).setSource(
					ElasticSearchHelper.createJSONObject(document).string());
			indexRequest.setConsistencyLevel(WriteConsistencyLevel.ONE);
			indexRequest.execute();
			logger.info("Rebuilt meta-index");
		} catch (IOException e) {
			logger.error("Failed to rebuild meta index", e);
			return false;
		}
		return true;
	}

}
