package org.gcube.couchbase;

import gr.uoa.di.madgik.commons.server.ConnectionManagerConfig;
import gr.uoa.di.madgik.commons.server.PortRange;
import gr.uoa.di.madgik.commons.server.TCPConnectionManager;
import gr.uoa.di.madgik.grs.proxy.tcp.TCPConnectionHandler;
import gr.uoa.di.madgik.grs.proxy.tcp.TCPStoreConnectionHandler;
import gr.uoa.di.madgik.grs.reader.GRS2ReaderException;
import gr.uoa.di.madgik.rr.ResourceRegistry;
import gr.uoa.di.madgik.rr.ResourceRegistryException;

import java.io.IOException;
import java.io.Serializable;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.TimeUnit;

import org.gcube.couchbase.entities.MetaIndex;
import org.gcube.couchbase.helpers.CouchBaseDataTypesHelper;
import org.gcube.couchbase.helpers.CouchBaseDataTypesHelper.DataType;
import org.gcube.couchbase.helpers.CouchBaseHelper;
import org.gcube.couchbase.helpers.CouchBaseRestHelper;
import org.gcube.indexmanagement.common.ForwardIndexField;
import org.gcube.indexmanagement.common.ForwardIndexType;
import org.gcube.indexmanagement.resourceregistry.RRadaptor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.couchbase.client.CouchbaseClient;
import com.couchbase.client.CouchbaseConnectionFactoryBuilder;

/**
 * 
 * @author Alex Antoniadis
 * 
 */
public class CouchBaseNode implements Serializable {
	private static final long serialVersionUID = 1L;
	private static final Logger logger = LoggerFactory.getLogger(CouchBaseNode.class);

	private static final String designDocName = "forward_index_des_doc";
	private static Integer proxyPort = 11215;

	private String nodeIP = null;
	private String nodePort = null;
	private String nodeAddress = null; //ns_1@blablabla
	private String bucketName = "default_new";
	private String username = "Administrator";
	private String password = "mycouchbase";
	private Integer ramQuota = 512;
	private Integer replicaNumber = 1;
	
	private String hostname;

	private MetaIndex metaIndex = null;
	private Map<String, CouchBaseDataTypesHelper.DataType> keys = null;
	private String scope;
	private transient RRadaptor rradaptor;
	private transient CouchbaseClient client = null;

	public MetaIndex getMetaIndex() {
		return this.metaIndex;
	}

	public void setMetaIndex(MetaIndex metaIndex) {
		this.metaIndex = metaIndex;
	}

	public Map<String, CouchBaseDataTypesHelper.DataType> getKeys() {
		return this.keys;
	}

	public RRadaptor getRradaptor() {
		return this.rradaptor;
	}

	public String getDesignDocName() {
		return designDocName;
	}

	public String getBucketName() {
		return this.bucketName;
	}
	
	public String getScope() {
		return this.scope;
	}

	// Constructors

	private CouchBaseNode(String hostname) throws ResourceRegistryException, InterruptedException{
		this.hostname = hostname;
		logger.info("given hostname : " + hostname);
		logger.info("initializing grs2 and resource registry");
		this.initialize();
	}
	
	public CouchBaseNode(String hostname, String clusterName, Integer replicas, Integer ramQuota, String scope) throws ResourceRegistryException, InterruptedException{
		this(hostname);
		this.bucketName = clusterName;
		this.replicaNumber = replicas;
		this.ramQuota = ramQuota;
		this.scope = scope;
		this.metaIndex = new MetaIndex();
		this.keys = new HashMap<String, CouchBaseDataTypesHelper.DataType>();
		
		
		logger.info("bucketName    : " + bucketName);
		logger.info("replicaNumber : " + replicaNumber);
		logger.info("scope         : " + scope);
		logger.info("metaIndex     : new ");
		logger.info("keys          : " + this.keys);
	}
	
	
	public CouchBaseNode(String hostname, String nodeIP, String nodePort, String username, String password, String clusterName, Integer replicas, Integer ramQuota, String scope) throws ResourceRegistryException, InterruptedException {
		this(hostname, nodeIP, nodePort, clusterName, replicas, ramQuota, scope);
		this.username = username;
		this.password = password;
		
		logger.info("username : " + username);
		logger.info("password : " + password);
	}
	
	
	public CouchBaseNode(String hostname, String nodeIP, String nodePort, String clusterName, Integer replicas, Integer ramQuota, String scope) throws ResourceRegistryException, InterruptedException{
		this(hostname, clusterName, replicas, ramQuota, scope);
		this.nodeIP = nodeIP;
		this.nodePort = nodePort;
		
		logger.info("nodeIP   : " + nodeIP);
		logger.info("nodePort : " + nodePort);
	}
	
	// Constructors end

	/**
	 * Connects the client to a node in the nodes list
	 * 
	 * @param nodes
	 * @throws IOException
	 */
	private void initializeClient(List<URI> nodes) throws IOException {
		if (this.client == null) {
			logger.info("initializing client with nodes : " + nodes);
			
			
			// client = new CouchbaseClient(nodes, this.bucketName, "");
			CouchbaseConnectionFactoryBuilder cfb = new CouchbaseConnectionFactoryBuilder();
			cfb.setOpTimeout(45000);  // wait up to 45 seconds for an operation
										// to succeed
			cfb.setOpQueueMaxBlockTime(20000); // wait up to 20 seconds when
												// trying to enqueue an
												// operation

			this.client = new CouchbaseClient(cfb.buildCouchbaseConnection(nodes, this.bucketName, ""));
		} else {
			logger.warn("Client already initialized");
		}
	}

	/**
	 * Creates the bucket that is set a the parameter. The creation bucket
	 * request will be send to self so it is required to be part of the cluster
	 * or no cluster to exist at all.
	 * 
	 * @throws Exception
	 */
	private void createBucket() throws Exception {
		CouchBaseRestHelper.createBucket(this.nodeIP, this.nodePort, this.username, this.password,
				this.bucketName, this.ramQuota, this.replicaNumber, proxyPort);
	}
	
	/**
	 * Checks if the bucket that is defined for the client exists
	 * @return true if bucket exists, false otherwise
	 */
	public boolean checkIfBucketExists() {
		boolean res =  CouchBaseRestHelper.checkIfBucketExists(this.nodeIP, this.nodePort, this.username, this.password,
				this.bucketName);
		logger.info("bucket exists returned : " + res);
		
		return res;
	}

	/**
	 * Deletes the bucket that is set a the parameter. The creation bucket
	 * request will be sent to self so it is required to be part of the cluster.
	 * 
	 * @throws Exception
	 */
	private void deleteBucket() throws Exception {
//		CouchBaseRestHelper
//				.deleteBucket(this.nodeIP, this.nodePort, this.username, this.password, this.getBucketName());
		CouchBaseRestHelper
			.deleteBucketSDK(this.nodeIP, this.nodePort, this.username, this.password, this.bucketName);
		
		if (CouchBaseRestHelper.checkIfBucketExists(this.nodeIP, this.nodePort, this.username, this.password, this.bucketName)) {
			throw new Exception("Deletion failed. Check the previous exeption");
		} else {
			logger.info("Deletion succeded. Please ignore the previous exception");
		}
	}

	/**
	 * Creates a cluster by creating a node and adding a node
	 * 
	 * @param forceCreate
	 * @throws Exception
	 */
	// TODO: maybe wrong. Needs testing
	public void createCluster(boolean forceCreate) throws Exception {
		
		if (forceCreate || this.checkIfBucketExists() == false) {
			logger.info("bucket does not exist. will be created");
			try {
				this.createBucket();
			} catch (Exception ex1) {
				logger.info("Error creating new bucket... Probably exists. Trying to delete it and recreate it", ex1);
				this.deleteBucket();
				logger.info("Recreating the bucket..");
				logger.info("Should wait a bit after creating the bucket again");
				
				int tries = 10;
				Exception ex = null;
				while (tries>0) { 
						logger.info("Waiting a bit (4 sec)");
						proxyPort++;
						logger.info("Trying a different proxyPort : " + proxyPort);
						logger.info("tries left : " + tries);
						Thread.sleep(4000);
					try {
						this.createBucket();
						break;
					} catch (Exception e) {
						ex = e;
						tries--;
					}
				}
				if (tries == 0)
					throw new Exception("Couldn't create bucket", ex);
			}
		}
		//this.nodeAddress = CouchBaseRestHelper.addNode(this.nodeIP, this.nodePort, this.username, this.password,
		//		this.nodeIP, this.username, this.username);
		
		int tries = 5;
		while (tries>0) { 
				logger.info("Waiting a bit (1 sec)");
				logger.info("tries left : " + tries);
				Thread.sleep(1000);
			try {
				this.connectTo(this.nodeIP, this.nodePort);
				break;
			} catch (Exception e) {
				tries--;
			}
		}
		if (tries == 0)
			throw new Exception("Couldn't connect to (self) cluster");
	}

	/**
	 * Initializes a client for the server located at <i>host:port</i>
	 * 
	 * @param host
	 * @param port
	 * @throws IOException
	 */
	public void connectTo(String host, String port) throws IOException {
		ArrayList<URI> nodes = new ArrayList<URI>();
		nodes.add(URI.create("http://" + host + ":" + port + "/pools"));
		this.initializeClient(nodes);
	}

	/**
	 * Joins a cluster of a known server
	 * 
	 * @param knownHostIP
	 * @param knownHostPort
	 * @throws Exception
	 */
	public void joinCluster(String knownHostIP, String knownHostPort) throws Exception {
		Map<String, String> clusterNodes = CouchBaseRestHelper.getClusterNodesAddressesAndPorts(knownHostIP, knownHostPort, this.username, this.password);

		
		// check if already added
		String port = clusterNodes.get(this.nodeIP);
		if (port != null && port.equalsIgnoreCase(this.nodePort)) {
			logger.info("Node already in cluster");
		} else {
			logger.info("Node not in cluster");
			logger.info("Adding node...");
			this.nodeAddress = CouchBaseRestHelper.addNode(knownHostIP, knownHostPort, this.username, this.password,
					this.nodeIP, this.username, this.password);
			logger.info("Rebalancing cluster...");
			CouchBaseRestHelper.rebalanceCluster(knownHostIP, knownHostPort, this.username, this.password);
		}

		// connect to self
		this.connectTo(this.nodeIP, this.nodePort);

		/*
		 * ArrayList<URI> nodes = new ArrayList<URI>(); for (Entry<String,
		 * String> clusterNode : clusterNodes.entrySet()) {
		 * nodes.add(URI.create("http://" + clusterNode.getKey() + ":" +
		 * clusterNode.getValue() + "/pools")); }
		 * 
		 * this.initializeClient(nodes);
		 */
	}
	
	public void joinCluster(Map<String, String> knownHosts) throws Exception {
		for (Entry<String, String> knownHost : knownHosts.entrySet()) {
			try {
				logger.info("Trying to connect to : " + knownHost.getKey() + " port : " + knownHost.getValue());
				this.joinCluster(knownHost.getKey(), knownHost.getValue());
				break;
			} catch (Exception e) {
				logger.error("Connecting to : "  + knownHost.getKey(), knownHost.getValue() + " failed!", e);
			}
		}
		
		
	}

	/**
	 * Removes the node from the cluster
	 * 
	 * @throws Exception
	 */
	private void leaveCluster() throws Exception {
		if (this.nodeAddress == null) {
			logger.info("nodeAddress has not been set yet");
			this.nodeAddress = "ns_1@" + this.nodeIP;
		}
		
		List<String> clusterNodes = CouchBaseRestHelper.getClusterNodes(this.nodeIP, this.nodePort, this.username, this.password);

		if (clusterNodes.size() == 1) {
			logger.info("This is the last node of the cluster. Cannot be removed");
			return;
		}
		
		CouchBaseRestHelper.removeNode(this.nodeIP, this.nodePort, this.username, this.password, this.nodeAddress);
	}

	/**
	 * Shuts down the node and leaves the cluster
	 * 
	 * @throws Exception
	 */
	public void shutdown() throws Exception {
		this.client.shutdown();
		this.leaveCluster();
	}

	/**
	 * Deletes the bucket and then shutdowns
	 * 
	 * @throws Exception
	 */
	public void destroy() throws Exception {
		// deletes the index
		this.deleteBucket();
		this.shutdown();
	}
	
	/**
	 * Deletes the bucket
	 * 
	 * @throws Exception
	 */
	public void delete() throws Exception {
		// deletes the index
		this.deleteBucket();
	}

	/**
	 * Recreates the bucket
	 * 
	 * @throws Exception
	 */
	public void clear() throws Exception {
		// delete and recreate bucket

		logger.info("Deleting the bucket..");
		this.deleteBucket();
		// since the bucket is down the client is down

		
		logger.info("Recreating the bucket..");
		logger.info("Should wait a bit after creating the bucket again");
		
		int tries = 10;
		Exception ex = null;
		while (tries>0) { 
				logger.info("Waiting a bit (3 sec)");
				logger.info("tries left : " + tries);
				Thread.sleep(3000);
			try {
				this.createBucket();
				break;
			} catch (Exception e) {
				ex = e;
				tries--;
			}
		}
		if (tries == 0)
			throw new Exception("Couldn't create bucket", ex);
		
		// reconnect
		
		// logger.info("Shutting down the client..");
		// this.client.shutdown();
		// logger.info("Reconnecting to the new created bucket..");
		// this.connectTo(this.nodeIP, this.nodePort);
		this.metaIndex = new MetaIndex();
		this.commitMetaIndexToDatabase();
	}

	/**
	 * Loads the keys from the metaIndex
	 */
	public void loadKeysFromMetaIndex() {
		
		for (Entry<String, String> kv : this.metaIndex.getIndexKeys().entrySet()) {
			String fieldName = kv.getKey();// "fwd_string_string";
			String indexTypeID = kv.getValue();// "fwd_string_string";
			
			ForwardIndexType indexType = new ForwardIndexType(indexTypeID, this.scope);
			
			ForwardIndexField keyField = indexType.getKeyField();
			
			if (keyField == null) {
				logger.error("No keyField found for indexTypeID : " + indexTypeID + " in scope : " + this.scope);
				logger.error("fieldName : " + fieldName);
				logger.error("Probably wrong indexTypeID given");
				
				continue;
			}
			
			logger.info("keyfield : " + keyField.getName() + " , datatype : " + keyField.getDataType());

			// ForwardIndexField valueField = indexType.getValueField();
			// valueField is ignored since it is always string...
			DataType datatype = CouchBaseDataTypesHelper.getDataType(keyField.getDataType());

			this.keys.put(fieldName, datatype);
		}
		
		logger.info("Loaded Keys : " + this.getKeys());
	}
	
	/**
	 * Given at map of <field_name, indexTypeID> elements it creates the keys of
	 * that hold the DataType of each field and then it creates the indexes for
	 * each field
	 * 
	 * @param keysValues
	 */
	public void setIndexType(Map<String, String> keysValues) {
		this.keys.clear();
		this.addIndexTypes(keysValues);
	}
	
	/**
	 * Given at map of <field_name, indexTypeID> elements it adds the keys of
	 * that hold the DataType of each field and then it creates the indexes for
	 * each field
	 * 
	 * @param keysValues
	 */
	public void addIndexTypes(Map<String, String> keysValues) {
		if (keysValues == null || keysValues.size() == 0) {
			logger.warn("No keysValues are given to build index");
			logger.warn("Skipping the index creation");
			return;
		}
		
		for (Entry<String, String> kv : keysValues.entrySet()) {

			String fieldName = kv.getKey();// "fwd_string_string";
			String indexTypeID = kv.getValue();// "fwd_string_string";


			// no need to cache anything since it is an operation that will be
			// executed once per field
			ForwardIndexType indexType = new ForwardIndexType(indexTypeID, this.scope);
			
			ForwardIndexField keyField = indexType.getKeyField();
			
			if (keyField == null) {
				logger.error("No keyField found for indexTypeID : " + indexTypeID + " in scope : " + this.scope);
				logger.error("fieldName : " + fieldName);
				logger.error("Probably wrong indexTypeID given");
				
				continue;
			}
			
			logger.info("keyfield : " + keyField.getName() + " , datatype : " + keyField.getDataType());

			// ForwardIndexField valueField = indexType.getValueField();
			// valueField is ignored since it is always string...
			DataType datatype = CouchBaseDataTypesHelper.getDataType(keyField.getDataType());

			this.keys.put(fieldName, datatype);
			

		}
		this.metaIndex.getIndexKeys().putAll(keysValues);
		
		logger.info("Keys : " + this.getKeys());

		CouchBaseHelper.createIndexes(this.client, this.getBucketName(), this.getDesignDocName(), this.getKeys());
	}

	/**
	 * Executes the queryString and returns the answers at a specific locator.
	 * The locator will be returned before all the answers are produced.
	 * 
	 * @param queryString
	 * @return gRS2 locator of the answers
	 */
	public String query(String queryString) {
		return CouchBaseHelper.query(this.client, this.metaIndex, this.bucketName, designDocName, this.keys,
				this.rradaptor, queryString);
	}
	
	/**
	 * Returns the documents count for a given collection
	 * @param collectionID
	 * @return the number of the documents in the collection
	 */
	public Long getCollectionDocuments(String collectionID) {
		Long count = CouchBaseHelper.countCollectionCouchBase(this.client, this.bucketName, designDocName, this.keys, this.metaIndex, collectionID);
		logger.info("collection : " + collectionID + " count : " + count);
		return count;
	}
	
	
	public Set<String> getIndicesOfCollection(String collection) {
		Set<String> indicesOfCollection = CouchBaseHelper.indicesOfCollection(this.client, this.bucketName, collection);
		
		logger.info("indicesOfCollection : " + collection + " count : " + indicesOfCollection);
		
		return indicesOfCollection;
	}
	
	public Set<String> getCollectionsOfIndex(String indexName) {
		Set<String> collectionsOfIndex = CouchBaseHelper.getAllCollectionsOfIndex(this.client, indexName, designDocName, this.keys);
		
		logger.info("collectionsOfIndex : " + indexName + " count : " + collectionsOfIndex);
		
		return collectionsOfIndex;
	}

	/**
	 * Feeds the index with values taken from a specific locator. The locator
	 * produces gRS2 GenericRecords that have 1 field which is the actual Rowset
	 * that will be parsed. See ForwardIndexDocument for the parsing and the
	 * format that is expected.
	 * 
	 * @param locator
	 * @return true if feeding succeed, false otherwise
	 * @throws GRS2ReaderException
	 * @throws URISyntaxException
	 */
	public boolean feedLocator(String locator) throws GRS2ReaderException, URISyntaxException {
		boolean feedResult = CouchBaseHelper.feedLocator(this.client, locator, this.getMetaIndex());

		logger.info("feedResult : " + feedResult);
		if (feedResult)
			this.commit();

		return feedResult;
	}

	/**
	 * Deletes all the documents with keys given in docIDs param
	 * 
	 * @param docIDs
	 */
	public void deleteDocuments(List<String> docIDs) {
		CouchBaseHelper.deleteDocsCouchBase(this.client, docIDs);
	}
	
	/**
	 * Deletes all the documents of the given collection
	 * 
	 * @param collectionID
	 */
	public Boolean deleteCollection(String collectionID) {
		return CouchBaseHelper.deleteCollectionCouchBase(this.client, this.bucketName, designDocName, this.keys, this.metaIndex, collectionID);
	}
	
	/**
	 * Count all the documents of the given collection
	 * 
	 * @param collectionID
	 */
	public Long countCollection(String collectionID) {
		return CouchBaseHelper.countCollectionCouchBase(this.client, this.bucketName, designDocName, this.keys, this.metaIndex, collectionID);
	}

	/**
	 * Executes a dummyQuery on each index to "warm" it. This warming will
	 * update the indexes so any update in the database will be reflected to the
	 * indexes and the queries will be consistent.
	 */
	private void commit() {
		CouchBaseHelper.commit(this.client, this.getBucketName(), this.getDesignDocName(), this.getKeys());
	}

	/**
	 * Loads the meta index from the database and discards current changes
	 */
	public void loadMetaFromDatabase() {
		this.metaIndex.loadFromDatabase(this.client);
	}
	
	/**
	 * Saves the current state of meta index in the database and overrides the previous contents
	 */
	public void commitMetaIndexToDatabase() {
		this.metaIndex.saveToDatabase(this.client);
	}
	
	/**
	 * Initializes the gRS2 connection manager and starts the ResourceRegistry
	 * bridging.
	 */
	private void initialize() throws ResourceRegistryException, InterruptedException {
		TCPConnectionManager.Init(new ConnectionManagerConfig(this.hostname,
				new ArrayList<PortRange>(), true));
		TCPConnectionManager.RegisterEntry(new TCPConnectionHandler());
		TCPConnectionManager.RegisterEntry(new TCPStoreConnectionHandler());
		logger.info("Initializing ResourceRegistry");
		try {
			ResourceRegistry.startBridging();
			TimeUnit.SECONDS.sleep(1);
			while (!ResourceRegistry.isInitialBridgingComplete()) {
				logger.info("registry not ready...sleeping");
				TimeUnit.SECONDS.sleep(10);
			}
		} catch (ResourceRegistryException e) {
			logger.error("Resource Registry could not be initialized", e);
			throw e;
		} catch (InterruptedException e) {
			logger.error("Resource Registry could not be initialized", e);
			throw e;
		}
		
		this.rradaptor = new RRadaptor();
		logger.info("Initializing ResourceRegistry is DONE");
	}

	/*public static void main(String[] args) throws Exception {
		ArrayList<URI> nodes = new ArrayList<URI>();

		// Add one or more nodes of your cluster (exchange the IP with yours)
		nodes.add(URI.create("http://192.168.56.103:8091/pools"));
		// nodes.add(URI.create("http://192.168.56.103:8091/pools"));

		// Try to connect to the client

		CouchBaseNode cbn = new CouchBaseNode();

		cbn.scope = "/gcube/devsec";
		cbn.bucketName = "couchbase-cluster-service-_gcube_devsec";

		cbn.createCluster(false);
		
		
//		cbn.createBucket();
//		Thread.sleep(10000);
//		cbn.initializeClient(nodes);
		cbn.setMetaIndex(new MetaIndex());
//		cbn.getMetaIndex().getPresentables().add("title");
//		cbn.getMetaIndex().getSearchables().add("title");
//
//		cbn.getMetaIndex().getPresentables().add("gDocCollectionID");
//		cbn.getMetaIndex().getSearchables().add("gDocCollectionID");

		// cbn.clear();

//		cbn.initialize();
//		cbn.keys = new HashMap<String, CouchBaseDataTypesHelper.DataType>();
//
////		cbn.getKeys().put("gDocCollectionID", DataType.STRING);
//
//		Map<String, String> kv = new HashMap<String, String>();
//		kv.put("gDocCollectionID", "fwd_string_string");
//		// kv.put("phylum", "fwd_string_string");
//		//
//		cbn.setIndexType(kv);
//		//
////		 String locator = "grs2-proxy://jazzman.di.uoa.gr:36228?key=c28b1578-0f73-4ffd-b04f-7fe84faaa8fe#TCP"; 
////		 cbn.feedLocator(locator);
//		//
//		String queryString = "((9be7e3d8-4d01-4dd0-a969-18f55edabebc == \"*\") and (((gDocCollectionID == 5b268db0-9d63-11de-8d8f-a04a2d1ca936) or (((gDocCollectionID == 18611ed0-f162-11dd-96f7-b87cc0f0b075) or (((gDocCollectionID == cba3b770-f1e2-11dd-836f-ed8453cc0b6c) or (gDocCollectionID == 9ac21cf0-f160-11dd-96f7-b87cc0f0b075)))))))) project 5efef600-1ee1-4ef2-8dc3-10f3faa96a5f";
//		//
//		cbn.query(queryString);

		Thread.sleep(2000);

//		cbn.shutdown();
	}*/

}
