package org.gcube.couchbase.entities;

import java.io.Serializable;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import net.spy.memcached.internal.OperationFuture;

import org.gcube.indexmanagement.common.IndexType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.couchbase.client.CouchbaseClient;
import com.google.gson.Gson;

/**
 * 
 * @author Alex Antoniadis
 * 
 */
public class MetaIndex implements Serializable {
	
	private static final long serialVersionUID = 1L;
	private static final Logger logger = LoggerFactory.getLogger(MetaIndex.class);
	private static Gson gson = new Gson();

	private static String META_INDEX_KEY = "METAINDEX";

	private Set<String> collections;
	private Set<String> fields;
	private Set<String> searchables;
	private Set<String> presentables;
	private Map<String, String> indexKeys;// it is actually keys but this name messes with the indexes (views) so we changed that to indexKeys

	public Set<String> getSearchables() {
		return this.searchables;
	}

	public Set<String> getPresentables() {
		return this.presentables;
	}
	
	public Set<String> getCollections() {
		return this.collections;
	}
	
	public Set<String> getFields() {
		return this.fields;
	}
	
	public Map<String, String> getIndexKeys() {
		return this.indexKeys;
	}

	public MetaIndex() {
		this.collections = new HashSet<String>();
		this.fields = new HashSet<String>();
		this.searchables = new HashSet<String>();
		this.presentables = new HashSet<String>();
		this.indexKeys = new HashMap<String, String>();
	}

	@Override
	public String toString() {
		return "MetaIndex [collections=" + this.collections + ", fields=" + fields + ", searchables=" + this.searchables
				+ ", presentables=" + this.presentables + ", indexKeys=" + this.indexKeys + "]";
	}

	/**
	 * Saves the metaindex in the database. If database has already a metaindex
	 * then union of local and remote will be saved.
	 * 
	 * @param client
	 * @return true if save succeeded, false if failed
	 */
	public Boolean saveToDatabase(CouchbaseClient client) {
		return saveMetaIndexToDatabase(client, this);
	}
	
	/**
	 * Like save but does not take into account the metaindex from the database
	 * 
	 * @param client
	 * @return true if save succeeded, false if failed
	 */
	public Boolean flushToDatabase(CouchbaseClient client) {
		return saveMetaIndexToDatabase(client, this, true);
	}

	public void updateFromDoc(ForwardIndexDocument doc) {
		updateMetaFromDoc(this, doc);
	}

	public void loadFromDatabase(CouchbaseClient client) {
		MetaIndex meta = loadMetaFromDatabase(client);
		this.collections = new HashSet<String>(meta.collections);
		this.searchables = new HashSet<String>(meta.searchables);
		this.presentables = new HashSet<String>(meta.presentables);
		this.fields = new HashSet<String>(meta.fields);
		this.indexKeys = new HashMap<String, String>(meta.indexKeys);
	}
	
	public void removeCollection(String collectionID) {
		this.collections.remove(collectionID);
		
		Set<String> deleteFields = new HashSet<String>();
		for (String field : this.fields)
			if (field.startsWith(collectionID + ":"))
				deleteFields.add(field);
		
		this.fields.removeAll(deleteFields);
		logger.info("MetaIndex deleted collection : " + collectionID);
		logger.info("MetaIndex deleted fields     : " + deleteFields);
	}

	// Static methods
	public static Boolean saveMetaIndexToDatabase(CouchbaseClient client, MetaIndex meta, Boolean override) {
		return saveMetaIndexToDatabase(client, meta.collections, meta.fields, meta.searchables,
				meta.presentables, meta.indexKeys, override);
	}

	public static Boolean saveMetaIndexToDatabase(CouchbaseClient client, MetaIndex meta) {
		return saveMetaIndexToDatabase(client, meta.collections, meta.fields, meta.searchables,
				meta.presentables, meta.indexKeys, false);
	}

	private static Boolean saveMetaIndexToDatabase(CouchbaseClient client, Set<String> collections, Set<String> fields,
			Set<String> searchables, Set<String> presentables, Map<String, String> indexKeys, Boolean override) {
		logger.info("The following will be added : ");
		logger.info("collections  : " + collections);
		logger.info("fields       : " + fields);
		logger.info("searchables  : " + searchables);
		logger.info("presentables : " + presentables);
		logger.info("index keys   : " + indexKeys);
		logger.info("override     : " + override);

		Object metaIndexValueObj = null;
		MetaIndex meta = null;
		
		if (override) {
			meta = new MetaIndex();
		}
		else {
			try {
				metaIndexValueObj = client.get(META_INDEX_KEY);
				if (metaIndexValueObj != null) { // it is not in the index
					String json = (String) metaIndexValueObj;
					meta = gson.fromJson(json, MetaIndex.class);
				} else {
					meta = new MetaIndex();
				}
			} catch (Exception e) { //exception while trying to get it so we recreate it
				meta = new MetaIndex();
			}
		}

		meta.collections.addAll(collections);
		meta.fields.addAll(fields);
		meta.searchables.addAll(searchables);
		meta.presentables.addAll(presentables);
		meta.indexKeys.putAll(indexKeys);

		String updatedValue = gson.toJson(meta);
		
		//try some times to write it
		
		OperationFuture<Boolean> res = null;
		
		int tries = 12;
		Exception ex = null;
		
		while (tries>0) { 
			logger.info("meta index to be stored    : " + updatedValue);
			res = client.set(META_INDEX_KEY, 0, updatedValue);
	
			try {
				Boolean result = res.get().booleanValue();
				if (result)
					logger.info("Meta Index successfully updated");
				else
					logger.info("Meta Index NOT successfully updated");
	
				return result;
			} catch (Exception e) {
				ex = e;
				tries--;
			}
			logger.info("Waiting a bit (3 sec)");
			logger.info("tries left : " + tries);
			try {
				Thread.sleep(3000);
			} catch (InterruptedException e) {
				logger.error("Error while sleeping: ", e);
			}
		}	
		
		if (tries == 0) {
			logger.error("Error while updating the meta index: ", ex);
		}
		return false;
	}

	public static MetaIndex loadMetaFromDatabase(CouchbaseClient client) {
		Object metaIndexValueObj = null;
		MetaIndex meta = null;

		try {
			int tries = 5;
			while (tries-- > 0) {
				try {
					metaIndexValueObj = client.get(META_INDEX_KEY);
					break;
				} catch (Exception e) {
					logger.warn("Could not load metaIndex. Retrying in 2 secs");
					Thread.sleep(2000);
				}
			}
			
			if (metaIndexValueObj != null) {
				String json = (String) metaIndexValueObj;
				meta = gson.fromJson(json, MetaIndex.class);
				logger.info("Meta index loaded");
				logger.info(meta.toString());
	
			} else {
				meta = new MetaIndex();
				logger.info("New Meta index created");
			}
		}catch (Exception e) {
			logger.info("Meta index could not be loaded. Creating new");
			meta = new MetaIndex();
			logger.info("New Meta index created");
		}
		return meta;
	}

	public static void updateMetaFromDoc(MetaIndex meta, ForwardIndexDocument doc) {
		String colId = doc.getColId();
		meta.collections.add(colId);

		meta.searchables.addAll(doc.getKeys().keySet());
		meta.presentables.addAll(doc.getFields().keySet());

		String fieldPrefix = colId + IndexType.SEPERATOR_FIELD_INFO + doc.getDocLang();

		for (String searchable : doc.getKeys().keySet())
			meta.fields.add(fieldPrefix + IndexType.SEPERATOR_FIELD_INFO + IndexType.SEARCHABLE_TAG
					+ IndexType.SEPERATOR_FIELD_INFO + searchable);

		for (String presentable : doc.getFields().keySet())
			meta.fields.add(fieldPrefix + IndexType.SEPERATOR_FIELD_INFO + IndexType.PRESENTABLE_TAG
					+ IndexType.SEPERATOR_FIELD_INFO + presentable);
	}
	
	
}
