package org.gcube.service.idm;

import org.gcube.common.security.secrets.Secret;
import org.gcube.idm.common.is.InfrastrctureServiceClient;
import org.gcube.idm.common.is.IsServerConfig;
import org.gcube.smartgears.ContextProvider;
import org.gcube.smartgears.configuration.container.ContainerConfiguration;
import org.gcube.smartgears.context.application.ApplicationContext;
import org.gcube.smartgears.context.container.ContainerContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import jakarta.ws.rs.InternalServerErrorException;

public abstract class AbstractClientFactory {

    // // Service endpoint properties
    // protected final String RUNTIME_RESOURCE_NAME = null;
    // protected final String CATEGORY = null;
    // protected final String END_POINT_NAME = null;
    // protected final boolean IS_ROOT_SERVICE = true;

    private static final Logger logger = LoggerFactory.getLogger(AbstractClientFactory.class);

    /**
     * server configuration
     * can be obtained from IS (fetchIsConfig) or created externally
     * using the singleton pattern, it's retrieved from IS only for the first
     * access, then kept in the singleton object
     */
    protected IsServerConfig config;
    protected Secret secret;

    public abstract String getRuntimeResourceName();

    public abstract String getCategory();

    public abstract String getEndPointName();

    public abstract boolean isRootService();

    // IS Secret
    public Secret getSecret() {
        return secret;
    }

    public void setSecret(Secret secret) {
        this.secret = secret;
    }

    /**
     * service configuration obtained from IS in the private constructor
     * using the singleton pattern, it's retrieved from IS only for the first
     * access, then kept in the singleton object
     */
    // set a custom config for the factory, skipping or overriding the fetch from IS

    public void setConfig(IsServerConfig configuration) {
        this.config = configuration;
    }

    public IsServerConfig fetchIsConfig() throws InternalServerErrorException {
        if (this.getSecret() == null) {
            this.setSecret(getSecretForInfrastructure());
        }
        return fetchIsConfig(this.getSecret());
    }

    public IsServerConfig fetchIsConfig(Secret secret) throws InternalServerErrorException {
        if (secret == null) {
            throw new InternalServerErrorException("null secret for fetchIsConfig");
        }
        try {
            IsServerConfig cfg = InfrastrctureServiceClient.serviceConfigFromIS(getRuntimeResourceName(), getCategory(),
                    getEndPointName(), isRootService(), secret);
            logger.info("KeycloakAPICredentials object built {} - {}", cfg.getServerUrl(), cfg.getName());

            return cfg;
        } catch (Exception e) {
            e.printStackTrace();
            throw new InternalServerErrorException(e);
        }
    }

    /**
     * Retrieve the secret from smartgear configuration
     * 
     * @return Secret
     */
    public static Secret getSecretForInfrastructure() {
        ApplicationContext ctx = ContextProvider.get();
        ContainerContext container = ctx.container();
        ContainerConfiguration configuration = container.configuration();
        String infra_context = configuration.infrastructure();
        Secret secret = ctx.container().authorizationProvider().getSecretForContext(infra_context);
        return secret;
    }

}
