package gr.uoa.di.madgik.grs.bridge;

import gr.uoa.di.madgik.grs.bridge.exceptions.GCubeBridgeException;
import gr.uoa.di.madgik.grs.buffer.IBuffer;

import java.net.URI;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.gcube.common.searchservice.searchlibrary.resultset.elements.ResultElementBase;
import org.gcube.common.searchservice.searchlibrary.rsclient.elements.RSLocator;

/**
 * Base abstract class that implements the downgrade procedure from a new gRS2 {@link IBuffer} to 
 * an "old style" RewsultSet. The class defines hooks that need to be implemented by extenders that will 
 * handle with the specific type of ResultSet type
 * 
 * @author gpapanikos
 *
 */
public abstract class BridgeDowngrade extends Thread
{
	/**
	 * The locator of the gRS2 {@link IBuffer} input that needs to be downgraded
	 */
	protected URI locator=null;
	/**
	 * The type of records that to be produced in the ResultSet output
	 */
	protected Class<? extends ResultElementBase> recordClass=null;

	/**
	 * Perform the downgrade. After all the configuration parameters are set, this is the only method the client
	 * needs to invoke. The operation takes pace in the background
	 * 
	 * @return The RS locator of the created ResultSet
	 * @throws GCubeBridgeException The downgrade could not be completed
	 */
	public RSLocator downgrade() throws GCubeBridgeException
	{
		RSLocator locator=null;
		this.initialize();
		locator=this.getLocator();
		this.setName("gRS2 bridge downgrade");
		this.setDaemon(true);
		this.start();
		return locator;
	}
	
	/**
	 * Set the URI locator of the gRS2 input {@link IBuffer} to downgrade
	 * 
	 * @param locator the locator
	 */
	public void setLocator(URI locator)
	{
		this.locator=locator;
	}
	
	/**
	 * Sets the type of records that to be produced in the ResultSet output
	 * 
	 * @param recordClass the record type to produce
	 */
	public void setRecordClass(Class<? extends ResultElementBase> recordClass)
	{
		this.recordClass=recordClass;
	}
	
	/**
	 * Instantiates the appropriate output record and populates it through the provided record
	 * 
	 * @param record the record to downgrade
	 * @return the new downgraded record
	 * @throws GCubeBridgeException the record downgrade could not be completed
	 */
	protected ResultElementBase getRecord(GCubeRecord record) throws GCubeBridgeException
	{
		ResultElementBase rec;
		try
		{
			rec = this.recordClass.newInstance();
			rec.RS_fromXML(record.downgrade());
			return rec;
		} catch (Exception e)
		{
			throw new GCubeBridgeException("Could not retrieve the downgraded version of the record", e);
		}
	}
	
	/**
	 * Iterate over the input, produce the output and store it in the output writer
	 * 
	 * @throws GCubeBridgeException the downgrade could not be completed
	 */
	public abstract void doDowngrade() throws GCubeBridgeException;
	
	/**
	 * Perform any initialization needed before the downgrade begins
	 * 
	 * @throws GCubeBridgeException the initialization could not be completed
	 */
	public abstract void initialize() throws GCubeBridgeException;
	
	/**
	 * Retrieves the locator of the create ResultSet
	 * 
	 * @return the RSlocator
	 * @throws GCubeBridgeException the locator could not be retrieved
	 */
	public abstract RSLocator getLocator() throws GCubeBridgeException;
	
	/**
	 * The logger defined by the extending class to use the same logger
	 * 
	 * @return the logger
	 */
	public abstract Logger getLogger();
	
	public void run()
	{
		try
		{
			this.doDowngrade();
		}catch(Exception ex)
		{
			if(this.getLogger().isLoggable(Level.WARNING)) this.getLogger().log(Level.WARNING,"Could not complete the downgrade procedure",ex);
		}
	}
}
