/*
 * Ext GWT 2.2.5 - Ext for GWT
 * Copyright(c) 2007-2010, Ext JS, LLC.
 * licensing@extjs.com
 * 
 * http://extjs.com/license
 */
package com.extjs.gxt.ui.client.widget.treegrid;

import com.extjs.gxt.ui.client.data.ModelData;
import com.extjs.gxt.ui.client.event.BaseEvent;
import com.extjs.gxt.ui.client.event.GridEvent;
import com.extjs.gxt.ui.client.store.TreeStore;
import com.extjs.gxt.ui.client.widget.grid.CellEditor;
import com.extjs.gxt.ui.client.widget.grid.ColumnModel;
import com.extjs.gxt.ui.client.widget.grid.EditorGrid.ClicksToEdit;
import com.google.gwt.user.client.Element;

/**
 * Adds editing capabilities to TreeGrid.
 * 
 * <dl>
 * <dt><b>Events:</b></dt>
 * 
 * <dd><b>BeforeEdit</b> : GridEvent(grid, record, property, value, rowIndex,
 * colIndex)<br>
 * <div>Fires before cell editing is triggered. Listeners can cancel the action
 * by calling {@link BaseEvent#setCancelled(boolean)}.</div>
 * <ul>
 * <li>grid : this</li>
 * <li>record : the record being edited</li>
 * <li>property : the property being edited</li>
 * <li>value : the value being edited</li>
 * <li>rowIndex : the current row</li>
 * <li>colIndex : the current column</li>
 * </ul>
 * </dd>
 * 
 * <dd><b>AfterEdit</b> : GridEvent(grid, record, property, value, startValue,
 * rowIndex, colIndex)<br>
 * <div>Fires after a cell is edited.</div>
 * <ul>
 * <li>grid : this</li>
 * <li>record : the record being edited</li>
 * <li>property : the property being edited</li>
 * <li>value : the value being set</li>
 * <li>startValue : the value before the edit</li>
 * <li>rowIndex : the current row</li>
 * <li>colIndex : the current column</li>
 * </ul>
 * </dd>
 * 
 * <dd><b>ValidateEdit</b> : GridEvent(grid, record, property, value,
 * startValue, rowIndex, colIndex)<br>
 * <div>Fires right before the record is updated. Listeners can cancel the
 * action by calling {@link BaseEvent#setCancelled(boolean)}.</div>
 * <ul>
 * <li>grid : this</li>
 * <li>record : the record being edited</li>
 * <li>property : the property being edited</li>
 * <li>value : the value being set</li>
 * <li>startValue : the value before the edit</li>
 * <li>rowIndex : the current row</li>
 * <li>colIndex : the current column</li>
 * </ul>
 * </dd>
 * 
 * </dl>
 */
public class EditorTreeGrid<M extends ModelData> extends TreeGrid<M> {

  @SuppressWarnings("rawtypes")
  public EditorTreeGrid(TreeStore store, ColumnModel cm) {
    super(store, cm);
    setSelectionModel(new CellTreeGridSelectionModel<M>());
    setTrackMouseOver(false);

    editSupport = getEditSupport();
    editSupport.bind(this);
  }

  /**
   * Returns the active editor.
   * 
   * @return the active editor
   */
  public CellEditor getActiveEditor() {
    return editSupport.getActiveEditor();
  }

  /**
   * Returns the clicks to edit.
   * 
   * @return the clicks to edit
   */
  public ClicksToEdit getClicksToEdit() {
    return editSupport.getClicksToEdit();
  }

  /**
   * Returns true if editing is active.
   * 
   * @return the editing state
   */
  public boolean isEditing() {
    return editSupport.isEditing();
  }

  @Override
  public void reconfigure(TreeStore<M> store, ColumnModel cm) {
    super.reconfigure(store, cm);
    editSupport.bind(this);
  }

  /**
   * Sets the number of clicks to edit (defaults to ONE).
   * 
   * @param clicksToEdit the clicks to edit
   */
  public void setClicksToEdit(ClicksToEdit clicksToEdit) {
    editSupport.setClicksToEdit(clicksToEdit);
  }

  /**
   * Starts editing the specified for the specified row/column.
   * 
   * @param row the row index
   * @param col the column index
   */
  public void startEditing(final int row, final int col) {
    editSupport.startEditing(row, col);
  }

  /**
   * Stops any active editing.
   */
  public void stopEditing() {
    editSupport.stopEditing();
  }

  /**
   * Stops any active editing.
   * 
   * @param cancel true to cancel, false to complete
   */
  public void stopEditing(boolean cancel) {
    editSupport.stopEditing(cancel);
  }

  @Override
  protected void onDoubleClick(GridEvent<M> e) {
    if (editSupport.onDoubleClick(e)) {
      return;
    }
    super.onDoubleClick(e);
  }

  @Override
  protected void onRender(Element target, int index) {
    super.onRender(target, index);
    editSupport.doRender();
  }
}
