/*
 * Decompiled with CFR 0.152.
 */
package org.apache.avro.specific;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.avro.Protocol;
import org.apache.avro.Schema;
import org.apache.avro.tool.Tool;

public class SpecificCompiler {
    private final Set<Schema> queue = new HashSet<Schema>();
    private final Protocol protocol;
    private static final Set<String> RESERVED_WORDS = new HashSet<String>(Arrays.asList("abstract", "assert", "boolean", "break", "byte", "case", "catch", "char", "class", "const", "continue", "default", "do", "double", "else", "enum", "extends", "false", "final", "finally", "float", "for", "goto", "if", "implements", "import", "instanceof", "int", "interface", "long", "native", "new", "null", "package", "private", "protected", "public", "return", "short", "static", "strictfp", "super", "switch", "synchronized", "this", "throw", "throws", "transient", "true", "try", "void", "volatile", "while"));
    private static final String FILE_HEADER = "/**\n * Autogenerated by Avro\n * \n * DO NOT EDIT DIRECTLY\n */\n";
    private static final Schema NULL_SCHEMA = Schema.create(Schema.Type.NULL);

    public SpecificCompiler(Protocol protocol) {
        for (Schema s : protocol.getTypes()) {
            this.enqueue(s);
        }
        this.protocol = protocol;
    }

    public SpecificCompiler(Schema schema) {
        this.enqueue(schema);
        this.protocol = null;
    }

    public static void compileProtocol(File src, File dest) throws IOException {
        Protocol protocol = Protocol.parse(src);
        SpecificCompiler compiler = new SpecificCompiler(protocol);
        compiler.compileToDestination(src, dest);
    }

    public static void compileSchema(File src, File dest) throws IOException {
        Schema schema = Schema.parse(src);
        SpecificCompiler compiler = new SpecificCompiler(schema);
        compiler.compileToDestination(src, dest);
    }

    static String mangle(String word) {
        if (RESERVED_WORDS.contains(word)) {
            return word + "$";
        }
        return word;
    }

    private void enqueue(Schema schema) {
        if (this.queue.contains(schema)) {
            return;
        }
        switch (schema.getType()) {
            case RECORD: {
                this.queue.add(schema);
                for (Schema.Field field : schema.getFields()) {
                    this.enqueue(field.schema());
                }
                break;
            }
            case MAP: {
                this.enqueue(schema.getValueType());
                break;
            }
            case ARRAY: {
                this.enqueue(schema.getElementType());
                break;
            }
            case UNION: {
                for (Schema s : schema.getTypes()) {
                    this.enqueue(s);
                }
                break;
            }
            case ENUM: 
            case FIXED: {
                this.queue.add(schema);
                break;
            }
            case STRING: 
            case BYTES: 
            case INT: 
            case LONG: 
            case FLOAT: 
            case DOUBLE: 
            case BOOLEAN: 
            case NULL: {
                break;
            }
            default: {
                throw new RuntimeException("Unknown type: " + schema);
            }
        }
    }

    Collection<OutputFile> compile() {
        ArrayList<OutputFile> out = new ArrayList<OutputFile>();
        for (Schema schema : this.queue) {
            out.add(this.compile(schema));
        }
        if (this.protocol != null) {
            out.add(this.compileInterface(this.protocol));
        }
        return out;
    }

    private void compileToDestination(File src, File dst) throws IOException {
        for (Schema schema : this.queue) {
            OutputFile o = this.compile(schema);
            File outputFile = new File(dst, o.path);
            o.writeToDestination(src, dst);
        }
        if (this.protocol != null) {
            this.compileInterface(this.protocol).writeToDestination(src, dst);
        }
    }

    private OutputFile compileInterface(Protocol protocol) {
        OutputFile outputFile = new OutputFile();
        String mangledName = SpecificCompiler.mangle(protocol.getName());
        outputFile.path = SpecificCompiler.makePath(mangledName, protocol.getNamespace());
        StringBuilder out = new StringBuilder();
        this.header(out, protocol.getNamespace());
        this.doc(out, 1, protocol.getDoc());
        this.line(out, 0, "public interface " + mangledName + " {");
        this.line(out, 1, "public static final org.apache.avro.Protocol PROTOCOL = org.apache.avro.Protocol.parse(\"" + SpecificCompiler.esc(protocol) + "\");");
        for (Map.Entry<String, Protocol.Message> e : protocol.getMessages().entrySet()) {
            String name = e.getKey();
            Protocol.Message message = e.getValue();
            Schema request = message.getRequest();
            String response = message.isOneWay() ? "void" : this.unbox(message.getResponse());
            this.doc(out, 1, e.getValue().getDoc());
            this.line(out, 1, response + " " + SpecificCompiler.mangle(name) + "(" + this.params(request) + ")" + (message.isOneWay() ? "" : " throws org.apache.avro.ipc.AvroRemoteException" + this.errors(message.getErrors())) + ";");
        }
        this.line(out, 0, "}");
        outputFile.contents = out.toString();
        return outputFile;
    }

    static String makePath(String name, String space) {
        if (space == null || space.isEmpty()) {
            return name + ".java";
        }
        return space.replace('.', File.separatorChar) + File.separatorChar + name + ".java";
    }

    private void header(StringBuilder out, String namespace) {
        if (namespace != null) {
            this.line(out, 0, "package " + namespace + ";\n");
        }
        this.line(out, 0, "@SuppressWarnings(\"all\")");
    }

    private String params(Schema request) {
        StringBuilder b = new StringBuilder();
        int count = 0;
        for (Schema.Field param : request.getFields()) {
            String paramName = SpecificCompiler.mangle(param.name());
            b.append(this.unbox(param.schema()));
            b.append(" ");
            b.append(paramName);
            if (++count >= request.getFields().size()) continue;
            b.append(", ");
        }
        return b.toString();
    }

    private String errors(Schema errs) {
        StringBuilder b = new StringBuilder();
        for (Schema error : errs.getTypes().subList(1, errs.getTypes().size())) {
            b.append(", ");
            b.append(SpecificCompiler.mangle(error.getFullName()));
        }
        return b.toString();
    }

    private OutputFile compile(Schema schema) {
        OutputFile outputFile = new OutputFile();
        String name = SpecificCompiler.mangle(schema.getName());
        outputFile.path = SpecificCompiler.makePath(name, schema.getNamespace());
        StringBuilder out = new StringBuilder();
        this.header(out, schema.getNamespace());
        switch (schema.getType()) {
            case RECORD: {
                this.doc(out, 0, schema.getDoc());
                this.line(out, 0, "public class " + name + (schema.isError() ? " extends org.apache.avro.specific.SpecificExceptionBase" : " extends org.apache.avro.specific.SpecificRecordBase") + " implements org.apache.avro.specific.SpecificRecord {");
                this.line(out, 1, "public static final org.apache.avro.Schema SCHEMA$ = org.apache.avro.Schema.parse(\"" + SpecificCompiler.esc(schema) + "\");");
                for (Schema.Field field : schema.getFields()) {
                    this.doc(out, 1, field.doc());
                    this.line(out, 1, "public " + this.unbox(field.schema()) + " " + SpecificCompiler.mangle(field.name()) + ";");
                }
                this.line(out, 1, "public org.apache.avro.Schema getSchema() { return SCHEMA$; }");
                this.line(out, 1, "// Used by DatumWriter.  Applications should not call. ");
                this.line(out, 1, "public java.lang.Object get(int field$) {");
                this.line(out, 2, "switch (field$) {");
                int i = 0;
                for (Schema.Field field : schema.getFields()) {
                    this.line(out, 2, "case " + i++ + ": return " + SpecificCompiler.mangle(field.name()) + ";");
                }
                this.line(out, 2, "default: throw new org.apache.avro.AvroRuntimeException(\"Bad index\");");
                this.line(out, 2, "}");
                this.line(out, 1, "}");
                this.line(out, 1, "// Used by DatumReader.  Applications should not call. ");
                this.line(out, 1, "@SuppressWarnings(value=\"unchecked\")");
                this.line(out, 1, "public void put(int field$, java.lang.Object value$) {");
                this.line(out, 2, "switch (field$) {");
                i = 0;
                for (Schema.Field field : schema.getFields()) {
                    this.line(out, 2, "case " + i++ + ": " + SpecificCompiler.mangle(field.name()) + " = (" + this.type(field.schema()) + ")value$; break;");
                }
                this.line(out, 2, "default: throw new org.apache.avro.AvroRuntimeException(\"Bad index\");");
                this.line(out, 2, "}");
                this.line(out, 1, "}");
                this.line(out, 0, "}");
                break;
            }
            case ENUM: {
                this.doc(out, 0, schema.getDoc());
                this.line(out, 0, "public enum " + name + " { ");
                StringBuilder b = new StringBuilder();
                int count = 0;
                for (String symbol : schema.getEnumSymbols()) {
                    b.append(SpecificCompiler.mangle(symbol));
                    if (++count >= schema.getEnumSymbols().size()) continue;
                    b.append(", ");
                }
                this.line(out, 1, b.toString());
                this.line(out, 0, "}");
                break;
            }
            case FIXED: {
                this.doc(out, 0, schema.getDoc());
                this.line(out, 0, "@org.apache.avro.specific.FixedSize(" + schema.getFixedSize() + ")");
                this.line(out, 0, "public class " + name + " extends org.apache.avro.specific.SpecificFixed {}");
                break;
            }
            case MAP: 
            case ARRAY: 
            case UNION: 
            case STRING: 
            case BYTES: 
            case INT: 
            case LONG: 
            case FLOAT: 
            case DOUBLE: 
            case BOOLEAN: 
            case NULL: {
                break;
            }
            default: {
                throw new RuntimeException("Unknown type: " + schema);
            }
        }
        outputFile.contents = out.toString();
        return outputFile;
    }

    private void doc(StringBuilder out, int indent, String doc) {
        if (doc != null) {
            this.line(out, indent, "/** " + this.escapeForJavaDoc(doc) + " */");
        }
    }

    private String escapeForJavaDoc(String doc) {
        return doc.replace("*/", "*&#47;");
    }

    private String type(Schema schema) {
        switch (schema.getType()) {
            case RECORD: 
            case ENUM: 
            case FIXED: {
                return SpecificCompiler.mangle(schema.getFullName());
            }
            case ARRAY: {
                return "java.util.List<" + this.type(schema.getElementType()) + ">";
            }
            case MAP: {
                return "java.util.Map<java.lang.CharSequence," + this.type(schema.getValueType()) + ">";
            }
            case UNION: {
                List<Schema> types = schema.getTypes();
                if (types.size() == 2 && types.contains(NULL_SCHEMA)) {
                    return this.type(types.get(types.get(0).equals(NULL_SCHEMA) ? 1 : 0));
                }
                return "java.lang.Object";
            }
            case STRING: {
                return "java.lang.CharSequence";
            }
            case BYTES: {
                return "java.nio.ByteBuffer";
            }
            case INT: {
                return "java.lang.Integer";
            }
            case LONG: {
                return "java.lang.Long";
            }
            case FLOAT: {
                return "java.lang.Float";
            }
            case DOUBLE: {
                return "java.lang.Double";
            }
            case BOOLEAN: {
                return "java.lang.Boolean";
            }
            case NULL: {
                return "java.lang.Void";
            }
        }
        throw new RuntimeException("Unknown type: " + schema);
    }

    private String unbox(Schema schema) {
        switch (schema.getType()) {
            case INT: {
                return "int";
            }
            case LONG: {
                return "long";
            }
            case FLOAT: {
                return "float";
            }
            case DOUBLE: {
                return "double";
            }
            case BOOLEAN: {
                return "boolean";
            }
        }
        return this.type(schema);
    }

    private void line(StringBuilder out, int indent, String text) {
        for (int i = 0; i < indent; ++i) {
            out.append("  ");
        }
        out.append(text);
        out.append("\n");
    }

    static String esc(Object o) {
        return o.toString().replace("\"", "\\\"");
    }

    public static void main(String[] args) throws Exception {
        SpecificCompiler.compileProtocol(new File(args[0]), new File(args[1]));
    }

    public static class SpecificCompilerTool
    implements Tool {
        @Override
        public int run(InputStream in, PrintStream out, PrintStream err, List<String> args) throws Exception {
            if (args.size() != 3) {
                System.err.println("Expected 3 arguments: (schema|protocol) inputfile outputdir");
                return 1;
            }
            String method = args.get(0);
            File input = new File(args.get(1));
            File output = new File(args.get(2));
            if ("schema".equals(method)) {
                SpecificCompiler.compileSchema(input, output);
            } else if ("protocol".equals(method)) {
                SpecificCompiler.compileProtocol(input, output);
            } else {
                System.err.println("Expected \"schema\" or \"protocol\".");
                return 1;
            }
            return 0;
        }

        @Override
        public String getName() {
            return "compile";
        }

        @Override
        public String getShortDescription() {
            return "Generates Java code for the given schema.";
        }
    }

    static class OutputFile {
        String path;
        String contents;

        OutputFile() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        File writeToDestination(File src, File destDir) throws IOException {
            File f = new File(destDir, this.path);
            if (src != null && f.exists() && f.lastModified() >= src.lastModified()) {
                return f;
            }
            f.getParentFile().mkdirs();
            FileWriter fw = new FileWriter(f);
            try {
                fw.write(SpecificCompiler.FILE_HEADER);
                fw.write(this.contents);
            }
            finally {
                fw.close();
            }
            return f;
        }
    }
}

