/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import java.io.File;
import java.io.IOError;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.Collection;
import java.util.HashMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.CRC32;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.RowMutation;
import org.apache.cassandra.db.commitlog.ReplayPosition;
import org.apache.cassandra.io.util.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CommitLogSegment {
    private static final Logger logger = LoggerFactory.getLogger(CommitLogSegment.class);
    private static final String FILENAME_PREFIX = "CommitLog-";
    private static final String FILENAME_EXTENSION = ".log";
    private static Pattern COMMIT_LOG_FILE_PATTERN = Pattern.compile("CommitLog-(\\d+).log");
    static final int ENTRY_OVERHEAD_SIZE = 20;
    private final HashMap<Integer, Integer> cfLastWrite = new HashMap();
    public final long id = System.nanoTime();
    private final File logFile = new File(DatabaseDescriptor.getCommitLogLocation(), "CommitLog-" + this.id + ".log");
    private RandomAccessFile logFileAccessor;
    private boolean needsSync = false;
    private final MappedByteBuffer buffer;
    private boolean closed;

    public static CommitLogSegment freshSegment() {
        return new CommitLogSegment(null);
    }

    CommitLogSegment(String filePath) {
        boolean isCreating = true;
        try {
            File oldFile;
            if (filePath != null && (oldFile = new File(filePath)).exists()) {
                logger.debug("Re-using discarded CommitLog segment for " + this.id + " from " + filePath);
                oldFile.renameTo(this.logFile);
                isCreating = false;
            }
            this.logFileAccessor = new RandomAccessFile(this.logFile, "rw");
            if (isCreating) {
                logger.debug("Creating new commit log segment " + this.logFile.getPath());
            }
            this.logFileAccessor.setLength(0x8000000L);
            this.buffer = this.logFileAccessor.getChannel().map(FileChannel.MapMode.READ_WRITE, 0L, 0x8000000L);
            this.buffer.putInt(0);
            this.buffer.position(0);
            this.needsSync = true;
        }
        catch (IOException e) {
            throw new IOError(e);
        }
    }

    public static long idFromFilename(String filename) {
        Matcher matcher = COMMIT_LOG_FILE_PATTERN.matcher(filename);
        try {
            if (matcher.matches()) {
                return Long.valueOf(matcher.group(1));
            }
            return -1L;
        }
        catch (NumberFormatException e) {
            return -1L;
        }
    }

    public static boolean possibleCommitLogFile(String filename) {
        return COMMIT_LOG_FILE_PATTERN.matcher(filename).matches();
    }

    public void discard() {
        this.close();
        try {
            FileUtils.deleteWithConfirm(this.logFile);
        }
        catch (IOException e) {
            throw new IOError(e);
        }
    }

    public CommitLogSegment recycle() {
        this.buffer.position(0);
        this.buffer.putInt(0);
        this.buffer.position(0);
        try {
            this.sync();
        }
        catch (IOException e) {
            logger.warn("I/O error flushing " + this + " " + e);
        }
        this.close();
        return new CommitLogSegment(this.getPath());
    }

    public boolean hasCapacityFor(RowMutation mutation) {
        long totalSize = RowMutation.serializer().serializedSize(mutation, 4) + 20L;
        return totalSize <= (long)this.buffer.remaining();
    }

    private void markDirty(RowMutation rowMutation, ReplayPosition repPos) {
        for (ColumnFamily columnFamily : rowMutation.getColumnFamilies()) {
            CFMetaData cfm = Schema.instance.getCFMetaData(columnFamily.id());
            if (cfm == null) {
                logger.error("Attempted to write commit log entry for unrecognized column family: " + columnFamily.id());
                continue;
            }
            this.markCFDirty(cfm.cfId, repPos.position);
        }
    }

    public ReplayPosition write(RowMutation rowMutation) throws IOException {
        assert (!this.closed);
        ReplayPosition repPos = this.getContext();
        this.markDirty(rowMutation, repPos);
        CRC32 checksum = new CRC32();
        byte[] serializedRow = rowMutation.getSerializedBuffer(4);
        checksum.update(serializedRow.length);
        this.buffer.putInt(serializedRow.length);
        this.buffer.putLong(checksum.getValue());
        this.buffer.put(serializedRow);
        checksum.update(serializedRow);
        this.buffer.putLong(checksum.getValue());
        if (this.buffer.remaining() >= 4) {
            this.buffer.putInt(0);
            this.buffer.position(this.buffer.position() - 4);
        }
        this.needsSync = true;
        return repPos;
    }

    public void sync() throws IOException {
        if (this.needsSync) {
            this.buffer.force();
            this.needsSync = false;
        }
    }

    public ReplayPosition getContext() {
        return new ReplayPosition(this.id, this.buffer.position());
    }

    public String getPath() {
        return this.logFile.getPath();
    }

    public String getName() {
        return this.logFile.getName();
    }

    public void close() {
        if (this.closed) {
            return;
        }
        try {
            this.logFileAccessor.close();
            this.closed = true;
        }
        catch (IOException e) {
            throw new IOError(e);
        }
    }

    private void markCFDirty(Integer cfId, Integer position) {
        this.cfLastWrite.put(cfId, position);
    }

    public void markClean(Integer cfId, ReplayPosition context) {
        Integer lastWritten = this.cfLastWrite.get(cfId);
        if (!(lastWritten == null || this.contains(context) && lastWritten >= context.position)) {
            this.cfLastWrite.remove(cfId);
        }
    }

    public Collection<Integer> getDirtyCFIDs() {
        return this.cfLastWrite.keySet();
    }

    public boolean isUnused() {
        return this.cfLastWrite.isEmpty();
    }

    public boolean contains(ReplayPosition context) {
        return context.segment == this.id;
    }

    public String dirtyString() {
        StringBuilder sb = new StringBuilder();
        for (Integer cfId : this.cfLastWrite.keySet()) {
            CFMetaData m = Schema.instance.getCFMetaData(cfId);
            sb.append(m == null ? "<deleted>" : m.cfName).append(" (").append(cfId).append("), ");
        }
        return sb.toString();
    }

    public String toString() {
        return "CommitLogSegment(" + this.getPath() + ')';
    }

    public int position() {
        return this.buffer.position();
    }
}

