package org.gcube.keycloak.account;

import java.net.URI;

import javax.ws.rs.ForbiddenException;
import javax.ws.rs.FormParam;
import javax.ws.rs.NotAuthorizedException;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.core.Response;

import org.gcube.keycloak.event.OrchestratorEventPublisherProviderFactory;
import org.jboss.logging.Logger;
import org.jboss.resteasy.annotations.cache.NoCache;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.RealmModel;
import org.keycloak.services.managers.AppAuthManager;
import org.keycloak.services.managers.AuthenticationManager;
import org.keycloak.services.resources.RealmsResource;

public class DeleteAccountResource {

    protected static final Logger logger = Logger.getLogger(DeleteAccountResource.class);

    public static final String STATE_CHECKER_ATTRIBUTE = "state_checker";
    public static final String STATE_CHECKER_PARAMETER = "stateChecker";

    private final KeycloakSession session;
    private final AuthenticationManager.AuthResult auth;

    public DeleteAccountResource(KeycloakSession session) {
        logger.info("Created new DeleteAccountResource object");
        this.session = session;
        auth = new AppAuthManager().authenticateIdentityCookie(session, session.getContext().getRealm());
    }

    @NoCache
    @POST()
    @Path("delete")
    public Response performDeleteAccount(@FormParam(STATE_CHECKER_PARAMETER) String stateChecker) {
        if (auth == null) {
            logger.debug("Invoked DELETE without authorization");
            throw new NotAuthorizedException("Cookie");
        }
        String requiredStateChecker = session.getAttribute(STATE_CHECKER_ATTRIBUTE, String.class);
        if (!requiredStateChecker.equals(stateChecker)) {
            throw new ForbiddenException("State");
        }
        logger.info("Invoked perform delete account");
        RealmModel realm = auth.getSession().getRealm();
        logger.debug("Sending delete account event to the orchestrator");
        new OrchestratorEventPublisherProviderFactory().create(session)
                .publish(new DeleteAccountEvent(auth.getUser(), realm));

        logger.debug("Forcing logout from all active sessions");
        session.sessions().removeUserSessions(realm);

        
        URI auccountLoginUri = session.getContext().getUri().getBaseUriBuilder().path(RealmsResource.class)
                .path("{realm}/account").build(realm.getName());

        logger.debugf("Finally redirecting to the account form login: %s", auccountLoginUri);
        return Response.status(302).location(auccountLoginUri).build();
    }

}
