/**
 * 
 */
package org.gcube.dataanalysis.copernicus.motu.client;

import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Vector;
import java.util.stream.Collectors;

import org.apache.log4j.Logger;
import org.gcube.dataanalysis.copernicus.motu.model.ProductMetadataInfo;
import org.gcube.dataanalysis.copernicus.motu.model.RequestSize;
import org.gcube.dataanalysis.copernicus.motu.model.Variable;
import org.gcube.dataanalysis.copernicus.motu.util.TimeUtil;

class MapUtil {
    public static <K, V extends Comparable<? super V>> Map<K, V> sortByValue(
            Map<K, V> map) {
        return map.entrySet().stream()
                .sorted(Map.Entry.comparingByValue(Collections.reverseOrder()))
                .collect(
                        Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue,
                                (e1, e2) -> e1, LinkedHashMap::new));
    }
}

/**
 *
 * This is an utility class to split a request for dataset into a number of
 * small requests whose response fit the maximum allowed size of the server.
 *
 * @author Paolo Fabriani
 *
 */

public class RequestSplitter {

    /**
     * A logger for this class.
     */
    private static final Logger LOGGER = Logger
            .getLogger(RequestSplitter.class);

    /**
     * A Motu client to use to connect to the server to get sizes.
     */
    private MotuClient motuClient;

    /**
     * The root request to be split.
     */
    private DownloadRequest request;

    /**
     * The maximum size of a chunk to be downloaded.
     */
    private Long chunkSize;
    

    public RequestSplitter(DownloadRequest request) {
        this.request = request;
    }
    
    /**
     * Split variables across a number of chunks, fitting the max size, using a
     * first-fit decreasing algorithm.
     * @param request
     * @return
     * @throws Exception
     */
    private Collection<DownloadRequestEnvelope> splitAlongVariables(
            DownloadRequestEnvelope request) throws Exception {
        // if variables are not set, ask the service for variables
        Collection<String> variables = new Vector<>();
        if (request.getVariables() == null
                || request.getVariables().size() == 0) {
            ProductMetadataInfo pmi = this.getMotuClient()
                    .describeProduct(request);
            for (Variable var : pmi.getVariables()) {
                variables.add(var.getName());
            }
        } else {
            variables.addAll(request.getVariables());
        }
        // get the size for each variable
        Map<String, Long> sizes = new HashMap<>();
        for (String var : variables) {
            DownloadRequestEnvelope envelope = new DownloadRequestEnvelope(
                    request);
            envelope.setVariables(null);
            envelope.addVariable(var);
            RequestSize size = this.getMotuClient().getSize(envelope);
            sizes.put(var, size.getSizeInBytes());
        }
        // sort sizes in ascending order
        Map<String, Long> sortedSizes = MapUtil.sortByValue(sizes);
        // prepare the output
        Collection<DownloadRequestEnvelope> output = new Vector<>();
        // this is to record the size of each chunk
        Map<DownloadRequestEnvelope, Long> chunkSizes = new HashMap<>();
        // max allowed size
        Long maxSize = this.getChunkSize(request);
        int seq = 0;
        for (Map.Entry<String, Long> entry : sortedSizes.entrySet()) {
            String variable = entry.getKey();
            Long varSize = entry.getValue();
            boolean found = false;
            // look for a request with space for the variable
            for (DownloadRequestEnvelope chunk : output) {
                Long chunkSize = chunkSizes.get(chunk);
                if (chunkSize + varSize < maxSize) {
                    // add the variable to the chunk
                    chunk.addVariable(variable);
                    // update the chunk size
                    chunkSizes.put(chunk, chunkSize + varSize);
                    found = true;
                    break;
                }
            }
            if (!found) {
                // no request has room for the variable. Create a new chunk
                DownloadRequestEnvelope chunk = new DownloadRequestEnvelope(
                        request);
                chunk.setVariables(null);
                chunk.addVariable(variable);
                output.add(chunk);
                String nameSuffix = String.format("-%s%02d", "v", seq);
                String name = request.getName() + nameSuffix;
                chunk.setName(name);
                seq++;
                chunkSizes.put(chunk, varSize);
            }
        }
        return output;
    }

    private Collection<DownloadRequestEnvelope> splitAlongTime(
            final DownloadRequestEnvelope request) throws Exception {

        LOGGER.info("splitting along time");
        
        // prepare output
        Collection<DownloadRequestEnvelope> output = new Vector<>();

        // extract time range from the request
        Long minTime = request.gettLo().getTimeInMillis();
        Long maxTime = request.gettHi().getTimeInMillis();

        // how many chunks are needed?
        Long reqSize = request.getSize().getSizeInBytes();
        Long maxSize = this.getChunkSize(request);

        // 1.1 factor to stay a bit lower than the maximum
        Long nr = Math.round(Math.ceil(reqSize * 1d / maxSize * 1.1));
        Double step = (maxTime * 1d - minTime * 1d) / nr * 1d;

        LOGGER.debug("we need to do " + nr + " requests to split along time");
        
        // build requests for chunks
        int seq = 0;
        for (int i = 0; i < nr; i++) {
            Double chunkStart = minTime + i * step;

            DownloadRequestEnvelope chunk = new DownloadRequestEnvelope(
                    request);
            Calendar from = Calendar.getInstance();
            from.setTimeInMillis(Math.round(chunkStart));
            Calendar to = Calendar.getInstance();
            to.setTimeInMillis(Math.round(chunkStart + step));
            chunk.settRange(from, to);

            String nameSuffix = String.format("-%s%02d", "t", seq);
            String name = request.getName() + nameSuffix;
            chunk.setName(name);
            output.add(chunk);
            seq++;
        }
        return output;
    }

    private Collection<DownloadRequestEnvelope> splitAlongAxis(
            DownloadRequestEnvelope request, String axis) throws Exception {

        Collection<DownloadRequestEnvelope> output = new Vector<>();

        Double min = Double
                .parseDouble(request.getParametersMap().getFirst(axis + "_lo"));
        Double max = Double
                .parseDouble(request.getParametersMap().getFirst(axis + "_hi"));

        Long reqSize = request.getSize().getSizeInBytes();
        // System.out.println("THE REQUESTED SIZE IS " + reqSize);
        Long maxSize = this.getChunkSize(request);
        // System.out.println("THE MAX SIZE IS " + reqSize);

        // System.out.println("CEIL IS " + Math.ceil(reqSize*1d / maxSize*1d));

        // 1.1 factor to stay a bit lower than the maximum
        Long nr = Math.round(Math.ceil(reqSize * 1d / maxSize * 1.1));
        // System.out.println("nr is " + nr);
        Double step = (max - min) / nr;

        int seq = 0;
        for (int i = 0; i < nr; i++) {
            Double var = min + i * step;
            // for (Double var = min; var < max; var += step) {
            DownloadRequestEnvelope chunk = new DownloadRequestEnvelope(
                    request);
            if (axis.equals("x")) {
                chunk.setxRange(var, Math.min(var + step, max));
            }
            if (axis.equals("y")) {
                chunk.setyRange(var, Math.min(var + step, max));
            }
            if (axis.equals("z")) {
                chunk.setzRange(var, Math.min(var + step, max));
            }
            String nameSuffix = String.format("-%s%02d", axis, seq);
            String name = request.getName() + nameSuffix;
            chunk.setName(name);
            seq++;
            output.add(chunk);
        }
        return output;
    }

    public Collection<DownloadRequestEnvelope> splitRequest(String... splitParameters) throws Exception {
        DownloadRequestEnvelope envelope = new DownloadRequestEnvelope(this.request);
        envelope.setName("root");
        envelope.setSize(this.getMotuClient().getSize(request));
        return this.splitRequest(envelope, splitParameters);
    }

    private Collection<DownloadRequestEnvelope> splitRequest(
            DownloadRequestEnvelope request, String... splitParameters)
            throws Exception {
        
        // get the size of the request
        Long size = request.getSize().getSizeInBytes();

        // how much should be chunks?
        Long chunkSize = this.getChunkSize(request); 

        Collection<DownloadRequestEnvelope> output = new Vector<>();

        // if below max we're done
        if (size <= chunkSize) {
            LOGGER.info("request size is within limits. Proceeding");
            output.add(request);
            // nothing to do
        } else {
            String splitParam = splitParameters[0];
            LOGGER.info("request size is too big. Splitting along the " + splitParam + " dimension");
            Collection<DownloadRequestEnvelope> chunks = null;
            if (splitParam.equals("v")) {
                chunks = this.splitAlongVariables(request);
            } else if (splitParam.equals("x") || splitParam.equals("y")
                    || splitParam.equals("z")) {
                chunks = this.splitAlongAxis(request, splitParam);
            } else if (splitParam.equals("t")) {
                chunks = this.splitAlongTime(request);
            } else {
                LOGGER.error("unable to split along " + splitParam);
            }

            String[] newsp = Arrays.copyOfRange(splitParameters, 1,
                    splitParameters.length);

            for(DownloadRequestEnvelope chunk:chunks) {
                if(chunk.getSize()==null) {
                    chunk.setSize(this.getMotuClient().getSize(chunk));
                }
            }

            if (splitParameters.length > 1) {
                for (DownloadRequestEnvelope chunk : chunks) {
                    output.addAll(this.splitRequest(chunk, newsp));
                }
            } else {
                output.addAll(chunks);
            }
        }
        return output;

    }

    private Long getChunkSize(DownloadRequestEnvelope request) throws Exception {
        if(this.chunkSize==null) {
            if(request.getSize()==null) {
                request.setSize(this.motuClient.getSize(request));
            }
            this.chunkSize = Math.min(request.getSize().getMaxAllowedSizeInBytes(),
                    this.getMotuClient().getPreferredDownloadSize());
        }
        return this.chunkSize;
    }

    public MotuClient getMotuClient() {
        return motuClient;
    }

    public void setMotuClient(MotuClient motuClient) {
        this.motuClient = motuClient;
    }

}
