package org.gcube.dataanalysis.ecoengine.transducers;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

import org.gcube.contentmanagement.graphtools.utils.DateGuesser;
import org.gcube.contentmanagement.lexicalmatcher.utils.AnalysisLogger;
import org.gcube.contentmanagement.lexicalmatcher.utils.DatabaseFactory;
import org.gcube.contentmanagement.lexicalmatcher.utils.DistanceCalculator;
import org.gcube.dataanalysis.ecoengine.configuration.AlgorithmConfiguration;
import org.gcube.dataanalysis.ecoengine.configuration.INFRASTRUCTURE;
import org.gcube.dataanalysis.ecoengine.datatypes.ColumnType;
import org.gcube.dataanalysis.ecoengine.datatypes.DatabaseType;
import org.gcube.dataanalysis.ecoengine.datatypes.InputTable;
import org.gcube.dataanalysis.ecoengine.datatypes.OutputTable;
import org.gcube.dataanalysis.ecoengine.datatypes.PrimitiveType;
import org.gcube.dataanalysis.ecoengine.datatypes.ServiceType;
import org.gcube.dataanalysis.ecoengine.datatypes.StatisticalType;
import org.gcube.dataanalysis.ecoengine.datatypes.enumtypes.PrimitiveTypes;
import org.gcube.dataanalysis.ecoengine.datatypes.enumtypes.ServiceParameters;
import org.gcube.dataanalysis.ecoengine.datatypes.enumtypes.TableTemplates;
import org.gcube.dataanalysis.ecoengine.interfaces.Transducerer;
import org.gcube.dataanalysis.ecoengine.test.regression.Regressor;
import org.gcube.dataanalysis.ecoengine.utils.DatabaseUtils;
import org.gcube.dataanalysis.ecoengine.utils.ResourceFactory;
import org.hibernate.SessionFactory;

public class OccurrencePointsMerger implements Transducerer {

	static protected String finalTableNameL = "final_Table_Name";
	static String longitudeColumn = "longitudeColumn";
	static String latitudeColumn = "latitudeColumn";
	static String recordedByColumn = "recordedByColumn";
	static String scientificNameColumn = "scientificNameColumn";
	static String eventDateColumn = "eventDateColumn";
	static String lastModificationColumn = "lastModificationColumn";
	static String rightTableNameF = "rightTableName";
	static String leftTableNameF = "leftTableName";
	static String finalTableNameF = "finalTableName";
	static String spatialTolerance = "spatialTolerance";
	static String confidence = "confidence";
	

	protected List<OccurrenceRecord> records_left;
	protected List<OccurrenceRecord> records_right;
	protected AlgorithmConfiguration config;

	protected String lonFld;
	protected String latFld;
	protected String recordedByFld;
	protected String scientificNameFld;
	protected String eventDatFld;
	protected String modifDatFld;
	protected String leftTableName;
	protected String rightTableName;
	protected String finalTableName;
	protected String finalTableLabel;
	protected float spatialToleranceValue;
	protected float confidenceValue;
	protected StringBuffer columns;
	protected List<OccurrenceRecord> objectstoinsert;
	protected List<OccurrenceRecord> objectstodelete;
	protected List<Object> columnsNames;
	protected SessionFactory dbconnection;
	protected float status;

	
	public OccurrencePointsMerger(){
		
	}
	
	protected class OccurrenceRecord {

		public String scientificName;
		public String recordedby;
		public Calendar eventdate;
		public Calendar modifdate;
		// public String locality;
		// public String country;
		public float x;
		public float y;

		// Map<String,String> metadata;
		public List<String> otherValues;

		public OccurrenceRecord() {
			otherValues = new ArrayList<String>();
		}
	}

	public static String convert2conventionalFormat(Calendar date) {
		if (date == null)
			return "";
		SimpleDateFormat formatter = new SimpleDateFormat("MM/dd/yy KK:mm:ss a");
		String formattedDate = formatter.format(new Date(date.getTimeInMillis()));
		return formattedDate;
	}

	boolean displaydateconvert = true;

	public OccurrenceRecord row2OccurrenceRecord(Object[] row) {
		OccurrenceRecord record = new OccurrenceRecord();
		int index = 0;

		for (Object name : columnsNames) {
			String name$ = "" + name;
			String value$ = "" + row[index];
			if (name$.equalsIgnoreCase(lonFld)) {
				record.x = Float.parseFloat(value$);
			} else if (name$.equalsIgnoreCase(latFld)) {
				record.y = Float.parseFloat(value$);
			} else if (name$.equalsIgnoreCase(recordedByFld)) {
				record.recordedby = value$;
			} else if (name$.equalsIgnoreCase(scientificNameFld)) {
				record.scientificName = value$;
			} else if (name$.equalsIgnoreCase(eventDatFld)) {
				if ((value$ == null) || (value$.length() == 0)) {
					record.eventdate = null;
				} else {
					/*
					 * SimpleDateFormat formatter = new SimpleDateFormat("MM/dd/yy KK:mm a",Locale.UK); try { Date d = (Date) formatter.parse(value$); Calendar cal = Calendar.getInstance(); cal.setTime(d); System.out.println("From "+value$+"->"+(cal.get(Calendar.MONTH)+1)+" "+cal.get(Calendar.DAY_OF_MONTH)+" "+cal.get(Calendar.YEAR)+" "+cal.get(Calendar.HOUR)+" "+cal.get(Calendar.MINUTE)); // System.out.println("->"+cal.toString()); } catch (ParseException e) { // TODO Auto-generated catch block e.printStackTrace(); }
					 */
					record.eventdate = DateGuesser.convertDate(value$);
					if (displaydateconvert) {
						AnalysisLogger.getLogger().trace("From " + value$ + "->" + convert2conventionalFormat(record.eventdate) + " pattern " + DateGuesser.getPattern(value$));
						displaydateconvert = false;
					}

				}
			} else if (name$.equalsIgnoreCase(modifDatFld)) {
				record.modifdate = DateGuesser.convertDate(value$);
			} else
				record.otherValues.add(value$);

			index++;
		}

		return record;
	}

	public String occurrenceRecord2String(OccurrenceRecord record) {
		StringBuffer buffer = new StringBuffer();
		int index = 0;
		int k = 0;
		int nNames = columnsNames.size();
		for (Object name : columnsNames) {

			String name$ = "" + name;
			String value$ = "NULL";
			if (name$.equalsIgnoreCase(lonFld)) {
				value$ = "'" + record.x + "'";
			} else if (name$.equalsIgnoreCase(latFld)) {
				value$ = "'" + record.y + "'";
			} else if (name$.equalsIgnoreCase(recordedByFld)) {
				if (record.recordedby != null)
					value$ = "'" + record.recordedby.replace("'", "") + "'";
			} else if (name$.equalsIgnoreCase(scientificNameFld)) {
				if (record.scientificName != null)
					value$ = "'" + record.scientificName.replace("'", "") + "'";
			} else if (name$.equalsIgnoreCase(eventDatFld)) {
				if (record.eventdate != null) {
					String dat = convert2conventionalFormat(record.eventdate);
					if ((dat!=null)&&(dat.length()>0))
						value$ = "'" + convert2conventionalFormat(record.eventdate) + "'";
					else
						value$ = "NULL";
					// value$="'"+record.eventdate.getTimeInMillis()+"'";
				}
			} else if (name$.equalsIgnoreCase(modifDatFld)) {
				if (record.modifdate != null) {
					String dat = convert2conventionalFormat(record.modifdate) ;
					if ((dat!=null)&&(dat.length()>0))
						value$ = "'" + convert2conventionalFormat(record.modifdate) + "'";
					else 
						value$ = "NULL";
					// value$="'"+record.modifdate.getTimeInMillis()+"'";
				}
			} else {
				if (record.otherValues != null) {
					value$ = "'" + record.otherValues.get(k).replace("'", "") + "'";
					k++;
				}
			}
			if (value$.equals("'null'"))
				value$ = "NULL";
			
			buffer.append(value$);
			if (index < nNames - 1) {
				buffer.append(",");
			}

			index++;
		}

		return buffer.toString();
	}


	@Override
	public List<StatisticalType> getInputParameters() {
		List<TableTemplates> templatesOccurrence = new ArrayList<TableTemplates>();
		templatesOccurrence.add(TableTemplates.OCCURRENCE_SPECIES);
		// occurrence points tables
		PrimitiveType p0 = new PrimitiveType(String.class.getName(), null, PrimitiveTypes.STRING, finalTableNameL,"the name of the produced table", "Occ_");
		
		InputTable p1 = new InputTable(templatesOccurrence, leftTableNameF, "the First table containing the occurrence points", "");
		InputTable p2 = new InputTable(templatesOccurrence, rightTableNameF, "the Second table containing the occurrence points", "");

		// string parameters
		ColumnType p3 = new ColumnType(leftTableNameF, longitudeColumn, "column with longitude values", "decimallongitude", false);
		ColumnType p4 = new ColumnType(leftTableNameF, latitudeColumn, "column with latitude values", "decimallatitude", false);
		ColumnType p5 = new ColumnType(leftTableNameF, recordedByColumn, "column with RecordedBy values", "recordedby", false);
		ColumnType p6 = new ColumnType(leftTableNameF, scientificNameColumn, "column with Scientific Names", "scientificname", false);
		ColumnType p7 = new ColumnType(leftTableNameF, eventDateColumn, "column with EventDate values", "eventdate", false);
		ColumnType p8 = new ColumnType(leftTableNameF, lastModificationColumn, "column with Modified values", "modified", false);
		ServiceType p9 = new ServiceType(ServiceParameters.RANDOMSTRING, finalTableNameF, "name of the resulting table", "processedOccurrences_");
		PrimitiveType p10 = new PrimitiveType(Float.class.getName(), null, PrimitiveTypes.NUMBER, spatialTolerance, "the tolerance in degree for assessing that two points could be the same", "0.5");
		PrimitiveType p11 = new PrimitiveType(Float.class.getName(), null, PrimitiveTypes.NUMBER, confidence, "the overall acceptance similarity threshold over which two points are the same - from 0 to 100", "80");

		List<StatisticalType> inputs = new ArrayList<StatisticalType>();
		inputs.add(p0);
		inputs.add(p1);
		inputs.add(p2);
		inputs.add(p3);
		inputs.add(p4);
		inputs.add(p5);
		inputs.add(p6);
		inputs.add(p7);
		inputs.add(p8);
		inputs.add(p9);
		inputs.add(p10);
		inputs.add(p11);
		
		DatabaseType.addDefaultDBPars(inputs);
		return inputs;
	}

	@Override
	public String getResources() {
		if ((status > 0) && (status < 100))
			return ResourceFactory.getResources(100f);
		else
			return ResourceFactory.getResources(0f);
	}

	ResourceFactory resourceManager;

	@Override
	public String getResourceLoad() {
		if (resourceManager == null)
			resourceManager = new ResourceFactory();
		return resourceManager.getResourceLoad(1);
	}

	@Override
	public float getStatus() {
		return status;
	}

	@Override
	public INFRASTRUCTURE getInfrastructure() {
		return INFRASTRUCTURE.LOCAL;
	}

	@Override
	public StatisticalType getOutput() {
		List<TableTemplates> templatesOccurrence = new ArrayList<TableTemplates>();
		templatesOccurrence.add(TableTemplates.OCCURRENCE_SPECIES);
		// occurrence points tables
		OutputTable p = new OutputTable(templatesOccurrence, finalTableLabel, finalTableName, "The output table containing the processed points");

		return p;
	}

	@Override
	public void init() throws Exception {

		AnalysisLogger.setLogger(config.getConfigPath() + AlgorithmConfiguration.defaultLoggerFile);
		lonFld = config.getParam(longitudeColumn);
		latFld = config.getParam(latitudeColumn);
		recordedByFld = config.getParam(recordedByColumn);
		scientificNameFld = config.getParam(scientificNameColumn);
		eventDatFld = config.getParam(eventDateColumn);
		modifDatFld = config.getParam(lastModificationColumn);
		leftTableName = config.getParam(leftTableNameF);
		rightTableName = config.getParam(rightTableNameF);
		finalTableName = config.getParam(finalTableNameF);
		finalTableLabel = config.getParam(finalTableNameL);
		spatialToleranceValue = Float.parseFloat(config.getParam(spatialTolerance));
		confidenceValue = Float.parseFloat(config.getParam(confidence));

		objectstoinsert = new ArrayList<OccurrencePointsMerger.OccurrenceRecord>();
		objectstodelete = new ArrayList<OccurrencePointsMerger.OccurrenceRecord>();
		status = 0;
	}

	@Override
	public void setConfiguration(AlgorithmConfiguration config) {
		this.config = config;
	}

	@Override
	public void shutdown() {

	}

	@Override
	public String getDescription() {
		return "An algorithm for merging two sets of occurrence points of species coming from the Species Discovery Facility of D4Science";
	}

	protected float probabilityStrings(String first, String second) {
		if ((first == null) || (second == null))
			return 1;

		return (float) new DistanceCalculator().CD(false, first, second);
	}

	protected float probabilityDates(Calendar first, Calendar second) {
		if ((first == null) || (second == null))
			return 1;
		if (first.compareTo(second) == 0)
			return 1;
		else
			return 0;
	}

	protected float extProb(OccurrenceRecord right, OccurrenceRecord left) {
		float probability = 0;
		float distance = (float) Math.sqrt(Math.abs(left.x - right.x) + Math.abs(left.y - right.y));
		if (distance > spatialToleranceValue)
			probability = 0;
		else {
			float pSpecies = probabilityStrings(right.scientificName, left.scientificName);
			float pRecordedBy = probabilityStrings(right.recordedby, left.recordedby);
			float pDates = probabilityDates(right.eventdate, left.eventdate);
			probability = pSpecies * pRecordedBy * pDates;
		}

		return probability * 100;
	}

	protected void manageHighProbability(float probability, OccurrenceRecord leftOcc, OccurrenceRecord rightOcc) {
		// insert the most recent:
		// if it is the left then leave it as is
		// otherwise put the left in the deletion list and the right in the insertion list
		
		if (
				((leftOcc.modifdate!=null)&&(rightOcc.modifdate!=null)&&leftOcc.modifdate.before(rightOcc.modifdate)) 
				|| 
				(leftOcc.modifdate==null)&&(rightOcc.modifdate!=null)
				)
		{
		
			objectstodelete.add(leftOcc);
			objectstoinsert.add(rightOcc);
		}
	}

	protected void manageLowProbability(float probability, OccurrenceRecord leftOcc, OccurrenceRecord rightOcc) {
		// if over the threshold then add to the element
		objectstoinsert.add(rightOcc);
	}

	protected void persist() throws Exception{

		// DELETE ELEMENTS IN THE DELETION LIST
		int todel = objectstodelete.size();
		int counter = 0;
		StringBuffer buffer = new StringBuffer();
		AnalysisLogger.getLogger().debug("Deleting " + todel + " objects");
		if (todel > 0) {
			for (OccurrenceRecord record : objectstodelete) {
				String rec = recordedByFld + "='" + record.recordedby.replace("'","")+"'";
				String sci = scientificNameFld + "='" + record.scientificName.replace("'","")+"'";
				String x = lonFld + "='" + record.x+"'";
				String y = latFld + "='" + record.y+"'";
				String event = null;
				String modified = null;
				if (record.eventdate!=null)
					event = eventDatFld + "='" + convert2conventionalFormat(record.eventdate)+"'";
				
				if (record.modifdate!=null)
					modified = modifDatFld + "='" + convert2conventionalFormat(record.modifdate)+"'";
				
				buffer.append("(");
				buffer.append(rec + " AND " + sci + " AND " + x + " AND " + y);
				if (event!=null)
					buffer.append(" AND "+event);
				if (modified!=null)
					buffer.append(" AND "+modified);
				
				buffer.append(")");
				if (counter < todel - 1)
					buffer.append(" OR ");

				counter++;
			}

			String updateQ = DatabaseUtils.deleteFromBuffer(finalTableName, buffer);
//			 System.out.println("Update:\n"+updateQ);
			DatabaseFactory.executeSQLUpdate(updateQ, dbconnection);
			AnalysisLogger.getLogger().debug("Objects deleted");
		}

		buffer = new StringBuffer();
		int toins = objectstoinsert.size();
		AnalysisLogger.getLogger().debug("Inserting " + toins + " objects");
		counter = 0;
		if (toins > 0) {
			for (OccurrenceRecord record : objectstoinsert) {
				buffer.append("(");
				buffer.append(occurrenceRecord2String(record));
				buffer.append(")");
				if (counter < toins - 1)
					buffer.append(",");

				counter++;
			}

			String updateQ = DatabaseUtils.insertFromBuffer(finalTableName, columns.toString(), buffer);
			 System.out.println("Update:\n"+updateQ);
			AnalysisLogger.getLogger().debug("Update:\n"+updateQ);
			DatabaseFactory.executeSQLUpdate(updateQ, dbconnection);
			AnalysisLogger.getLogger().debug("Objects inserted");
		}

	}
	protected void prepareFinalTable() throws Exception{
		DatabaseFactory.executeSQLUpdate(DatabaseUtils.duplicateTableStatement(leftTableName, finalTableName), dbconnection);
	}
	
	protected void extractColumnNames() throws Exception{
		// take the description of the table
					columnsNames = DatabaseFactory.executeSQLQuery(DatabaseUtils.getColumnsNamesStatement(rightTableName), dbconnection);

					int nCols = columnsNames.size();
					columns = new StringBuffer();
					for (int i = 0; i < nCols; i++) {
						columns.append("\"" + columnsNames.get(i) + "\"");
						if (i < nCols - 1)
							columns.append(",");
					}
	}
	@Override
	public void compute() throws Exception {

		try {
			// init DB connection
			AnalysisLogger.getLogger().trace("Initializing DB Connection");
			dbconnection = DatabaseUtils.initDBSession(config);
			AnalysisLogger.getLogger().trace("Taking Table Description");
			extractColumnNames();

			AnalysisLogger.getLogger().trace("Taken Table Description: " + columns);
			AnalysisLogger.getLogger().trace("Creating final table: " + finalTableName);
			// create new merged table
			try{
			DatabaseFactory.executeSQLUpdate(DatabaseUtils.dropTableStatement(finalTableName), dbconnection);
			}catch(Exception e1){
				
			}
			prepareFinalTable();
			// take the elements from sx table
			AnalysisLogger.getLogger().trace("Taking elements from left table: " + leftTableName);
			List<Object> leftRows = DatabaseFactory.executeSQLQuery(DatabaseUtils.getColumnsElementsStatement(leftTableName, columns.toString(), ""), dbconnection);
			// take the elements from dx table
			AnalysisLogger.getLogger().trace("Taking elements from right table: " + rightTableName);
			List<Object> rightRows = DatabaseFactory.executeSQLQuery(DatabaseUtils.getColumnsElementsStatement(rightTableName, columns.toString(), ""), dbconnection);
			// for each element in dx
			List<OccurrenceRecord> leftRecords = new ArrayList<OccurrencePointsMerger.OccurrenceRecord>();
			AnalysisLogger.getLogger().trace("Processing " + leftTableName + " vs " + rightTableName);
			status = 10;
			int rightCounter = 0;
			int similaritiesCounter = 0;
			int allrightrows = rightRows.size();
			for (Object rRow : rightRows) {
				// transform into an occurrence object
				OccurrenceRecord rightOcc = row2OccurrenceRecord((Object[]) rRow);
				// for each element in sx
				int k = 0;
				int leftrecordsSize = 0;
				boolean found = false;
				float p = 0;
				OccurrenceRecord bestleftOcc = null;
				for (Object lRow : leftRows) {
					OccurrenceRecord leftOcc = null;
					// only for the first iteration on the left occurrences perform the transformation
					if (leftrecordsSize <= k) {
						// transform into an occurrence object
						leftOcc = row2OccurrenceRecord((Object[]) lRow);
						leftRecords.add(leftOcc);
						leftrecordsSize++;
						// System.out.println("ADDED "+k+"-th elements size: "+leftRecords.size());
					} else
						leftOcc = leftRecords.get(k);

					// evaluate P(dx,sx)
					p = extProb(leftOcc, rightOcc);

					if (p >= confidenceValue) {
						bestleftOcc = leftOcc;
						found = true;
						similaritiesCounter++;
						AnalysisLogger.getLogger().trace("Found a similarity with P=" + p + " between (" + "\"" + leftOcc.scientificName + "\"" + "," + leftOcc.x + "\"" + "," + "\"" + leftOcc.y + "\"" + "," + "\"" + leftOcc.recordedby + "\"" + "," + "\"" + convert2conventionalFormat(leftOcc.eventdate) + "\"" + ") VS " + "(" + "\"" + rightOcc.scientificName + "\"" + "," + "\"" + rightOcc.x + "\"" + "," + "\"" + rightOcc.y + "\"" + "," + "\"" + rightOcc.recordedby + "\"" + "," + "\"" + convert2conventionalFormat(rightOcc.eventdate) + "\"" + ")");
						break;
					}
					k++;
				}
				rightCounter++;
				if (found)
					manageHighProbability(p, bestleftOcc, rightOcc);
				else
					manageLowProbability(p, bestleftOcc, rightOcc);

				status = Math.min(90, 10f + (80 * ((float) rightCounter) / ((float) allrightrows)));
			}

			AnalysisLogger.getLogger().trace("Found " + similaritiesCounter + " similarities on " + rightCounter + " elements");
			status = 90;
			// transform the complete list into a table
			persist();
			// close DB connection
		} catch (Exception e) {
			throw e;
		} finally {
			if (dbconnection != null)
				dbconnection.close();
			status = 100;
			AnalysisLogger.getLogger().trace("Occ Points Processing Finished and db closed");
		}
	}
	
	public static void main(String[] args) throws Exception {
		AlgorithmConfiguration config = Regressor.getConfig();
		config.setNumberOfResources(1);
		config.setParam(longitudeColumn, "decimallongitude");
		config.setParam(latitudeColumn, "decimallatitude");
		config.setParam(recordedByColumn, "recordedby");
		config.setParam(scientificNameColumn, "scientificname");
		config.setParam(eventDateColumn, "eventdate");
		config.setParam(lastModificationColumn, "modified");
		config.setParam(rightTableNameF, "whitesharkoccurrences2");
		config.setParam(leftTableNameF, "whitesharkoccurrences1");
		config.setParam(finalTableNameF, "whitesharkoccurrencesmerged");
		config.setParam(spatialTolerance, "0.5");
		config.setParam(confidence, "0.8");

		OccurrencePointsMerger occm = new OccurrencePointsMerger();
		occm.setConfiguration(config);
		occm.init();
		occm.compute();
	}
}
