package org.gcube.dataanalysis.geo.retrieval;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;

import org.gcube.contentmanagement.lexicalmatcher.utils.AnalysisLogger;
import org.gcube.dataanalysis.ecoengine.configuration.AlgorithmConfiguration;
import org.gcube.dataanalysis.ecoengine.signals.SignalProcessing;
import org.gcube.dataanalysis.ecoengine.utils.Tuple;
import org.gcube.dataanalysis.geo.meta.features.FeaturesManager;
import org.gcube.dataanalysis.geo.utils.EnvDataExplorer;
import org.gcube.dataanalysis.geo.utils.ThreddsDataExplorer;
import org.opengis.metadata.Metadata;
import org.opengis.metadata.identification.Identification;

public class GeoIntersector {

	private FeaturesManager featurer;
	private String configDir;
	public GeoIntersector(String scope, String cfgDir) {
		featurer = new FeaturesManager();
		featurer.setScope(scope);
		this.configDir=cfgDir;
	}

	public LinkedHashMap<String, Double> getFeaturesInTime(String layerTitle, double x, double y) throws Exception {
		return getFeaturesInTime(layerTitle, x, y, 0);
	}

	public LinkedHashMap<String, Double> getFeaturesInTime(String layerTitle, double x, double y, double z) throws Exception {
		LinkedHashMap<String, Double> features = new LinkedHashMap<String, Double>();
		// get the layer
		// Metadata meta = featurer.getGNInfobyTitle(layerTitle);
		Metadata meta = featurer.checkForMetadatabyTitle(FeaturesManager.treatTitleForGN(layerTitle), layerTitle);
		// if the layer is good
		if (meta != null) {
			String layer = featurer.getLayerName(meta);

			if (layer == null)
				layer = layerTitle;

			// check if it is a NetCDF
			if (featurer.isThreddsFile(meta)) {
				Identification id = meta.getIdentificationInfo().iterator().next();
				String title = id.getCitation().getTitle().toString();

				AnalysisLogger.getLogger().debug("found a netCDF file with title " + title + " and layer name " + layer);
				features = getFeaturesFromNetCDF(featurer.getOpenDapLink(meta), layer, x, y, z);
			} else {
				AnalysisLogger.getLogger().debug("found a Geo Layer with title " + layerTitle + " and layer name " + layer);
				features = getFeaturesFromWFS(featurer.getWFSLink(meta), layer, x, y);
			}
		}

		return features;
	}

	public List<LinkedHashMap<String, Double>> getFeaturesInTime(String layerTitle, List<Tuple<Double>> triplets) throws Exception {
		List<LinkedHashMap<String, Double>> featuresSets = new ArrayList<LinkedHashMap<String, Double>>();
		// get the layer
		Metadata meta = featurer.checkForMetadatabyTitle(FeaturesManager.treatTitleForGN(layerTitle), layerTitle);
		// if the layer is good
		if (meta != null) {
			String layer = featurer.getLayerName(meta);
			if (layer == null)
				layer = layerTitle;
			// check if it is a NetCDF
			if (featurer.isThreddsFile(meta)) {
				Identification id = meta.getIdentificationInfo().iterator().next();
				String title = id.getCitation().getTitle().toString();
				AnalysisLogger.getLogger().debug("found a netCDF file with title " + title + " and layer name " + layer);
				for (Tuple<Double> triplet : triplets) {
					double x = triplet.getElements().get(0);
					double y = triplet.getElements().get(1);
					double z = 0;
					if (triplet.getElements().size() > 2)
						z = triplet.getElements().get(2);
					AnalysisLogger.getLogger().debug("Taking point: (" + x + "," + y + "," + z + ")");
					LinkedHashMap<String, Double> features = new LinkedHashMap<String, Double>();
					features = getFeaturesFromNetCDF(featurer.getOpenDapLink(meta), layer, x, y, z);
					AnalysisLogger.getLogger().debug("Got: (" + features + ")");
					featuresSets.add(features);
				}

			} else {
				AnalysisLogger.getLogger().debug("found a Geo Layer with title " + layerTitle + " and layer name " + layer);
				for (Tuple<Double> triplet : triplets) {
					double x = triplet.getElements().get(0);
					double y = triplet.getElements().get(1);
					AnalysisLogger.getLogger().debug("Taking point: (" + x + "," + y + ")");
					LinkedHashMap<String, Double> features = new LinkedHashMap<String, Double>();
					features = getFeaturesFromWFS(featurer.getWFSLink(meta), layer, x, y);
					featuresSets.add(features);
				}
			}
		}

		return featuresSets;
	}

	private LinkedHashMap<String, Double> getFeaturesFromNetCDF(String opendapURL, String layer, double x, double y, double z) {
		if (opendapURL == null)
			return null;

		return ThreddsDataExplorer.retrieveDataFromNetCDF(opendapURL, layer, x, y, z);
	}

	private LinkedHashMap<String, Double> getFeaturesFromWFS(String wfsUrl, String layer, double x, double y) {
		if (wfsUrl == null)
			return null;

		return EnvDataExplorer.getFeatures(wfsUrl, layer, x, y);
	}

	public double[][] takeLastTimeChunk(String layerTitle, double x1, double x2, double y1, double y2, double z, double xResolution, double yResolution) throws Exception {
		AnalysisLogger.getLogger().debug("Bounding box: (" + x1 + "," + x2 + ";" + y1 + "," + y2 + ")");
		if ((x2 < x1) || (y2 < y1)) {
			AnalysisLogger.getLogger().debug("ERROR: BAD BOUNDING BOX!!!");
			return new double[0][0];
		}

		int ysteps = (int) ((y2 - y1) / yResolution);
		int xsteps = (int) ((x2 - x1) / xResolution);
		double[][] slice = new double[ysteps + 1][xsteps + 1];
		List<Tuple<Double>> tuples = new ArrayList<Tuple<Double>>();
		AnalysisLogger.getLogger().debug("Building the points grid according to YRes:" + yResolution + " and XRes:" + xResolution);
		// build the tuples according to the desired resolution
		for (int i = 0; i < ysteps + 1; i++) {
			double y = (i * yResolution) + y1;
			if (i == ysteps)
				y = y2;
			for (int j = 0; j < xsteps + 1; j++) {
				double x = (j * xResolution) + x1;
				if (j == xsteps)
					x = x2;
				tuples.add(new Tuple<Double>(x, y, z));
			}
		}

		AnalysisLogger.getLogger().debug("Taking " + ysteps + " values...");
		List<LinkedHashMap<String, Double>> time = getFeaturesInTime(layerTitle, tuples);
		AnalysisLogger.getLogger().debug("Taken " + time.size() + " values");

		// build back the values matrix
		int k = 0;
		int g = 0;
		int ntriplets = ysteps * xsteps;
		//cycle on all the triplets to recontruct the matrix
		for (int t = 0; t < ntriplets; t++) {
			//take the corresponding (time,value) pair
			LinkedHashMap<String, Double> tvalues = time.get(t);
			//if there is value, then set it, otherwise set NaN
			//the layer is undefined in that point and a value must be generated
			double value = Double.NaN;
			for (Double val : tvalues.values()) {
				value = val;
				break;
			}
			//assign a value to the matrix
			slice[k][g] = value;
			//increase the x step according to the matrix
			g++;
			if (g == xsteps) {
				g = 0;
				k++;
			}
		}
		AnalysisLogger.getLogger().debug("Applying nearest Neighbor to all the rows");
		//apply nearest neighbor to each row
		AlgorithmConfiguration config = new AlgorithmConfiguration();
		config.setConfigPath(configDir);
		for (int i=0;i<slice.length;i++){
			AnalysisLogger.getLogger().debug("Checking for unfilled values");
			boolean tofill = false;
			for (int j=0;j<slice[i].length;j++) {
				if (slice[i][j]==Double.NaN)
					tofill = true;
			}
			if (tofill){
				AnalysisLogger.getLogger().debug("Filling signal");
				slice[i] = SignalProcessing.fillSignal(slice[i]);
			}
			else
				AnalysisLogger.getLogger().debug("Signal yet complete");
		}
		return slice;
	}
}
