package org.gcube.dataanalysis.dataminer.poolmanager.util;

import static org.gcube.common.authorization.client.Constants.authorizationService;
import static org.gcube.resources.discovery.icclient.ICFactory.clientFor;
import static org.gcube.resources.discovery.icclient.ICFactory.queryFor;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.List;

import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.gcube.common.authorization.client.exceptions.ObjectNotFound;
import org.gcube.common.authorization.library.AuthorizationEntry;
import org.gcube.common.authorization.library.provider.SecurityTokenProvider;
import org.gcube.common.resources.gcore.GCoreEndpoint;
import org.gcube.dataanalysis.dataminer.poolmanager.clients.configuration.DMPMClientConfiguratorManager;
import org.gcube.dataanalysis.dataminer.poolmanager.util.exception.EMailException;
import org.gcube.resources.discovery.client.api.DiscoveryClient;
import org.gcube.resources.discovery.client.queries.api.SimpleQuery;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SendMail {

	private Logger logger = LoggerFactory.getLogger(SendMail.class);
	private final String WRITE_MESSAGE_ADDRESS_PATH = "2/messages/write-message?gcube-token=",
			USER_ROLES_ADDRESS_PATH = "2/users/get-usernames-by-role?role-name=DataMiner-Manager&gcube-token=",
			SOCIAL_SERVICE_QUERY_CONDITION = "$resource/Profile/ServiceName/text() eq 'SocialNetworking'",
			SOCIAL_SERVICE_URI = "jersey-servlet", JSON_MIME_TYPE = "application/json";

	private String socialServiceAddress;

	public SendMail() {

	}

	public void sendNotification(String subject, String body) throws EMailException {
		logger.debug("SendNotification");
		logger.debug("Notification Subject: " + subject);
		logger.debug("Notification Body: " + body);

		retrieveSocialService();

		String postBody = createPostBody(subject, body);
		String requestForMessage = getRequestMessage(WRITE_MESSAGE_ADDRESS_PATH);
		
		sendPostRequest(requestForMessage, postBody);

	}

	private String createPostBody(String subject, String body) throws EMailException {
		try {
			List<String> recipientsList = getRecipients();
			if (recipientsList == null || recipientsList.isEmpty()) {
				logger.error("Invalid recipient list: " + recipientsList);
				throw new EMailException("Unable to send email notification. Invalid recipient list:" + recipientsList);
			}

			// {"subject": "subject-content", "body": "body-content",
			// "recipients":[{"id":"userid"}]}

			JSONObject data = new JSONObject();
			data.put("subject", subject);
			data.put("body", body);

			JSONArray recipients = new JSONArray();
			for (String recipient : recipientsList) {
				JSONObject d = new JSONObject();
				d.put("id", recipient);
				recipients.put(d);
			}
			data.put("recipients", recipients);

			logger.info("Post Body: " + data);
			return data.toString();

		} catch (EMailException e) {
			throw e;
		} catch (Throwable e) {
			logger.error("Error creating the notification body: " + e.getLocalizedMessage(), e);
			throw new EMailException(e);

		}

	}

	private void retrieveSocialService() throws EMailException {
		try {
			SimpleQuery query = queryFor(GCoreEndpoint.class);
			query.addCondition(SOCIAL_SERVICE_QUERY_CONDITION);
			DiscoveryClient<GCoreEndpoint> client = clientFor(GCoreEndpoint.class);
			List<GCoreEndpoint> resources = client.submit(query);
			socialServiceAddress = resources.get(0).profile().endpointMap().get(SOCIAL_SERVICE_URI).uri().toString();
			logger.info("Retrieved Social Service Address: " + socialServiceAddress);
			if (socialServiceAddress == null || socialServiceAddress.isEmpty()) {
				throw new EMailException(
						"Unable to send email notification. Invalid address in GCoreEndpoint resource on IS.");
			}
		} catch (EMailException e) {
			logger.error(e.getLocalizedMessage(), e);
			throw e;
		} catch (Throwable e) {
			logger.error(e.getLocalizedMessage(), e);
			throw new EMailException(e);
		}

	}

	private String getRequestMessage(String addressPath) {
		StringBuilder requestMessageBuilder = new StringBuilder(socialServiceAddress);

		if (!socialServiceAddress.endsWith("/"))
			requestMessageBuilder.append('/');

		requestMessageBuilder.append(addressPath).append(SecurityTokenProvider.instance.get());
		String requestForMessage = requestMessageBuilder.toString();
		logger.debug("Request " + requestForMessage);
		return requestForMessage;
	}

	private String username(String token) throws ObjectNotFound, Exception {
		AuthorizationEntry entry = authorizationService().get(token);
		logger.debug(entry.getClientInfo().getId());
		return entry.getClientInfo().getId();
	}

	private void sendPostRequest(String endpoint, String postBody) throws EMailException {
		logger.info("Execute Post:" + endpoint);
		logger.info("Post Body:" + postBody);
		try {

			// Send the request
			URL url = new URL(endpoint);
			URLConnection conn = url.openConnection();
			conn.setRequestProperty("Accept", JSON_MIME_TYPE);
			conn.setRequestProperty("Content-Type", JSON_MIME_TYPE);
			conn.setDoOutput(true);

			OutputStreamWriter writer = new OutputStreamWriter(conn.getOutputStream());
			writer.write(postBody);
			writer.flush();

			// Get the response
			StringBuffer answer = new StringBuffer();
			BufferedReader reader = new BufferedReader(new InputStreamReader(conn.getInputStream()));
			String line;
			while ((line = reader.readLine()) != null) {
				answer.append(line);
			}
			writer.close();
			reader.close();

			logger.debug("Operation completed");
			String response = answer.toString();
			logger.info("Notification Response: " + response);
			checkResponse(response);

		} catch (EMailException e) {
			throw e;
		} catch (MalformedURLException e) {
			logger.error("Invalid URL: " + e.getLocalizedMessage(), e);
			throw new EMailException(e);
		} catch (IOException e) {
			logger.error("Error in the IO process: " + e.getLocalizedMessage(), e);
			throw new EMailException(e);
		} catch (Throwable e) {
			logger.error("Error executing post:" + e.getLocalizedMessage(), e);
			throw new EMailException(e);
		}

	}

	private void checkResponse(String response) throws EMailException {
		if (response == null) {
			logger.error("Invalid notification response: " + response);
			throw new EMailException();
		} else {
			try {
				JSONObject res = new JSONObject(response);
				boolean success = res.getBoolean("success");
				if (!success) {
					String message = res.getString("message");
					logger.error("Error in send email notification: " + message);
					throw new EMailException("Error in send email notification: "+message);
				}

			} catch (JSONException e) {
				logger.error("Invalid notification response: " + response);
				throw new EMailException(e);
			}

		}
	}

	private List<String> getRecipients() {
		try {
			List<String> recipients = new ArrayList<String>();
			String dataMinerManagers = retrieveDataMinerManagers();
			logger.debug("Retrieved DataMiner Managers: " + dataMinerManagers);
			if (dataMinerManagers != null && !dataMinerManagers.isEmpty()) {
				JSONObject obj = new JSONObject(dataMinerManagers);
				JSONArray data = obj.getJSONArray("result");
				if (data != null) {
					for (int i = 0; i < data.length(); i++) {
						recipients.add(data.getString(i));
					}
				}
			} else {
				logger.info("Use the default admins how workaround ");
				List<String> defaultManagers = DMPMClientConfiguratorManager.getInstance().getDefaultAdmins();
				recipients.addAll(defaultManagers);
			}
			recipients.add(this.username(SecurityTokenProvider.instance.get()));

			logger.info("Retrieved Recipients: " + recipients);
			return recipients;
		} catch (Exception e) {
			logger.error("Error retrieving recipients: " + e.getLocalizedMessage(), e);
			logger.info("Use the default admins how workaround ");
			return DMPMClientConfiguratorManager.getInstance().getDefaultAdmins();
		}

	}

	private String retrieveDataMinerManagers() throws Exception {
		// Try to retrieve a url like this:
		// https://api.d4science.org/social-networking-library-ws/rest/2/users/get-usernames-by-role?role-name=DataMiner-Manager&gcube-token=xxx-xxxx-xxxx-xxx

		String requestAdminsUrl = getRequestMessage(USER_ROLES_ADDRESS_PATH);

		logger.info("Request Admins Url: " + requestAdminsUrl);

		CloseableHttpClient client = HttpClientBuilder.create().build();
		HttpGet getReq = new HttpGet(requestAdminsUrl);
		getReq.setHeader("accept", JSON_MIME_TYPE);
		getReq.setHeader("content-type", JSON_MIME_TYPE);
		logger.info("Response: " + EntityUtils.toString(client.execute(getReq).getEntity()));

		return EntityUtils.toString(client.execute(getReq).getEntity());

	}

}
