/**
 * 
 */
package org.gcube.dataanalysis.copernicus.cmems.importer.service.service;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.nio.file.FileAlreadyExistsException;
import java.text.ParseException;
import java.util.Calendar;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.Vector;

import javax.xml.bind.JAXBException;

import org.gcube.dataanalysis.copernicus.cmems.importer.service.exception.AlreadyThereException;
import org.gcube.dataanalysis.copernicus.cmems.importer.service.exception.InvalidParameterException;
import org.gcube.dataanalysis.copernicus.cmems.importer.service.exception.MalformedElementException;
import org.gcube.dataanalysis.copernicus.cmems.importer.service.exception.NotThereException;
import org.gcube.dataanalysis.copernicus.cmems.importer.service.exception.StoreException;
import org.gcube.dataanalysis.copernicus.cmems.importer.service.model.comparator.TaskLastExecutionTimeComparator;
import org.gcube.dataanalysis.copernicus.cmems.importer.task.Execution;
import org.gcube.dataanalysis.copernicus.cmems.importer.task.ExecutionReport;
import org.gcube.dataanalysis.copernicus.cmems.importer.task.ImportTask;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Paolo Fabriani
 *
 */
public class TaskStore {

    // TODO: avoid colon (:) in ids and names

    private static Logger logger = LoggerFactory.getLogger(TaskStore.class);
    
    /**
     * Where tasks, execution and reports are kept in the local file system.
     */
    private static final String DEFAULT_PERSISTENCY_PATH = "/tmp/cmems-tasks";

    /**
     * How many items are returned, by default
     */
    private static final Integer DEFAULT_LIMIT = 100;
    
    private TaskStoreHelper fileStoreHelper;
    
    /**
     * Basic constructor.
     */
    public TaskStore() throws InvalidParameterException {
        this(DEFAULT_PERSISTENCY_PATH);
    }

    /**
     * Basic constructor.
     */
    public TaskStore(String persistencyPath) throws InvalidParameterException {
        if(persistencyPath==null) {
            logger.error("no persistency path given");
            throw new InvalidParameterException("", "bad persistency path: " + persistencyPath);
        }
        File file = new File(persistencyPath);
        if(!file.exists()) {
            logger.info("persistency path does not exist. Creating it: " + persistencyPath);
            file.mkdirs();
        }
        if(file.exists() && !file.isDirectory()) {
            logger.error("persistency path is not a directory");
            throw new InvalidParameterException("", "persistency path: " + persistencyPath + " is not a directory");
        }
        logger.debug("Using report store at " + persistencyPath);
        fileStoreHelper = new TaskStoreHelper(persistencyPath);
    }

    public File getRoot() {
        return this.fileStoreHelper.getRoot();
    }
    
    /**
     * Search tasks matching the given filters.
     * @param user
     * @param scope
     * @param token
     * @param limit
     * @return
     */
    public List<ImportTask> searchTasks(String user, String scope, String token, Integer limit) {
        
        // default limit, if none is given
        if(limit==null) {
            limit = DEFAULT_LIMIT;
        }
        
        // scan all tasks and filter them
        SortedSet<ImportTask> sortedTasks = new TreeSet<>(new TaskLastExecutionTimeComparator());
        for(String id:fileStoreHelper.getAllTasks()) {
            try {
                ImportTask t = this.getTask(id);
                // filter
                if(user!=null && !user.equals(t.getSubmissionInfo().getUser())) {
                    continue;
                }
                if(scope!=null && t.getSubmissionInfo().getScheduled()!=null && !t.getSubmissionInfo().getScope().startsWith(scope)) {
                    continue;
                }
                if(token!=null && !token.equals(t.getSubmissionInfo().getToken())) {
                    continue;
                }
                sortedTasks.add(t);
            } catch(StoreException | NotThereException | MalformedElementException | InvalidParameterException e) {
                e.printStackTrace();
            }
        }
        
        // only the first 'limit' ones.
        List<ImportTask> out = new Vector<>();
        int i=0;
        for(ImportTask t:sortedTasks) {
            if(i>=limit) {
                break;
            }
            out.add(t);
            i++;
        }

        // hide tokens
        for(ImportTask t:out) {
            t.getSubmissionInfo().setToken(null);
        }
        
        return out;
    }
    
    /**
     * Return the given task. The task is enriched with the last execution.
     * @param taskId
     * @return
     * @throws Exception
     */
    public ImportTask getTask(String taskId) throws InvalidParameterException, MalformedElementException, NotThereException, StoreException {
        
        if(taskId==null || taskId.trim().isEmpty())
            throw new InvalidParameterException("id", "task id is null");
        
        ImportTask t;
        try {
            t = this.fileStoreHelper.deserializeTask(taskId);
        } catch(ParseException e) {
            e.printStackTrace();
            throw new MalformedElementException(taskId);
        } catch(FileNotFoundException e) {
            logger.info(e.getMessage());
            throw new NotThereException(taskId);
        } catch(IOException | JAXBException e) {
            e.printStackTrace();
            throw new StoreException(taskId);
        }
        // add the last execution (if any)
        String lastExecId = this.fileStoreHelper.getLastExecution(taskId);
        if(lastExecId!=null) {
            t.setLastExecution(this.getExecution(taskId,  lastExecId));
        }
        // hide the token
        t.getSubmissionInfo().setToken(null);
        return t;

    }
    
    public boolean hasExecution(String taskId, String executionId) {
        File f = this.fileStoreHelper.getExecutionInfoFile(taskId, executionId);
        return f.exists();
    }
    
    /**
     * Return all executions for the given task.
     * @param taskId
     * @return a list of executions
     * @throws NotThereException when no taskId is found
     */
    public List<Execution> getExecutions(String taskId) throws InvalidParameterException, StoreException, MalformedElementException, NotThereException {

        if(taskId==null || taskId.trim().isEmpty())
            throw new InvalidParameterException("id", "task id is null");
        
        List<Execution> out = new Vector<>();
        File executionsDir = this.fileStoreHelper.getExecutionsRoot(taskId);
        if(executionsDir.exists()) {
            for(String executionId:this.fileStoreHelper.getAllExecutions(taskId)) {
                out.add(this.getExecution(taskId, executionId));
            }
        }
        return out;
    }
    
    /**
     * Return the given execution.
     * @param taskId
     * @param executionId
     * @return
     * @throws NotThereException if the task can't be retrieved
     * @throws MalformedElementException if the execution can't be parsed
     */
    public Execution getExecution(String taskId, String executionId) throws InvalidParameterException, StoreException, NotThereException, MalformedElementException {

        if(taskId==null || taskId.trim().isEmpty())
            throw new InvalidParameterException("id", "task id is null");
        if(executionId==null || executionId.trim().isEmpty())
            throw new InvalidParameterException("id", "execution id is null");
        
        try {
            // retrieve the execution
            Execution e = this.fileStoreHelper.deserializeExecution(taskId, executionId);
            // enrich it with reports
            e.setReports(this.getReports(taskId, executionId));
            return e;
        } catch(FileNotFoundException e) {
            logger.info(e.getMessage());
            throw new NotThereException(taskId+"/"+executionId);
        } catch(IOException | JAXBException e) {
            e.printStackTrace();
            throw new StoreException(taskId+"/"+executionId);
        } catch(ParseException e) {
            e.printStackTrace();
            throw new MalformedElementException(taskId+"/"+executionId);
        }
    }
    
    public Execution getLastExecution(String taskId)  throws InvalidParameterException, StoreException, NotThereException, MalformedElementException {
        String lastExecId = this.fileStoreHelper.getLastExecution(taskId);
        if(lastExecId==null) {
            lastExecId="non-existing-execution-id";
        }
        return this.getExecution(taskId,  lastExecId);
    }

    public String getLastExecutionId(String taskId) {
        return this.fileStoreHelper.getLastExecution(taskId);
    }

    /**
     * Return all reports for the given execution.
     * @param taskId
     * @param executionId
     * @return
     * @throws Exception
     */
    public List<ExecutionReport> getReports(String taskId, String executionId) throws InvalidParameterException, NotThereException, MalformedElementException {

        if(taskId==null || taskId.trim().isEmpty())
            throw new InvalidParameterException("id", "task id is null");
        if(executionId==null || executionId.trim().isEmpty())
            throw new InvalidParameterException("id", "execution id is null");
        
        // retrieve reports, if any.
        List<ExecutionReport> out = new Vector<>();
        File reportsDir = this.fileStoreHelper.getExecutionReportsRoot(taskId, executionId);
        if(reportsDir.exists()) {
            for(String r:this.fileStoreHelper.getAllExecutionReports(taskId, executionId)) {
                ExecutionReport report = this.getReport(taskId, executionId, r);
                // remove text here; only snippet.
                report.setText(null);
                out.add(report);
            }
        }
        return out;
    }
    
    /**
     * Return the given report.
     * @param taskId
     * @param executionId
     * @param reportName
     * @return
     * @throws Exception
     */
    public ExecutionReport getReport(String taskId, String executionId, String reportName) throws InvalidParameterException, NotThereException, MalformedElementException {

        if(taskId==null || taskId.trim().isEmpty())
            throw new InvalidParameterException("id", "task id is null");
        if(executionId==null || executionId.trim().isEmpty())
            throw new InvalidParameterException("id", "execution id is null");
        if(reportName==null || reportName.trim().isEmpty())
            throw new InvalidParameterException("name", "report name is null");

        // be sure the report is there
        File reportFile;
        try {
            reportFile = this.fileStoreHelper.ensureReport(taskId, executionId, reportName);
        } catch(FileNotFoundException e) {
            logger.info(e.getMessage());
            throw new NotThereException(taskId+"/"+executionId+"/"+reportName);
        }
        
        // read it
        ExecutionReport out = new ExecutionReport();
        out.setName(reportName);
        out.setSize(reportFile.length());
        String text;
        try {
            text = TaskStoreHelper.getContent(reportFile);
        } catch(IOException e) {
            e.printStackTrace();
            throw new MalformedElementException(taskId+"/"+executionId+"/"+reportName);
        }
        out.setText(text);
        if(text.length()>1000)  {
            out.setSnippet(text.substring(0,  500) + "...");
        } else {
            out.setSnippet(text);
        }
        
        return out;
    }
    
    /**
     * Update the given execution.
     * @param taskId
     * @param execution
     * @return
     * @throws Exception
     */
    public Execution update(String taskId, Execution execution) throws InvalidParameterException, NotThereException, MalformedElementException, StoreException {
        
        if(taskId==null)
            throw new InvalidParameterException("id", "task id is null");
        if(execution==null)
            throw new InvalidParameterException("execution", "execution is null");
        if(execution.getId()==null)
            throw new InvalidParameterException("id", "execution id is null");
        
        // be sure the execution file is there
        File execFile;
        try {
            execFile = this.fileStoreHelper.ensureExecutionInfo(taskId, execution.getId());
        } catch(FileNotFoundException e) {
            logger.info(e.getMessage());
            throw new NotThereException(taskId+"/"+execution.getId());
        }

        // update it
        Execution existing = this.getExecution(taskId, execution.getId());
        if(execution.getEnd()!=null)
            existing.setEnd(execution.getEnd());
        if(execution.getStatus()!=null)
            existing.setStatus(execution.getStatus());
        if(execution.getProgress()!=null)
            existing.setProgress(execution.getProgress());
        
        // mark the execution as updated now
        existing.setLastUpdate(Calendar.getInstance());
        
        try {
            TaskStoreHelper.serializeExecution(execFile, existing);
        } catch(IOException | JAXBException e) {
            e.printStackTrace();
            throw new StoreException(taskId+"/"+execution.getId());
        }

        // return a fresh copy
        return this.getExecution(taskId, execution.getId());
        
    }
    
    private void touchExecution(String taskId, String executionId) throws InvalidParameterException, StoreException, MalformedElementException, NotThereException {
        Execution execution = this.getExecution(taskId, executionId);
        this.touchExecution(taskId, execution);
    }

    private void touchExecution(String taskId, Execution execution) throws InvalidParameterException, StoreException, MalformedElementException, NotThereException {
        this.update(taskId, execution);
    }

    /**
     * Update the given report. All report properties except the text are ignored.
     * @param taskId
     * @param executionId
     * @param report
     * @return
     * @throws Exception
     */
    public ExecutionReport update(String taskId, String executionId, ExecutionReport report) throws InvalidParameterException, NotThereException, MalformedElementException, StoreException {

        if(taskId==null)
            throw new InvalidParameterException("id", "task id is null");
        if(executionId==null)
            throw new InvalidParameterException("id", "execution id is null");
        if(report==null)
            throw new InvalidParameterException("report", "report is null");
        if(report.getName()==null||report.getName().trim().isEmpty())
            throw new InvalidParameterException("name", "report name is null");
        
        // be sure the report is there
        File reportFile;
        try {
            reportFile = this.fileStoreHelper.ensureReport(taskId, executionId, report.getName());
        } catch(FileNotFoundException e) {
            logger.info(e.getMessage());
            throw new NotThereException(taskId+"/"+executionId+"/"+report.getName());
        }

        // update it 
        try {
            TaskStoreHelper.writeContent(reportFile, report.getText());
        } catch(IOException e) {
            e.printStackTrace();
            throw new StoreException(taskId+"/"+executionId+"/"+report.getName());
        }

        // update execution timestamp
        this.touchExecution(taskId, executionId);

        // return a fresh copy
        return this.getReport(taskId, executionId, report.getName());
        
    }

    /**
     * Add a new task.
     * @param task
     * @return
     * @throws Exception
     */
    public ImportTask addTask(ImportTask task) throws InvalidParameterException, AlreadyThereException, StoreException, NotThereException, MalformedElementException {

        if(task==null)
            throw new InvalidParameterException("task", "task is null");
        if(task.getId()==null)
            throw new InvalidParameterException("id", "task id is null");
        
        // be sure the task is not there
        File taskDir;
        try {
            taskDir = this.fileStoreHelper.ensureTaskMiss(task.getId());
        } catch(FileAlreadyExistsException e) {
            throw new AlreadyThereException(task.getId());
        }

        // write it
        try {
            TaskStoreHelper.serializeTask(taskDir, task);
        } catch(IOException | JAXBException e) {
            e.printStackTrace();
            throw new StoreException(task.getId());
        }
        
        // return a fresh copy
        return this.getTask(task.getId());
    }

    /**
     * Add an execution to an existing task.
     * @param taskId
     * @param execution
     * @return
     * @throws Exception
     */
    public Execution addExecution(String taskId, Execution execution) throws InvalidParameterException, AlreadyThereException, NotThereException, StoreException, MalformedElementException {
        
        if(taskId==null)
            throw new InvalidParameterException("id", "task id is null");
        if(execution==null)
            throw new InvalidParameterException("execution", "execution is null");
        if(execution.getId()==null)
            throw new InvalidParameterException("id", "execution id is null");
        
        // be sure the execution is not there
        File executionFile;
        try {
            executionFile = this.fileStoreHelper.ensureExecutionInfoMiss(taskId, execution.getId());
        } catch(FileAlreadyExistsException e) {
            e.printStackTrace();
            throw new AlreadyThereException(taskId+"/"+execution.getId());
        } catch(FileNotFoundException e) {
            logger.info(e.getMessage());
            throw new NotThereException(taskId);
        }
        
        // mark the execution as updated now
        execution.setLastUpdate(Calendar.getInstance());

        // write it
        try {
            TaskStoreHelper.serializeExecution(executionFile, execution);
        } catch(IOException | JAXBException e) {
            e.printStackTrace();
            throw new StoreException(taskId+"/"+execution.getId());
        }

        // return a fresh copy
        return this.getExecution(taskId, execution.getId());
    }

    /**
     * Add a report to an existing execution.
     * @param taskId
     * @param executionId
     * @param report
     * @return
     * @throws Exception
     */
    public ExecutionReport addReport(String taskId, String executionId, ExecutionReport report)  throws InvalidParameterException, NotThereException, AlreadyThereException, StoreException, MalformedElementException {

        if(taskId==null)
            throw new InvalidParameterException("id", "task id is null");
        if(executionId==null)
            throw new InvalidParameterException("id", "execution id is null");
        if(report==null)
            throw new InvalidParameterException("report", "report is null");
        if(report.getName()==null||report.getName().trim().isEmpty())
            throw new InvalidParameterException("name", "report name is null");

        // be sure the report is not there
        File reportFile;
        try {
            reportFile = this.fileStoreHelper.ensureReportMiss(taskId, executionId, report.getName());
        } catch(FileAlreadyExistsException e) {
            e.printStackTrace();
            throw new AlreadyThereException(taskId+"/"+executionId+"/"+report.getName());
        } catch(FileNotFoundException e) {
            logger.info(e.getMessage());
            throw new NotThereException(taskId+"/"+executionId);
        }

        // write it
        try {
            TaskStoreHelper.serializeReport(reportFile, report);
        } catch(IOException e) {
            e.printStackTrace();
            throw new StoreException(taskId+"/"+executionId+"/"+report.getName());
        }
        
        // update execution timestamp
        this.touchExecution(taskId, executionId);

        // return a fresh copy
        return this.getReport(taskId, executionId, report.getName());
        
    }
    
}
