/**
 * 
 */
package org.gcube.dataanalysis.copernicus.cmems.importer.service.util;

import java.text.ParseException;
import java.util.Calendar;
import java.util.List;

import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.core.UriInfo;

import org.gcube.dataanalysis.copernicus.cmems.client.SearchOptions;
import org.gcube.dataanalysis.copernicus.cmems.importer.api.ChunkTimespan;
import org.gcube.dataanalysis.copernicus.cmems.importer.api.ImportOptions;
import org.gcube.dataanalysis.copernicus.cmems.importer.service.exception.InvalidParameterException;
import org.gcube.dataanalysis.copernicus.motu.client.DownloadRequest;
import org.gcube.dataanalysis.datasetimporter.util.TimeUtil;

/**
 * @author Paolo Fabriani
 *
 */
public class ParamUtils {

    public static Integer getIntegerParameter(MultivaluedMap<String, String> queryParams, String parName) throws InvalidParameterException {
        String par = queryParams.getFirst(parName);
        if(par!=null) {
            try {
                Integer out = Integer.parseInt(par);
                return out;
            } catch(NumberFormatException e) {
                throw new InvalidParameterException(parName, String.format("bad value '%s' for Integer parameter '%s'", par, parName));
            }
        } else {
            return null;
        }
    }
    
    public static Boolean getBoooleanParameter(MultivaluedMap<String, String> queryParams, String parName) throws InvalidParameterException {
        String par = queryParams.getFirst(parName);
        if(par!=null) {
            if(par!=null) {
                if("true".equalsIgnoreCase(par.trim()))
                    return Boolean.TRUE;
                else if("false".equalsIgnoreCase(par.trim()))
                    return Boolean.FALSE;
            }            
            throw new InvalidParameterException(parName, String.format("bad value '%s' for Boolean parameter '%s'", par, parName));
        } else {
            return null;
        }
    }

    public static Double getDoubleParameter(MultivaluedMap<String, String> queryParams, String parName) throws InvalidParameterException {
        String par = queryParams.getFirst(parName);
        if(par!=null) {
            try {
                Double out = Double.parseDouble(par);
                return out;
            } catch(NumberFormatException e) {
                throw new InvalidParameterException(parName, String.format("bad value '%s' for Double parameter '%s'", par, parName));
            }
        } else {
            return null;
        }
    }

    public static Calendar getCalendarParameter(MultivaluedMap<String, String> queryParams, String parName) throws InvalidParameterException {
        String par = queryParams.getFirst(parName);
        if(par!=null) {
            try {
                Calendar out = TimeUtil.toCalendar(par);
                return out;
            } catch(ParseException e) {
                throw new InvalidParameterException(parName, String.format("bad value '%s' for Calendar parameter '%s'", par, parName));
            }
        } else {
            return null;
        }
    }
    
    public static ImportOptions queryParamsToImportOptions(UriInfo uriInfo) throws InvalidParameterException {
        MultivaluedMap<String, String> queryParams = uriInfo.getQueryParameters();
        return queryParamsToImportOptions(queryParams);
    }

    public static ImportOptions queryParamsToImportOptions(MultivaluedMap<String, String> queryParams) throws InvalidParameterException {
        
        ImportOptions options = new ImportOptions();

        // back time
        Integer backTime = getIntegerParameter(queryParams, "b");
        if(backTime!=null)
            options.setBackTime(backTime);
        
        // chunk span (s)
        String par = queryParams.getFirst("s");
        if(par!=null) {
            try {
                options.setChunkSpan(ChunkTimespan.fromString(par));
            } catch(Exception e) {
                throw new InvalidParameterException("s", String.format("bad value '%s' for String parameter '%s'", par, "s"));
            }
        }

        // motu (m)
        par = queryParams.getFirst("m");
        options.setMotu(par);

        // product (p)
        par = queryParams.getFirst("p");
        options.setProduct(par);
        
        // dataset (d)
        par = queryParams.getFirst("d");
        options.setDataset(par);

        // import schedule (f)
        par = queryParams.getFirst("f");
        options.setImportSchedule(par);
        
        // t low (tlo)
        Calendar tlo = getCalendarParameter(queryParams, "tlo");
        if(tlo!=null)
            options.settLo(tlo);

        // t high (thi)
        Calendar thi = getCalendarParameter(queryParams, "thi");
        if(thi!=null)
            options.settHi(thi);

        // x low (xlo)
        Double xlo = getDoubleParameter(queryParams, "xlo");
        if(xlo!=null)
            options.setxLo(xlo);

        // x high (xhi)
        Double xhi = getDoubleParameter(queryParams, "xhi");
        if(xhi!=null)
            options.setxHi(xhi);

        // y low (ylo)
        Double ylo = getDoubleParameter(queryParams, "ylo");
        if(ylo!=null)
            options.setyLo(ylo);

        // y high (yhi)
        Double yhi = getDoubleParameter(queryParams, "yhi");
        if(yhi!=null)
            options.setyHi(yhi);

        // z low (zlo)
        Double zlo = getDoubleParameter(queryParams, "zlo");
        if(zlo!=null)
            options.setzLo(zlo);

        // z high (zhi)
        Double zhi = getDoubleParameter(queryParams, "zhi");
        if(zhi!=null)
            options.setzHi(zhi);

        // variables (v)
        List<String> variables = queryParams.get("v");
        if(variables!=null) {
            for(String v:variables) {
                options.addVariable(v);
            }
        }
        
        return options;
    }
    
    public static SearchOptions queryParamsToSearchOptions(UriInfo uriInfo) throws InvalidParameterException {
        MultivaluedMap<String, String> queryParams = uriInfo.getQueryParameters();
        return queryParamsToSearchOptions(queryParams);
    }

    public static SearchOptions queryParamsToSearchOptions(MultivaluedMap<String, String> queryParams) throws InvalidParameterException {
        
        SearchOptions options = new SearchOptions();

        if(queryParams==null)
            return options;
        
        // match keys (k)
        List<String> keys = queryParams.get("k");
        if(keys!=null) {
            for(String k:keys) {
                options.addMatchedKey(k);
            }
        }

        // un-match key (u)
        keys = queryParams.get("u");
        if(keys!=null) {
            for(String k:keys) {
                options.addUnMatchedKey(k);
            }
        }

        // ignore case (i)
        Boolean ignoreCase = getBoooleanParameter(queryParams, "i");
        if(ignoreCase!=null)
            options.setIgnoreCase(ignoreCase);
        
        // variables (v)
        List<String> variables = queryParams.get("v");
        if(variables!=null) {
            for(String v:variables) {
                options.addVariable(v);
            }
        }

        // from f
        Calendar from = getCalendarParameter(queryParams, "f");
        if(from!=null)
            options.setFrom(from);
        
        // to (t)
        Calendar to = getCalendarParameter(queryParams, "t");
        if(to!=null)
            options.setTo(to);
        
        // whole time range (w)
        Boolean whole = getBoooleanParameter(queryParams, "w");
        if(whole!=null)
            options.setWholeTimeRange(whole);
        
        return options;
    }
    
    public static DownloadRequest importOptionsToDownloadRequest(ImportOptions options) {
        DownloadRequest req = new DownloadRequest();
        req.setService(options.getProduct()+"-TDS");
        req.setProduct(options.getDataset());
        req.setVariables(options.getVariables());
        req.settLo(options.gettLo());
        req.settHi(options.gettHi());
        req.setxLo(options.getxLo());
        req.setxHi(options.getxHi());
        req.setyLo(options.getyLo());
        req.setyHi(options.getyHi());
        req.setzLo(options.getzLo());
        req.setzHi(options.getzHi());
        return req;
    }
}
