/**
 * 
 */
package org.gcube.dataanalysis.copernicus.cmems.importer.service.rest.task;

import java.util.Calendar;
import java.util.Collection;
import java.util.List;
import java.util.UUID;

import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.GenericEntity;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.Response.Status;

import org.gcube.dataanalysis.copernicus.cmems.importer.api.ImportOptions;
import org.gcube.dataanalysis.copernicus.cmems.importer.api.ValidationError;
import org.gcube.dataanalysis.copernicus.cmems.importer.service.exception.AlreadyThereException;
import org.gcube.dataanalysis.copernicus.cmems.importer.service.exception.InvalidParameterException;
import org.gcube.dataanalysis.copernicus.cmems.importer.service.exception.MalformedElementException;
import org.gcube.dataanalysis.copernicus.cmems.importer.service.exception.NotThereException;
import org.gcube.dataanalysis.copernicus.cmems.importer.service.exception.StoreException;
import org.gcube.dataanalysis.copernicus.cmems.importer.service.rest.cmems.AbstractImporterResource;
import org.gcube.dataanalysis.copernicus.cmems.importer.service.service.SmartExecutorClient;
import org.gcube.dataanalysis.copernicus.cmems.importer.service.service.TaskStore;
import org.gcube.dataanalysis.copernicus.cmems.importer.service.util.SecurityUtils;
import org.gcube.dataanalysis.copernicus.cmems.importer.task.ImportTask;
import org.gcube.dataanalysis.copernicus.cmems.importer.task.SubmissionInfo;
import org.gcube.vremanagement.executor.exception.ExecutorException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Paolo Fabriani
 *
 */
@Path("/tasks")
public class TasksResource extends AbstractImporterResource {

    private static Logger logger = LoggerFactory.getLogger(TaskResource.class);
    
    /**
     * Return a list of tasks.
     * @return
     */
    @GET
    @Produces(MediaType.APPLICATION_XML)
    public Response listTasks(@QueryParam("scope") String scope, @QueryParam("token") String token, @QueryParam("user") String user, @QueryParam("limit") Integer limit) {
        try {
            List<ImportTask> tasks = new TaskStore().searchTasks(user, scope, token, limit);
            GenericEntity<Collection<ImportTask>> out = new GenericEntity<Collection<ImportTask>>(tasks) {};
            return Response.ok(out).build();
        } catch(InvalidParameterException e) {
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }
    }

    /**
     * Return the given task.
     * @param taskId
     * @return
     */
    @Path("{taskId}")
    public TaskResource getTask(@PathParam("taskId") String taskId) {
        return new TaskResource(taskId);
    }    

    /**
     * Submit a new task. Persist metadata.
     * @param options
     * @return
     */
    @POST
    @Produces(MediaType.APPLICATION_XML)
    @Consumes(MediaType.APPLICATION_XML)
    public Response schedule(ImportOptions options) {

        // extract context (user, token, scope)
        SubmissionInfo info = new SecurityUtils().getCurrentSubmissionInfo();

        // check all security info are present
        if(info.getToken()==null || info.getScope()==null || info.getUser()==null) {
            return Response.status(Status.UNAUTHORIZED).build();
        }

        logger.info(String.format("Received a schedule request from user '%s' in scope '%s'", info.getUser(), info.getScope()));

        // check parameters
        try {
            Collection<ValidationError> errors = this.check(options);
            if(!errors.isEmpty()) {
                if(!errors.isEmpty()) {
                    GenericEntity<Collection<ValidationError>> entity = new GenericEntity<Collection<ValidationError>>(errors) {};
                    return Response.status(Status.BAD_REQUEST).entity(entity).build();
                }
            }
        } catch(Exception e) {
            logger.error(e.toString());
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }

        // build the request context
        info.setScheduled(Calendar.getInstance());
        
        // submit the request to the smart executor
        UUID taskId = null;
        try {
            taskId = new SmartExecutorClient().submit(options);
        } catch(ExecutorException e) {
            e.printStackTrace();
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        }

        // build the task to serialize
        ImportTask task = new ImportTask();
        task.setId(taskId.toString());
        task.setSubmissionInfo(info);
        task.setImportParameters(options);
                
        try {
            ImportTask out = new TaskStore().addTask(task);
            return Response.ok().entity(out).build();
        } catch (MalformedElementException | StoreException | InvalidParameterException e) {
            return Response.status(Status.INTERNAL_SERVER_ERROR).build();
        } catch (NotThereException e) {
            return Response.status(Status.NOT_FOUND).build();
        } catch (AlreadyThereException e) {
            return Response.status(Status.BAD_REQUEST).build();
        }
        
    }
    
    /**
     * Cancel the execution of the given task
     * @param options
     * @return the task id
     */
    @DELETE
    public void unschedule(String taskId) {
        // TODO: connect with the SmartExecutor to cancel the task
    }
    
    /**
     * Return a sandbox
     * @return
     */
    @Path("preview")
    public PreviewResource getPreview() {
        return new PreviewResource();
    }
    
}
