/**
 * 
 */
package org.gcube.dataanalysis.copernicus.cmems.importer.service.rest.cmems;

import java.util.Collection;

import javax.ws.rs.core.Context;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.core.UriInfo;

import org.gcube.dataanalysis.copernicus.cmems.importer.api.ImportOptions;
import org.gcube.dataanalysis.copernicus.cmems.importer.api.ValidationError;
import org.gcube.dataanalysis.copernicus.cmems.importer.service.exception.InvalidParameterException;
import org.gcube.dataanalysis.copernicus.cmems.importer.service.service.DatasetBrowser;
import org.gcube.dataanalysis.copernicus.cmems.importer.service.service.validation.RequestValidator;
import org.gcube.dataanalysis.copernicus.cmems.importer.service.util.ParamUtils;
import org.gcube.dataanalysis.copernicus.cmems.model.CmemsDataset;
import org.gcube.dataanalysis.copernicus.cmems.model.CmemsProduct;
import org.gcube.dataanalysis.copernicus.motu.model.ProductMetadataInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * @author Paolo Fabriani
 *
 */
public abstract class AbstractImporterResource {
    
    private static Logger logger = LoggerFactory.getLogger(AbstractImporterResource.class);

    protected ImportOptions extractImportParameters(@Context UriInfo uriInfo) throws InvalidParameterException {
        // get the import options
        MultivaluedMap<String, String> queryParams = uriInfo.getQueryParameters();
        // extract parameters and check syntax
        ImportOptions options = ParamUtils.queryParamsToImportOptions(queryParams);
        return options;
    }    

    protected Collection<ValidationError> checkSchedule(ImportOptions options) throws Exception {
        return RequestValidator.checkScheduleOptions(options);
    }

    protected Collection<ValidationError> check(ImportOptions options) throws Exception {
        
        // check the dataset can be retrieved
        Collection<ValidationError> errors = RequestValidator.validateForProductMetadata(options);

        // return if any error at this stage
        if(!errors.isEmpty())
            return errors;

        logger.debug("Product and datasets are set in the request:");
        logger.debug(options.toString());
        
        // check if the product exists
        DatasetBrowser datasetBrowser = new DatasetBrowser();
        CmemsProduct product = datasetBrowser.getProduct(options.getProduct());
        if(product==null)
            errors.add(new ValidationError("product", "No such product: " + options.getProduct()));

        // return if any error at this stage
        if(!errors.isEmpty())
            return errors;

        logger.debug("Product retrieved:");
        logger.debug(product.toString());
        
        // fill the request with available defaults
        RequestValidator.applyDefaults(options, product);

        logger.debug("Defaults applied. Here is the product:");
        logger.debug(product.toString());

        // check if motu, if given, is correct
        if(!RequestValidator.isEmpty(options.getMotu())) {
            if(!options.getMotu().equals(product.getMotuServer())) {
                errors.add(new ValidationError("motu", "Invalid motu server for product: " + options.getProduct()));
            }
        }
        
        // return if any error at this stage
        if(!errors.isEmpty())
            return errors;

        logger.debug("Motu is OK:");
        logger.debug(options.toString());

        // check if the dataset exists
        boolean found = false;
        for(CmemsDataset dataset: product.getTdsDatasets()) {
            if(dataset.getName().equals(options.getDataset())) {
                found=true;
                break;
            }
        }
        if(!found) {
            errors.add(new ValidationError("dataset", "No such dataset: " + options.getDataset()));
        }

        // return if any error at this stage
        if(!errors.isEmpty())
            return errors;

        logger.debug("Dataset found in CMEMS Online Catalogue.");

        // retrive the motu dataset
        ProductMetadataInfo dataset = datasetBrowser.getMotuDataset(options.getProduct(), options.getDataset());

//        logger.debug("Retrieved dataset from Motu:");
//        logger.debug(dataset.toString());
        
        // apply defaults from motu
        logger.debug("Filling missing values...");
        RequestValidator.applyDefaults(options, dataset);

        logger.debug("Filling missing values with metadata from motu:");
        logger.debug(options.toString());

        // check the submission parameters are compabible with the datasets
        errors = RequestValidator.validate(options, dataset);

        logger.debug("Request validated. Errors found: " + errors.size());

        return errors;
    }
}
