package org.gcube.dataanalysis.copernicus.motu.model;

import java.text.ParseException;
import java.util.Calendar;
import java.util.Collection;
import java.util.List;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.Vector;

import javax.xml.bind.annotation.XmlAttribute;
import javax.xml.bind.annotation.XmlElement;
import javax.xml.bind.annotation.XmlElementWrapper;
import javax.xml.bind.annotation.XmlRootElement;

import org.gcube.dataanalysis.datasetimporter.util.TimeUtil;

@XmlRootElement(name = "productMetadataInfo")
public class ProductMetadataInfo {

    private String code;
    private String msg;
    private String lastUpdate;
    private String url;
    private String title;
    private String id;
    private String rawAvailableTimeCodes;
    private String rawAvailableDepths;
    private TimeCoverage timeCoverage;
    private Collection<Variable> variables;
    private Collection<Axis> dataGeospatialCoverage;

    @XmlAttribute
    public String getCode() {
        return code;
    }

    public void setCode(String code) {
        this.code = code;
    }

    @XmlAttribute
    public String getMsg() {
        return msg;
    }

    public void setMsg(String msg) {
        this.msg = msg;
    }

    @XmlAttribute
    public String getLastUpdate() {
        return lastUpdate;
    }

    public void setLastUpdate(String lastUpdate) {
        this.lastUpdate = lastUpdate;
    }

    @XmlAttribute
    public String getUrl() {
        return url;
    }

    public void setUrl(String url) {
        this.url = url;
    }

    @XmlAttribute
    public String getTitle() {
        return title;
    }

    public void setTitle(String title) {
        this.title = title;
    }

    @XmlAttribute
    public String getId() {
        return id;
    }

    public void setId(String id) {
        this.id = id;
    }

    @XmlElement(name="availableTimes")
    public String getRawAvailableTimeCodes() {
        return rawAvailableTimeCodes;
    }
    
    public List<Calendar> getAvailableTimeCodes() {
        SortedSet<Calendar> out = new TreeSet<>();
        String codes = this.getRawAvailableTimeCodes();
        if (codes != null) {
            for (String timeCode : codes.split(";")) {
                try {
                    Calendar c = TimeUtil.toCalendar(timeCode);
                    out.add(c);
                } catch (ParseException e) {
                    e.printStackTrace();
                }
            }
        }
        return new Vector<>(out);
    }
    
    public List<Calendar> getTimeCodesAfter(Calendar from) {
       List<Calendar> out = new Vector<>();
       for(Calendar c:this.getAvailableTimeCodes()) {
           if(c.after(from))
               out.add(c);
       }
       return out;
    }

    public List<Calendar> getTimeCodesBefore(Calendar to) {
        List<Calendar> out = new Vector<>();
        for(Calendar c:this.getAvailableTimeCodes()) {
            if(c.before(to))
                out.add(c);
        }
        return out;
     }
    
    public List<Calendar> getTimeCodesInInterval(Calendar from, Calendar to) {
        List<Calendar> out = this.getTimeCodesAfter(from);
        out.retainAll(this.getTimeCodesBefore(to));
        return out;
    }

    public Calendar getFirstAvailableTimeCode() {
        List<Calendar> timeCodes = this.getAvailableTimeCodes();
        if (timeCodes.size() > 0) {
            return timeCodes.get(0);
        } else {
            return null;
        }
    }

    public Calendar getLastAvailableTimeCode() {
        List<Calendar> timeCodes = this.getAvailableTimeCodes();
        if (timeCodes.size() > 0) {
            return timeCodes.get(timeCodes.size() - 1);
        } else {
            return null;
        }
    }

    /**
     * Return the time resolution in hours.
     * @return the time resolution in hours
     */
    public Long getTimeResolution() {
        final double millisInHour = 1000d * 60 * 60;
        Calendar from = this.getFirstAvailableTimeCode();
        Calendar to = this.getLastAvailableTimeCode();
        Integer howmany = this.getAvailableTimeCodes().size() - 1;
        Double elapsed = ((to.getTimeInMillis() - from.getTimeInMillis())
                / millisInHour);
        return Math.round(elapsed / howmany);
    }
    
    public void setRawAvailableTimeCodes(String rawAvailableTimeCodes) {
        this.rawAvailableTimeCodes = rawAvailableTimeCodes;
    }

    @XmlElement(name="availableDepths")
    public String getRawAvailableDepths() {
        return rawAvailableDepths;
    }

    public List<Double> getAvailableDepths() {
        SortedSet<Double> out = new TreeSet<>();
        String depths = this.getRawAvailableDepths();
        if (depths != null) {
            for (String depth : depths.split(";")) {
                try {
                    Double d = Double.parseDouble(depth);
                    out.add(d);
                } catch (NumberFormatException e) {
                    if(depth.equalsIgnoreCase("surface"))
                        out.add(0d);
                    else
                        e.printStackTrace();
                }
            }
        }
        return new Vector<>(out);
    }

    public Double getFirstAvailableDepth() {
        List<Double> depths = this.getAvailableDepths();
        if (depths.size() > 0) {
            return depths.get(0);
        } else {
            return null;
        }
    }

    public Double getLastAvailableDepth() {
        List<Double> depths = this.getAvailableDepths();
        if (depths.size() > 0) {
            return depths.get(depths.size() - 1);
        } else {
            return null;
        }
    }
    
    public void setRawAvailableDepths(String rawAvailableDepths) {
        this.rawAvailableDepths = rawAvailableDepths;
    }

    @XmlElement
    public TimeCoverage getTimeCoverage() {
        return timeCoverage;
    }

    public void setTimeCoverage(TimeCoverage timeCoverage) {
        this.timeCoverage = timeCoverage;
    }

    @XmlElement(name = "variable")
    @XmlElementWrapper(name = "variables")
    public Collection<Variable> getVariables() {
        return variables;
    }

    public void setVariables(Collection<Variable> variables) {
        this.variables = variables;
    }

    @XmlElement(name = "axis")
    @XmlElementWrapper(name = "dataGeospatialCoverage")
    public Collection<Axis> getDataGeospatialCoverage() {
        return dataGeospatialCoverage;
    }

    public void setDataGeospatialCoverage(
            Collection<Axis> dataGeospatialCoverage) {
        this.dataGeospatialCoverage = dataGeospatialCoverage;
    }
    
    public boolean canBeMergedWith(ProductMetadataInfo other) {
        // conditions for merge: everything 'similar' except variables
        // same time codes
        if(!this.getRawAvailableTimeCodes().equals(other.getRawAvailableTimeCodes()))
            return false;
        // same area
        // same depth (?)
        return true;
    }
    
}
