package org.gcube.datatransfer.agent.impl.db;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.util.List;
import java.util.Properties;
import java.util.Set;

import javax.jdo.JDOHelper;
import javax.jdo.PersistenceManager;
import javax.jdo.Query;

import org.gcube.common.core.contexts.GHNContext;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.datatransfer.agent.impl.context.ServiceContext;
import org.gcube.datatransfer.agent.impl.grs.GRSOutComeWriter;
import org.gcube.datatransfer.agent.impl.jdo.Transfer;
import org.gcube.datatransfer.agent.impl.jdo.TransferObject;
import org.gcube.datatransfer.common.outcome.TransferStatus;
import org.gcube.datatransfer.agent.stubs.datatransferagent.MonitorTransferReportMessage;

/**
 * 
 * @author Andrea Manzi(CERN)
 *
 */
public class DataTransferDBManager  extends DBManager implements Runnable {

	GCUBELog logger = new GCUBELog(DataTransferDBManager.class);

	/**
	 * Default constructor
	 */
	public DataTransferDBManager() throws Exception{

		dbFileBaseFolder =(String)ServiceContext.getContext().getPersistenceRoot().getAbsolutePath() + File.separator + "DataTransferDB"; 
		dbName = "data_transfer";
		dbFileName =  dbFileBaseFolder +File.separator+ dbName+".db";

		backupFolder = new File ((System.getenv("HOME") + File.separator + "DataTransferDBBackup"));

		Properties prop = new Properties();
		
		//getting datanucleus conf file from JNDI
		String propFile = GHNContext.getContext().getLocation() + File.separator +
				ServiceContext.getContext().getProperty("configDir", true) + File.separator + 
				ServiceContext.getContext().getDbConfigurationFileName();
			
		try {
			prop.load(new FileInputStream(new File(propFile)));
		} catch (IOException e) {
			e.printStackTrace();

		}

		persistenceFactory =JDOHelper.getPersistenceManagerFactory(prop);

		Thread t = new Thread(this);
		t.start();

	}

	/**
	 * 
	 * @param Transfer
	 * @param transferObjects
	 * @throws Exception
	 */
	public void storeTransfer(Transfer transfer) throws Exception{

		//storing transfer Main object
		PersistenceManager persistenceManager = getPersistenceManager();

		try
		{
			persistenceManager.currentTransaction().begin();
			persistenceManager.makePersistent(transfer);
			persistenceManager.currentTransaction().commit();

		}catch (Exception e)
		{
			e.printStackTrace();	
			throw e;
		}
		finally
		{
			if (persistenceManager.currentTransaction().isActive()) 
				persistenceManager.currentTransaction().rollback();
			persistenceManager.close();
		}



	}

	public void storeTransferObject (Set<TransferObject> transferObjects) throws Exception{

		PersistenceManager persistenceManager = getPersistenceManager();

		try {

			for (TransferObject obj : transferObjects){
				persistenceManager.currentTransaction().begin();
				persistenceManager.makePersistent(obj);
				persistenceManager.currentTransaction().commit();
			}
		}
		catch (Exception e){
			e.printStackTrace();
			throw e;
		}
		finally
		{
			if (persistenceManager.currentTransaction().isActive()) 
				persistenceManager.currentTransaction().rollback();
			persistenceManager.close();
		}

	}

	public String getTransferStatus(String transferId) throws Exception {

		PersistenceManager persistenceManager = getPersistenceManager();

		Transfer t = (Transfer) persistenceManager.getObjectById(Transfer.class,transferId);
		persistenceManager.close();

		return t.getStatus();
	}


	public void updateTransferObjectStatus(String objId,String status) throws Exception{

		PersistenceManager persistenceManager= getPersistenceManager();
		Transfer t = (Transfer) persistenceManager.getObjectById(Transfer.class,objId);
		logger.debug("Old status " + t.getStatus());
		t.setStatus(status);
		logger.debug("New Status " + t.getStatus());
		persistenceManager.makePersistent(t);
		persistenceManager.close();
	}
	

	public void updateTransferObjectInfo(String objId,long size) throws Exception{

		PersistenceManager persistenceManager= getPersistenceManager();
		Transfer t = (Transfer) persistenceManager.getObjectById(Transfer.class,objId);
		//logger.debug("Old byteTransferred" + t.getSizeTransferred());
		t.setSizeTransferred(t.getSizeTransferred()+size);
		//t.setTransfersCompleted(t.getTransfersCompleted()+1);
		//logger.debug("New byteTransferred" + t.getSizeTransferred());
		persistenceManager.makePersistent(t);
		persistenceManager.close();
	}
	
	public void addTransferObjectCompleted(String objId) throws Exception{

		PersistenceManager persistenceManager= getPersistenceManager();
		Transfer t = (Transfer) persistenceManager.getObjectById(Transfer.class,objId);
		logger.debug("Old completed" + t.getTransfersCompleted());
		t.setTransfersCompleted(t.getTransfersCompleted()+1);
		logger.debug("New completed" + t.getTransfersCompleted());
		persistenceManager.makePersistent(t);
		persistenceManager.close();
	}
	
	public void updateTransferJDO(String transferId, String[] inputURIs, long totalsize) throws Exception{
		PersistenceManager persistenceManager= getPersistenceManager();
		Transfer t = (Transfer) persistenceManager.getObjectById(Transfer.class,transferId);
		
		t.setId(transferId);
		t.setStatus(TransferStatus.STARTED.name());
		t.setSubmitter("N/A");
		t.setTransfersCompleted(0);
		t.setTotalSize(totalsize);
		t.setTransfersCompleted(0);
		t.setTotalTransfers(inputURIs.length);
		persistenceManager.makePersistent(t);
		persistenceManager.close();
	}
	

	/**
	 * (non-Javadoc)
	 * @see java.lang.Runnable#run()
	 */
	public void run() {
		do {
			try {
				Thread.sleep(backupIntervalMS);
				this.backup();
			} catch (InterruptedException e) {
				logger.error("Unable to sleep", e);
			} catch (Exception e) {
				logger.error("Unable to backup", e);
			}
		} while (! Thread.interrupted());

	}

	public PersistenceManager getPersistenceManager(){
		return persistenceFactory.getPersistenceManager();
	}

	public String getTransferObjectOutComeAsRS(String transferId) throws Exception {

		GRSOutComeWriter outcomeWriter = null;

		PersistenceManager persistenceManager= getPersistenceManager();

		Transfer t = (Transfer) persistenceManager.getObjectById(Transfer.class,transferId);

		if (!(TransferStatus.valueOf(t.getStatus()).hasCompleted())){
			outcomeWriter = new GRSOutComeWriter(1);
			outcomeWriter.putField(transferId,"",new Long(0),new Exception("The submitted transfer is not yet completed"));
		}
		else
		{
			Query query = persistenceManager.newQuery(TransferObject.class);
			query.setFilter("transferID == \""+transferId+"\"");

			List<TransferObject> list = (List<TransferObject>)query.execute();
			
			if (list == null || list.size()==0) 
				throw new Exception("The Transfer Objects list is empty");
			
			logger.debug("Getting " + list.size() +" outcomes");
			outcomeWriter = new GRSOutComeWriter(list.size());
			for (TransferObject obj :list){
				if (obj.getStatus().compareTo(TransferStatus.FAILED.name())==0)
					outcomeWriter.putField(obj.getSourceURI(),obj.getDestURI(),obj.getTransferTime(),new Exception (obj.getOutcome()));
				else outcomeWriter.putField(obj.getSourceURI(),obj.getDestURI(),obj.getTransferTime());
				persistenceManager.deletePersistent(obj);
			}
		
		}

		persistenceManager.close();
		return outcomeWriter.writer.getLocator().toString();
	}

	public MonitorTransferReportMessage getTrasferProgress(String transferId) {
		MonitorTransferReportMessage message = new MonitorTransferReportMessage();
		PersistenceManager persistenceManager= getPersistenceManager();
		Transfer t = (Transfer) persistenceManager.getObjectById(Transfer.class,transferId);
		message.setTransferID(transferId);
		message.setTotalBytes(t.getTotalSize());
		message.setTotalTransfers((int)t.getTotaltransfers());
		message.setTransferCompleted((int)t.getTransfersCompleted());
		message.setBytesTransferred(t.getSizeTransferred());
		message.setTransferStatus(t.getStatus());
		persistenceManager.close();
		return message;
	}
} 
