package org.gcube.data.spd.asfis.capabilities;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.data.spd.asfis.AsfisPlugin;
import org.gcube.data.spd.asfis.Utils;
import org.gcube.data.spd.asfis.dbconnection.ConnectionPool;
import org.gcube.data.spd.model.Condition;
import org.gcube.data.spd.model.Conditions;
import org.gcube.data.spd.model.exceptions.ExternalRepositoryException;
import org.gcube.data.spd.model.exceptions.IdNotValidException;
import org.gcube.data.spd.model.exceptions.StreamBlockingException;
import org.gcube.data.spd.model.products.TaxonomyItem;
import org.gcube.data.spd.model.products.TaxonomyStatus;
import org.gcube.data.spd.model.products.TaxonomyStatus.Status;
import org.gcube.data.spd.plugin.fwk.capabilities.ClassificationCapability;
import org.gcube.data.spd.plugin.fwk.writers.ClosableWriter;
import org.gcube.data.spd.plugin.fwk.writers.ObjectWriter;


public class ClassificationCapabilityImpl extends ClassificationCapability {

	GCUBELog logger = new GCUBELog(ClassificationCapabilityImpl.class);

	@Override
	public Set<Conditions> getSupportedProperties() {
		return Collections.emptySet();
	}

	@Override
	public void retrieveTaxonByIds(Iterator<String> ids,
			ClosableWriter<TaxonomyItem> writer) {
		try{
			while(ids.hasNext()) {
				String id = ids.next(); 
				TaxonomyItem item = null;
				try {
					item = retrieveTaxonById(id);
					if (item!=null){
						if (writer.isAlive() && (item!=null))
							writer.write(item);
						else
							break;
					}
				} catch (IdNotValidException e) {
					logger.error("Id Not Valid", e);
				}
			}
		}catch (Exception e) {
			writer.write(new StreamBlockingException("BrazilianFlora", ""));					
		}finally{
			writer.close();
		}
	}


	@Override
	public TaxonomyItem retrieveTaxonById(String id)
			throws IdNotValidException {

		ConnectionPool pool = null;
		ResultSet results = null;
		TaxonomyItem tax = null;

		try {
			pool = ConnectionPool.getConnectionPool();

			String query ="select Scientific_name, Author, Family, Order_rank, English_name, French_name, Spanish_name from "+ AsfisPlugin.table + " where TAXOCODE = ?";			

			results =  pool.selectPrestatement(query, id);	

			if(results.next()) {	

				String scientific_name = results.getString(1);
				String author = results.getString(2);
				String family = results.getString(3);
				String order = results.getString(4);
				//common names
				String englishName = results.getString(5);
				String frenchName = results.getString(6);
				String spanishName = results.getString(7);

				tax = createTaxonomyItem(id, scientific_name, author, family, order, englishName, frenchName, spanishName);

			}				
		}
		catch (Throwable e) {
			logger.error("Id not valid exception", e);

		}finally{
			if (results!=null){
				try {
					results.close();
				} catch (SQLException e) {
					logger.error("general Error", e);
				}	
			}
		}
		return tax;

	}


	@Override
	public List<TaxonomyItem> retrieveTaxonChildrenByTaxonId(String idParent)
			throws IdNotValidException, ExternalRepositoryException {
		
		logger.info("retrieveTaxonChildrenByTaxonId " + idParent);
		List<TaxonomyItem> list = null;
		ResultSet results = null;
		Boolean kingdom = false;
		Boolean order = false;
		try {
			//check if it's a kingdom
			if (idParent.equals(AsfisPlugin.ANIMALIA)){
				results = getOrders();
				kingdom = true;
				logger.info(idParent + " is kingdom");
			}
			//check if it's a species
			else 
				results = createResultItemChilds(idParent);

			//check if it's an order
			if (!results.next()) {
				results = getFamilies(idParent);
				if (results.next()){
					order = true;
					logger.info(idParent +" is order");
				}
			} 

			list = new ArrayList<TaxonomyItem>(); 
			do {
				TaxonomyItem tax = null;
				if ((kingdom) & (!results.getString(1).equals(0))){
					tax = createBaseTaxItem(results.getString(1), AsfisPlugin.ORDER);

					//set rank

					TaxonomyItem kingdomTaxon = createRankTaxItem(AsfisPlugin.ANIMALIA, AsfisPlugin.ANIMALIA, null);

					tax.setParent(kingdomTaxon);	
				}
				else if ((order) & (!results.getString(1).equals(""))){

					tax = createBaseTaxItem(results.getString(1), AsfisPlugin.FAMILY);

					//set rank

					TaxonomyItem kingdomTaxon = createRankTaxItem(AsfisPlugin.ANIMALIA, AsfisPlugin.ANIMALIA, null);

					TaxonomyItem familyTaxon = createRankTaxItem(idParent, AsfisPlugin.FAMILY, kingdomTaxon);		

					tax.setParent(familyTaxon);	
				}
				else if (!(kingdom) & (!order)){
					//					TAXOCODE, Scientific_name, Author, Family, Order_rank 
					String id = results.getString(1);
					String scientific_name = results.getString(2);
					String author = results.getString(3);
					String family = results.getString(4);
					String order_rank = results.getString(5);
					//common names
					String englishName = results.getString(6);
					String frenchName = results.getString(7);
					String spanishName = results.getString(8);

					tax = createTaxonomyItem(id, scientific_name, author, family, order_rank, englishName, frenchName, spanishName);

				}
				if (tax!=null)
					list.add(tax);
			} while (results.next());


		} catch (SQLException e) {
			logger.error("sql Error", e);
		}finally{
			if (results!=null){
				try {
					results.close();
				} catch (SQLException e) {
					logger.error("general Error", e);
				}
			}
		}
		return list;	
	}


	//set base information to taxonomy item
	private TaxonomyItem createBaseTaxItem(String result, String rank) {
		TaxonomyItem tax = new TaxonomyItem(result);
		tax.setScientificName(result);
		tax.setRank(rank);
		tax.setStatus(new TaxonomyStatus(AsfisPlugin.VALIDATED, Status.VALID));

		tax.setCitation(Utils.createCitation());
		tax.setCredits(Utils.createCredits());
		return tax;

	}

	/**
	 * Create Taxonomy Item
	 * @param spanishName 
	 * @param frenchName 
	 * @param englishName 
	 * @param order 
	 * @param family 
	 * @param author 
	 * @param scientific_name 
	 * @param id 
	 */

	private TaxonomyItem createTaxonomyItem(String id, String scientific_name, String author, String family, String order, String englishName, String frenchName, String spanishName) throws SQLException {
		TaxonomyItem item = null;
		try{

			item = new TaxonomyItem(id);
			item.setScientificName(scientific_name);
			item.setRank(AsfisPlugin.DEFAULTRANK);
			item.setAuthor(author);
			item.setStatus(new TaxonomyStatus(AsfisPlugin.VALIDATED, Status.VALID));

			//set common names
			item.setCommonNames(Utils.setCommonNames(englishName, frenchName, spanishName));

			item.setCitation(Utils.createCitation());
			item.setCredits(Utils.createCredits());

			//set rank

			TaxonomyItem kingdomTaxon = createRankTaxItem(AsfisPlugin.ANIMALIA, AsfisPlugin.ANIMALIA, null);

			TaxonomyItem familyTaxon = createRankTaxItem(family, AsfisPlugin.FAMILY, kingdomTaxon);		

			TaxonomyItem orderTaxon = createRankTaxItem(order, AsfisPlugin.ORDER, familyTaxon);

			item.setParent(orderTaxon);	

		}catch (Exception e) {
			logger.error("ID not valid Exception", e);
		}

		return item;
	}


	//create rank taxon Item
	private TaxonomyItem createRankTaxItem(String name, String rank, TaxonomyItem parent) {
		TaxonomyItem rankTaxon = new TaxonomyItem(name);
		rankTaxon.setScientificName(name);
		rankTaxon.setRank(rank);
		rankTaxon.setParent(parent);
		return rankTaxon;

	}


	/**
	 * Return orders
	 */
	private ResultSet getOrders() {

		ConnectionPool pool = null;
		ResultSet results = null;
		try {
			pool = ConnectionPool.getConnectionPool();
			String query = "select distinct(order_rank) from "+ AsfisPlugin.table;
			results =  pool.selectPrestatement(query, null);
		}
		catch (Throwable e) {
			logger.error("general Error", e);
		}
		return results;

	}

	//get families by order
	private ResultSet getFamilies(String order) {
		ConnectionPool pool = null;
		ResultSet results = null;
		try {
			pool = ConnectionPool.getConnectionPool();

			String term = "%" + order + "%";

			String query = "select distinct(family) from "+ AsfisPlugin.table + " where UPPER(order_rank) like UPPER(?)";
			results =  pool.selectPrestatement(query, term);
		}
		catch (Throwable e) {
			logger.error("general Error", e);
		}
		return results;
	}

	/**
	 * Return a ResultSet of Children
	 */
	private ResultSet createResultItemChilds(String id) {

		ConnectionPool pool = null;
		ResultSet results = null;
		try {
			pool = ConnectionPool.getConnectionPool();
			String query = "select TAXOCODE, Scientific_name, Author, Family, Order_rank, English_name, French_name, Spanish_name from "+ AsfisPlugin.table + " where family = ?";
			results =  pool.selectPrestatement(query, id);
		}
		catch (Throwable e) {
			logger.error("general Error", e);
		}
		return results;

	}

	@Override
	public void searchByScientificName(String name,
			ObjectWriter<TaxonomyItem> writer, Condition... properties) {
		logger.trace("Retrive taxa by name " + name);

		ResultSet results = null;
		try {
			results = Utils.createResultSetByName(name);
			if (results!=null){
				while(results.next()) {	

					String id = results.getString(1);
					String scientific_name = results.getString(2);
					String author = results.getString(3);
					String family = results.getString(4);
					String order = results.getString(5);
					//common names
					String englishName = results.getString(6);
					String frenchName = results.getString(7);
					String spanishName = results.getString(8);

					TaxonomyItem tax = createTaxonomyItem(id, scientific_name, author,family, order, englishName, frenchName, spanishName);
					if (writer.isAlive() && (tax!=null))
						writer.write(tax);
					else
						break;
				}
			}

		}catch (Exception e) {
			writer.write(new StreamBlockingException("AsfisPlugin",""));
		}finally{

			if (results!=null){
				try {
					results.close();
				} catch (SQLException e) {
					logger.error("general Error", e);
				}	
			}
		}

	}

}
