package org.gcube.data.publishing.gis.geoserver.db;

import java.io.Serializable;
import java.sql.Connection;
import java.sql.Date;
import java.sql.DriverManager;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Time;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.Map.Entry;

import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.data.publishing.gis.geoserver.Utils;
import org.gcube.data.publishing.gis.publisher.plugin.fwk.model.GeometryPoint;
import org.postgis.PGgeometry;

public class DBUtils {



	private static final GCUBELog logger=new GCUBELog(DBUtils.class);

	//	private static final String csquareCodeField="csquarecode";

	private static final String GEOM_FIELD="the_geom";
	private static final String GID_FIELD="gid";


	public static Connection getConnection(String endpoint, String user, String password)throws SQLException, ClassNotFoundException{
		Connection toReturn=DriverManager.getConnection(endpoint,user,password);
		toReturn.setAutoCommit(false);
		((org.postgresql.PGConnection)toReturn).addDataType("geometry",Class.forName("org.postgis.PGgeometry"));
		return toReturn;
	}

	public static PreparedStatement prepareStatementForInsert(Connection conn,Table theTable,GeometryPoint point) throws SQLException{
		StringBuilder fieldsName=new StringBuilder("("+GEOM_FIELD+",");
		StringBuilder fieldsValues=new StringBuilder("(?,");
		for (String f: point.getAttributes().keySet()){
			if(!f.equalsIgnoreCase(GEOM_FIELD)&&!f.equalsIgnoreCase(GID_FIELD)){
				fieldsValues.append("?,");
				fieldsName.append(f+",");
			}
		}

		fieldsValues.deleteCharAt(fieldsValues.length()-1);
		fieldsValues.append(")");
		fieldsName.deleteCharAt(fieldsName.length()-1);
		fieldsName.append(")");

		String query="INSERT INTO "+theTable.getTableName()+" "+fieldsName+" VALUES "+fieldsValues;
		logger.debug("the prepared statement is :"+ query);
		PreparedStatement ps= conn.prepareStatement(query,Statement.NO_GENERATED_KEYS);
		return ps;
	}

	public static Table createTable(Connection conn,GeometryPoint point) throws SQLException{
		Table toReturn=new Table();

		toReturn.setTableName("t"+Utils.getUUID());

		StringBuilder createQuery=new StringBuilder("CREATE TABLE "+toReturn.getTableName()+" ( "+GID_FIELD+" "+getTypeDefinition(SQLType.SERIAL)+" NOT NULL,");
		createQuery.append(GEOM_FIELD+" "+getTypeDefinition(SQLType.GEOMETRY)+",");

		toReturn.addField(GID_FIELD, SQLType.SERIAL, false);
		toReturn.addField(GEOM_FIELD, SQLType.GEOMETRY,true);

		for(Entry<String,Serializable> field:point.getAttributes().entrySet()){
			if(!field.getKey().equalsIgnoreCase(GID_FIELD)&&!field.getKey().equalsIgnoreCase(GEOM_FIELD)){
				SQLType type=getSQLType(field.getValue());
				createQuery.append(" "+field.getKey()+" "+getTypeDefinition(type)+",");
				toReturn.addField(field.getKey(), type,true);
			}
		}
		createQuery.deleteCharAt(createQuery.lastIndexOf(","));
		createQuery.append(")");
		String query=createQuery.toString();
		Statement stmt=null;
		try{
			stmt=conn.createStatement();
			logger.debug("Goin to execute "+query);
			stmt.execute(query);
			return toReturn;
		}finally{
			if(stmt!=null)stmt.close();
		}
	}


	public static int insertPoint(Table theTable, PreparedStatement ps,GeometryPoint point) throws SQLException{
		for(int index=1;index<=theTable.getInsertQueryFields().size();index++){
			String fieldName=theTable.getInsertQueryFields().get(index-1);
			SQLType type=theTable.getFields().get(fieldName);
			if(fieldName.equals(GEOM_FIELD))
				ps.setObject(index, point.getTheGeometry());
			else if(!fieldName.equalsIgnoreCase(GID_FIELD)){ //gid is auto generated
				Object value=point.getAttributes().get(fieldName);
				if(value==null)ps.setNull(index, getType(type));
				switch(type){
				case BOOLEAN : ps.setBoolean(index, (Boolean) value);
				break;
				case DATE : ps.setDate(index, (Date) value);
				break;
				case FLOAT : ps.setFloat(index, (Float) value);
				break;
				case GEOMETRY : ps.setObject(index, value);
				break;
				case INTEGER : ps.setInt(index, (Integer) value);
				break;
				case LONG : ps.setLong(index, (Long) value);
				break;
				case TIME : ps.setTime(index, (Time) value);
				break;
				case TIMESTAMP : ps.setTimestamp(index, (Timestamp) value);
				break;
				default : ps.setString(index, value.toString());
				}
			}
		}
		return ps.executeUpdate();
	}



	//*************************** TYPES

	private static SQLType getSQLType(Object obj){
		if(obj.getClass().isAssignableFrom(Integer.class)) return SQLType.INTEGER;
		if(obj.getClass().isAssignableFrom(Long.class)) return SQLType.LONG;
		if(obj.getClass().isAssignableFrom(Float.class)) return SQLType.INTEGER;
		if(obj.getClass().isAssignableFrom(Date.class)||obj.getClass().isAssignableFrom(java.util.Date.class)) return SQLType.DATE;
		if(obj.getClass().isAssignableFrom(Time.class)) return SQLType.TIME;
		if(obj.getClass().isAssignableFrom(Timestamp.class)) return SQLType.TIMESTAMP;
		if(obj.getClass().isAssignableFrom(Boolean.class)) return SQLType.BOOLEAN;
		if(obj.getClass().isAssignableFrom(PGgeometry.class))return SQLType.GEOMETRY;
		return SQLType.TEXT;
	}

	private static int getType(SQLType type){
		switch(type){
		case BOOLEAN : return Types.BOOLEAN;
		case DATE : return Types.DATE;
		case FLOAT : return Types.DECIMAL;
		case GEOMETRY : return Types.OTHER;
		case INTEGER : return Types.INTEGER;
		case LONG : return Types.BIGINT;
		case STRING : return Types.VARCHAR;
		case TEXT : return Types.LONGNVARCHAR;
		case TIME : return Types.TIME;
		case TIMESTAMP : return Types.TIMESTAMP;
		case SERIAL : return Types.BIGINT;
		default : return Types.NULL;
		}
	}


	private static String getTypeDefinition(SQLType type){
		switch(type){
		case BOOLEAN : return "boolean";
		case DATE : return "date";
		case FLOAT : return "decimal";
		case GEOMETRY : return "geometry";
		case INTEGER : return "integer";
		case LONG : return "bigint";
		case STRING : return "varchar (200)";
		case TEXT : return "text";
		case TIME : return "time";
		case TIMESTAMP : return "timestamp";
		case SERIAL : return "serial";
		default : return "";
		}
	}

}
