package org.gcube.data.harmonization.occurrence.queue;

import java.io.IOException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;

import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.data.harmonization.occurrence.services.ServiceContext;

import com.rabbitmq.client.Channel;
import com.rabbitmq.client.QueueingConsumer;

public abstract class QueueReceiver extends Thread{

	private static GCUBELog logger = new GCUBELog(QueueReceiver.class);

	private static final boolean AUTO_ACK=false;
	private static final int PREFETCH_COUNT=1;
	
	
	protected AtomicBoolean continueListening = new AtomicBoolean(true);
	protected AtomicInteger readCount=new AtomicInteger(0);
	protected AtomicInteger errorCount=new AtomicInteger(0);

	private Channel channel;
	private String routingKey;

	protected String currentMessage;

	
	public QueueReceiver() {
		super();		
	}

	
	public void bind(Channel channel, String routingKey)throws IOException {
		this.channel = channel;
		this.routingKey = routingKey;
		this.channel.exchangeDeclare(getExchangeName(), "topic");
		this.channel.queueDeclare(getExchangeName(), true, false, false, null);		
		this.channel.queueBind(getExchangeName(), getExchangeName(), this.routingKey);		
		this.channel.basicQos(PREFETCH_COUNT);
	}
	
	
	public String getExchangeName(){
		return ServiceContext.getContext().getName();
	}

	public void run(){
		logger.debug("Started listening to "+routingKey);
		QueueingConsumer consumer = new QueueingConsumer(channel);
		try{
			channel.basicConsume(getExchangeName(), AUTO_ACK, consumer);
			while(continueListening.get()){
				try{
					QueueingConsumer.Delivery delivery=consumer.nextDelivery();				
					currentMessage=new String(delivery.getBody());
					handleMessage();
					channel.basicAck(delivery.getEnvelope().getDeliveryTag(), false);				
				}catch(Exception e){
					rollback();
					errorCount.incrementAndGet();
				}finally{
					cleanup();
					readCount.incrementAndGet();
				}
			}
		}catch(Exception e){
			logger.fatal("Unable to consume messages from queue ",e);
		}
	}


	public void stopListening(){
		continueListening.set(false);
	}

	public int getComputedCount(){
		return readCount.intValue();
	}

	public abstract void handleMessage()throws Exception;
	public abstract void rollback()throws Exception;
	public abstract void cleanup();
}
