package org.gcube.data.fishfinder.tmplugin.repository;

import static org.gcube.data.streams.dsl.Streams.pipe;

import java.io.IOException;
import java.io.ObjectOutputStream;
import java.util.Calendar;

import org.gcube.data.fishfinder.tmplugin.FishFinderPlugin;
import org.gcube.data.fishfinder.tmplugin.dbconnection.FishFinderTable;
import org.gcube.data.fishfinder.tmplugin.repository.iterators.FishFinderRecord;
import org.gcube.data.fishfinder.tmplugin.repository.iterators.RecordIterator;
import org.gcube.data.fishfinder.tmplugin.requests.Request;
import org.gcube.data.fishfinder.tmplugin.utils.FishFinderTreeGenerator;
import org.gcube.data.fishfinder.tmplugin.utils.Utils;
import org.gcube.data.streams.Stream;
import org.gcube.data.streams.exceptions.StreamException;
import org.gcube.data.streams.generators.Generator;
import org.gcube.data.tmf.api.exceptions.UnknownTreeException;
import org.gcube.data.trees.data.Tree;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Default {@link Repository} implementation
 * 
 * @author "Valentina Marioli valentina.marioli@isti.cnr.it"
 *
 */
public class BaseRepository implements Repository{

	private static final long serialVersionUID = 1L;

	private static final Logger log = LoggerFactory.getLogger(BaseRepository.class);

	transient private String name;
	transient private String description;

	private FishFinderTreeGenerator binder;


	// sees repository as a whole, no set boundaries
	public BaseRepository(Request request) throws Exception {
		this.description = request.getDescription();
		this.name = request.getName();
		binder = new FishFinderTreeGenerator(request);
	}


	/**
	 * {@inheritDoc}
	 */
	@Override
	public String name() {
		return name;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String description() {
		return description;
	}



	/**
	 * {@inheritDoc}
	 */
	@Override
	public Tree get(String id) throws UnknownTreeException, Exception {
		log.trace("Get figis records by ID {}", id );
		//get result set
		FishFinderRecord record = null;
		try{

			FishFinderTable figisTable = new FishFinderTable();
			record = figisTable.getRecordByID(id);

			return binder.bind(record);

		}
		catch (Exception e) {
			throw new UnknownTreeException(id,e);
		}

	}

	@Override
	public Stream<Tree> getRecords() throws UnknownTreeException, Exception {

		log.trace("Get all figis records");
		Stream<FishFinderRecord> records = new RecordIterator();
		Generator<FishFinderRecord,Tree> parser = new Generator<FishFinderRecord,Tree>() {
			public Tree yield(FishFinderRecord record) {
				try {
					return binder.bind(record);
				}
				catch(Exception e) {
					throw new StreamException();
				}
			}
		};

		return pipe(records).through(parser);

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public Summary summary() throws Exception {

		log.trace("update summary");
		Calendar lastUpdate=null;
				
		try{
			lastUpdate = Utils.lastupdate();
		}catch(Exception e) {
			log.error("could not retrieve last update",e);
		}
		
		Stream<FishFinderRecord> records = new RecordIterator();
		long cardinality=0;
		while (records.hasNext()){
			try {
				records.next();
				cardinality++;				
			}
			catch(Exception e) {
				log.error("could not count record",e);
			}
		}

		return new Summary(lastUpdate, cardinality);		
	}


	private void writeObject(ObjectOutputStream out) throws IOException {
		out.defaultWriteObject();
	}


	// invoked upon deserialisation, resets non-serializable defaults
	private void readObject(java.io.ObjectInputStream in) throws IOException,
	ClassNotFoundException {

		in.defaultReadObject();

		// check invariants
		if ((FishFinderPlugin.jdbc==null) || (FishFinderPlugin.dbLogin==null) || (FishFinderPlugin.dbPassword==null) || (FishFinderPlugin.baseUrl==null) || (FishFinderPlugin.extractAllCsv==null) || (FishFinderPlugin.reinitialize==null))			
			Utils.getRuntimeResource();

	}



}
