/*
 * Decompiled with CFR 0.152.
 */
package com.terradue.gtuploader;

import com.ning.http.client.AsyncHandler;
import com.ning.http.client.AsyncHttpClient;
import com.ning.http.client.AsyncHttpClientConfig;
import com.ning.http.client.HttpResponseStatus;
import com.ning.http.client.Realm;
import com.ning.http.client.Request;
import com.ning.http.client.RequestBuilder;
import com.ning.http.client.filter.ResponseFilter;
import com.terradue.gtuploader.BoundingBox;
import com.terradue.gtuploader.Coverage;
import com.terradue.gtuploader.ErrorMessage;
import com.terradue.gtuploader.GeoTIFFUploadException;
import com.terradue.gtuploader.GetStatusAsyncHandler;
import com.terradue.gtuploader.JAXBEntityWriter;
import com.terradue.gtuploader.Layer;
import com.terradue.gtuploader.StatusResponseFilter;
import com.terradue.gtuploader.Store;
import com.terradue.gtuploader.Workspace;
import java.io.File;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Formatter;
import java.util.Hashtable;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import org.gdal.gdal.Dataset;
import org.gdal.gdal.gdal;
import org.gdal.gdalconst.gdalconstConstants;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/*
 * This class specifies class file version 49.0 but uses Java 6 signatures.  Assumed Java 6.
 */
public final class GeoTIFFUploader {
    private static final Logger log = LoggerFactory.getLogger(GeoTIFFUploader.class);
    private static final Pattern SDS_DESCRIPTION = Pattern.compile("\\[.+\\] (.+) \\(.+\\)");
    private static final String HEADING = "GeoTIFF upload errors:%n%n";
    private final AsyncHttpClient httpClient;
    private final Marshaller xmlMarshaller;
    private final String workspaceURL;
    private final String coverageURL;
    private final String uploadGeoTIFFCoverageURL;
    private final String layerURL;
    private final File tmpDir;

    private GeoTIFFUploader(String restServerUsername, String restServerPassword, String workspaceURL, String coverageURL, String uploadGeoTIFFCoverageURL, String layerURL, int requestTimeout, File tmpDir) {
        try {
            JAXBContext context = JAXBContext.newInstance((Class[])new Class[]{Workspace.class, Coverage.class, BoundingBox.class, Layer.class});
            this.xmlMarshaller = context.createMarshaller();
        }
        catch (JAXBException e) {
            throw new IllegalStateException("Impossible to inizialize the JAXB context - that should not happen", e);
        }
        this.httpClient = new AsyncHttpClient(new AsyncHttpClientConfig.Builder().setAllowPoolingConnection(true).setRequestTimeoutInMs(requestTimeout * 1000).addResponseFilter((ResponseFilter)new StatusResponseFilter()).setRealm(new Realm.RealmBuilder().setPrincipal(restServerUsername).setPassword(restServerPassword).setUsePreemptiveAuth(true).setScheme(Realm.AuthScheme.BASIC).build()).build());
        this.workspaceURL = workspaceURL;
        this.coverageURL = coverageURL;
        this.uploadGeoTIFFCoverageURL = uploadGeoTIFFCoverageURL;
        this.layerURL = layerURL;
        this.tmpDir = tmpDir;
        gdal.AllRegister();
    }

    public void convertAndUpload(File data) throws GeoTIFFUploadException {
        Hashtable subSetMetaData;
        if (data == null) {
            throw new IllegalArgumentException("Argument 'gdalData' must be not null.");
        }
        if (!data.exists()) {
            throw new GeoTIFFUploadException("File '%s' does not exist.", data);
        }
        if (data.isDirectory()) {
            throw new GeoTIFFUploadException("File '%s' cannot be converted, is a directory.", data);
        }
        if (log.isInfoEnabled()) {
            log.info("Performing GeoTIFF conversions running GDAL {}", (Object)gdal.VersionInfo());
        }
        String workspaceName = String.format("%s-%s", data.getName().substring(0, data.getName().lastIndexOf(46)), UUID.randomUUID());
        Dataset sourceDataset = gdal.Open((String)data.getAbsolutePath(), (int)gdalconstConstants.GA_ReadOnly);
        if (sourceDataset == null) {
            throw new GeoTIFFUploadException("File '%s' cannot be read - [%s] %s.", data, gdal.GetLastErrorNo(), gdal.GetLastErrorMsg());
        }
        if (log.isInfoEnabled()) {
            log.info("Checking if workspace {} needs to be created on Geoserver", (Object)workspaceName);
        }
        Workspace workspace = new Workspace();
        workspace.setName(workspaceName);
        if (!this.exists(MessageFormat.format(this.workspaceURL, workspaceName))) {
            if (log.isInfoEnabled()) {
                log.info("Workspace not already present on Geoserver, creating it...");
            }
            this.sendXML("POST", MessageFormat.format(this.workspaceURL, ""), workspace);
            if (log.isInfoEnabled()) {
                log.info("Done, workspace now available on Geoserver");
            }
        } else if (log.isInfoEnabled()) {
            log.info("Workspace already present on Geoserver");
        }
        if ((subSetMetaData = sourceDataset.GetMetadata_Dict("Subdatasets")).isEmpty()) {
            throw new GeoTIFFUploadException("File '%s' does not contain any sub data set", data);
        }
        int metadataSize = subSetMetaData.size() / 2;
        ArrayList<ErrorMessage> errors = new ArrayList<ErrorMessage>(metadataSize);
        File targetDir = new File(this.tmpDir, workspaceName);
        if (!targetDir.exists() && !targetDir.mkdirs()) {
            throw new GeoTIFFUploadException("Impossible to create %s directory, please verify current user has enough permissions", targetDir);
        }
        for (int i = 1; i <= metadataSize; ++i) {
            ErrorMessage error;
            ErrorMessage error2;
            String subDataSetName = (String)subSetMetaData.get(String.format("SUBDATASET_%d_NAME", i));
            String description = String.format("SUBDATASET_%d_DESC", i);
            Matcher matcher = SDS_DESCRIPTION.matcher((CharSequence)subSetMetaData.get(description));
            if (!matcher.matches()) {
                errors.add(new ErrorMessage("Sub data set description format %s not recognized", description));
                continue;
            }
            String subDataSetDesc = matcher.group(1).replace(' ', '_');
            if (log.isInfoEnabled()) {
                log.info("-- Processing sub meta data {} [{}/{}]", new Object[]{subDataSetDesc, i, metadataSize});
            }
            String targetFileName = String.format("%s.geotiff", subDataSetDesc);
            Dataset sourceSubDataset = gdal.Open((String)subDataSetName, (int)gdalconstConstants.GA_ReadOnly);
            File convertedFile = new File(targetDir, targetFileName);
            if (log.isInfoEnabled()) {
                log.info("Converting {} data into target file {}...", (Object)subDataSetName, (Object)convertedFile);
            }
            try {
                Dataset dump = gdal.GetDriverByName((String)"GTiff").CreateCopy(convertedFile.getAbsolutePath(), sourceSubDataset, 0, new String[]{"TILED=YES"});
                if (dump == null) {
                    error2 = new ErrorMessage("File '%s' cannot be produced, GDAL error #%s: %s", convertedFile, gdal.GetLastErrorNo(), gdal.GetLastErrorMsg());
                    log.warn(error2.toString());
                    errors.add(error2);
                    continue;
                }
                dump.delete();
            }
            catch (Throwable t) {
                error2 = new ErrorMessage("File '%s' cannot be produced, GDAL error #%s: %s", convertedFile, gdal.GetLastErrorNo(), gdal.GetLastErrorMsg());
                log.warn(error2.toString());
                errors.add(error2);
                continue;
            }
            if (log.isInfoEnabled()) {
                log.info("Done! Uploading converted geotiff on remote server...");
            }
            try {
                this.geoTiffPut(MessageFormat.format(this.uploadGeoTIFFCoverageURL, workspaceName, subDataSetDesc), convertedFile);
            }
            catch (GeoTIFFUploadException e) {
                error2 = new ErrorMessage("Converted data can not be uploated to remote server", e);
                log.warn(error2.toString());
                errors.add(error2);
                continue;
            }
            if (log.isInfoEnabled()) {
                log.info("Done! Creating Coverage...");
            }
            Dataset convertedSubDataset = gdal.Open((String)convertedFile.getAbsolutePath(), (int)gdalconstConstants.GA_ReadOnly);
            BoundingBox nativeBoundingBox = new BoundingBox();
            nativeBoundingBox.setMinX(0.0);
            nativeBoundingBox.setMinY(convertedSubDataset.getRasterYSize());
            nativeBoundingBox.setMaxX(convertedSubDataset.getRasterXSize());
            nativeBoundingBox.setMaxY(0.0);
            double[] adfGeoTransform = new double[6];
            convertedSubDataset.GetGeoTransform(adfGeoTransform);
            BoundingBox latLonBoundingBox = new BoundingBox();
            latLonBoundingBox.setMinX(GeoTIFFUploader.getGeoreferencedX(adfGeoTransform, 0.0, convertedSubDataset.getRasterYSize()));
            latLonBoundingBox.setMinY(GeoTIFFUploader.getGeoreferencedY(adfGeoTransform, 0.0, convertedSubDataset.getRasterYSize()));
            latLonBoundingBox.setMaxX(GeoTIFFUploader.getGeoreferencedX(adfGeoTransform, convertedSubDataset.getRasterXSize(), 0.0));
            latLonBoundingBox.setMaxY(GeoTIFFUploader.getGeoreferencedY(adfGeoTransform, convertedSubDataset.getRasterXSize(), 0.0));
            Coverage coverage = new Coverage();
            coverage.setName(subDataSetDesc);
            coverage.setWorkspace(workspaceName);
            coverage.setLatLonBoundingBox(latLonBoundingBox);
            Store store = new Store();
            store.setName(subDataSetDesc);
            coverage.setStore(store);
            try {
                this.sendXML("PUT", MessageFormat.format(this.coverageURL, workspaceName, subDataSetDesc), coverage);
            }
            catch (GeoTIFFUploadException e) {
                error = new ErrorMessage(String.format("Coverage for file %s can not be created", convertedFile), e);
                log.warn(error.toString());
                errors.add(error);
                continue;
            }
            if (log.isInfoEnabled()) {
                log.info("Done! Enabling layer...");
            }
            try {
                this.sendXML("PUT", MessageFormat.format(this.layerURL, workspaceName, subDataSetDesc), new Layer());
            }
            catch (GeoTIFFUploadException e) {
                error = new ErrorMessage(String.format("Coverage for file %s can not be created", convertedFile), e);
                log.warn(error.toString());
                errors.add(error);
                continue;
            }
            if (!log.isInfoEnabled()) continue;
            log.info("Done.");
        }
        if (log.isInfoEnabled()) {
            log.info("All data processed.");
        }
        if (!errors.isEmpty()) {
            Formatter fmt = new Formatter().format(HEADING, new Object[0]);
            int index = 1;
            for (ErrorMessage errorMessage : errors) {
                fmt.format("%s) %s%n", index++, errorMessage.getMessage());
                Throwable cause = errorMessage.getCause();
                if (cause != null) {
                    StringWriter writer = new StringWriter();
                    cause.printStackTrace(new PrintWriter(writer));
                    fmt.format("Caused by: %s", writer.getBuffer());
                }
                fmt.format("%n", new Object[0]);
            }
            if (errors.size() == 1) {
                fmt.format("1 error", new Object[0]);
            } else {
                fmt.format("skipped %s over %s sub data set", errors.size(), metadataSize);
            }
            throw new GeoTIFFUploadException(fmt.toString(), new Object[0]);
        }
    }

    private boolean exists(String url) throws GeoTIFFUploadException {
        HttpResponseStatus status = null;
        try {
            status = (HttpResponseStatus)this.httpClient.prepareGet(url).execute((AsyncHandler)new GetStatusAsyncHandler()).get();
        }
        catch (Exception e) {
            throw new GeoTIFFUploadException(String.format("Impossible to complete %s request, see nested exceptions.", url), e);
        }
        switch (status.getStatusCode()) {
            case 200: {
                return true;
            }
            case 404: {
                return false;
            }
        }
        throw new GeoTIFFUploadException("Impossible to complete %s request, server replied %s - %s", url, status.getStatusCode(), status.getStatusText());
    }

    private void geoTiffPut(String url, File toBePut) throws GeoTIFFUploadException {
        HttpResponseStatus status = null;
        try {
            status = (HttpResponseStatus)((AsyncHttpClient.BoundRequestBuilder)this.httpClient.preparePut(url).addHeader("Content-type", "image/tiff").setBody(toBePut)).execute((AsyncHandler)new GetStatusAsyncHandler()).get();
        }
        catch (Exception e) {
            throw new GeoTIFFUploadException(String.format("Impossible to complete %s request, see nested exceptions.", url), e);
        }
        if (201 != status.getStatusCode() && 200 != status.getStatusCode()) {
            throw new GeoTIFFUploadException("Impossible to complete %s request, server replied %s %s", url, status.getStatusCode(), status.getStatusText());
        }
    }

    private <T> void sendXML(String method, String url, T toBeSent) throws GeoTIFFUploadException {
        HttpResponseStatus status = null;
        try {
            status = (HttpResponseStatus)this.httpClient.executeRequest(new RequestBuilder(method).setUrl(url).addHeader("Content-type", "text/xml").setBody((Request.EntityWriter)new JAXBEntityWriter(this.xmlMarshaller, toBeSent)).build(), (AsyncHandler)new GetStatusAsyncHandler()).get();
        }
        catch (Exception e) {
            throw new GeoTIFFUploadException(String.format("Impossible to complete %s request, see nested exceptions.", url), e);
        }
        if (201 != status.getStatusCode() && 200 != status.getStatusCode()) {
            throw new GeoTIFFUploadException("Impossible to complete %s request, server replied %s %s", url, status.getStatusCode(), status.getStatusText());
        }
    }

    public void close() {
        gdal.GDALDestroyDriverManager();
    }

    private static double getGeoreferencedX(double[] adfGeoTransform, double x, double y) {
        return adfGeoTransform[0] + adfGeoTransform[1] * x + adfGeoTransform[2] * y;
    }

    private static double getGeoreferencedY(double[] adfGeoTransform, double x, double y) {
        return adfGeoTransform[3] + adfGeoTransform[4] * x + adfGeoTransform[5] * y;
    }

    public static class Builder {
        private static final String WORKSPACES = "http://%s:%s/%s/rest/workspaces/{0}";
        private static final String COVERAGE = "http://%s:%s/%s/rest/workspaces/{0}/coveragestores/{1}/coverages/{1}";
        private static final String UPLOAD_COVERAGE = "http://%s:%s/%s/rest/workspaces/{0}/coveragestores/{1}/file.geotiff?configure=first&coverageName={1}";
        private static final String LAYER = "http://%s:%s/%s/rest/layers/{0}:{1}";
        private String restServerHost;
        private int restServerPort;
        private int requestTimeout;
        private String restBasePath;
        private File tmpDir = new File(System.getProperty("java.io.tmpdir"));
        private String restServerUsername;
        private String restServerPassword;

        public Builder setRestServerHost(String restServerHost) {
            this.restServerHost = restServerHost;
            return this;
        }

        public Builder setRestServerPort(int restServerPort) {
            this.restServerPort = restServerPort;
            return this;
        }

        public Builder setRestBasePath(String restBasePath) {
            this.restBasePath = restBasePath;
            return this;
        }

        public Builder setRequestTimeout(int requestTimeout) {
            this.requestTimeout = requestTimeout;
            return this;
        }

        public Builder setRestServerUsername(String restServerUsername) {
            this.restServerUsername = restServerUsername;
            return this;
        }

        public Builder setRestServerPassword(String restServerPassword) {
            this.restServerPassword = restServerPassword;
            return this;
        }

        public Builder setTmpDir(File tmpDir) {
            this.tmpDir = tmpDir;
            return this;
        }

        public GeoTIFFUploader newGeoTIFFUploader() {
            return new GeoTIFFUploader(this.restServerUsername, this.restServerPassword, String.format(WORKSPACES, this.restServerHost, this.restServerPort, this.restBasePath), String.format(COVERAGE, this.restServerHost, this.restServerPort, this.restBasePath), String.format(UPLOAD_COVERAGE, this.restServerHost, this.restServerPort, this.restBasePath), String.format(LAYER, this.restServerHost, this.restServerPort, this.restBasePath), this.requestTimeout, this.tmpDir);
        }
    }
}

