package org.gcube.contentmanagement.timeseriesservice.impl.curation;

import java.sql.ResultSet;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import org.apache.axis.components.uuid.UUIDGen;
import org.apache.axis.components.uuid.UUIDGenFactory;
import org.apache.axis.message.addressing.EndpointReferenceType;
import org.gcube.common.core.faults.GCUBEFault;
import org.gcube.common.core.faults.GCUBEUnrecoverableFault;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.types.VOID;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.common.dbinterface.attributes.SimpleAttribute;
import org.gcube.common.dbinterface.persistence.ObjectPersistency;
import org.gcube.common.dbinterface.pool.DBSession;
import org.gcube.common.dbinterface.queries.Select;
import org.gcube.common.dbinterface.tables.Table;
import org.gcube.contentmanagement.codelistmanager.entities.CodeList;
import org.gcube.contentmanagement.timeseriesservice.impl.context.CurationContext;
import org.gcube.contentmanagement.timeseriesservice.impl.context.ServiceContext;
import org.gcube.contentmanagement.timeseriesservice.impl.curation.rules.Rule;
import org.gcube.contentmanagement.timeseriesservice.impl.curation.state.CurationResource;
import org.gcube.contentmanagement.timeseriesservice.impl.timeseries.operations.util.FilterExplorer;
import org.gcube.contentmanagement.timeseriesservice.impl.utils.Util;
import org.gcube.contentmanagement.timeseriesservice.stubs.CurationItem;
import org.gcube.contentmanagement.timeseriesservice.stubs.CurationItemsArray;
import org.gcube.contentmanagement.timeseriesservice.stubs.GetFieldValuesRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.GetItemRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.StartRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.StartResponse;
import org.gcube.contentmanagement.timeseriesservice.stubs.SaveRuleRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.RuleItem;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.RulesArray;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.StringArray;

/**
 * 
 * @author lucio
 *
 */
public class CurationFactory {

	private static final UUIDGen uuidGen = UUIDGenFactory.getUUIDGen();
	
	private GCUBELog logger = new GCUBELog(CurationFactory.class);
	
	
	/**
	 * 
	 * @param StartRequest request - the parameters to create the curation 
	 * @return StartResponse the Epr and the id of the new curation 
	 * @throws GCUBEFault -
	 */
	public StartResponse startCuration(StartRequest request) throws GCUBEFault{
		String id=uuidGen.nextUUID();
		StartResponse response;
		CurationResource resource=null;
		logger.debug("creating curation resource with id "+id);
		try{
			resource= (CurationResource) CurationContext.getPortTypeContext().getWSHome().create(
				CurationContext.getPortTypeContext().makeKey(id), id, request.getTitle(), request.getDescription(), request.getCreator(), request.getSource());
			
			response= new StartResponse(resource.getEPR(), resource.getId());
			
			logger.trace("new resource epr: "+resource.getEPR());
			resource.store();
		}catch(Exception re){ 
			if (resource!=null) resource.remove();
			logger.error("error creating curation resource", re);
			throw new GCUBEFault(re,"error creating curation resource");
		}
		return response; 
	}
	
	/**
	 * 
	 * @param curationId - the id of the curation
	 * @return the Epr of the open curation
	 * @throws GCUBEFault -
	 */
	public EndpointReferenceType open(String curationId) throws GCUBEFault{
		try{
			logger.debug("opening curation resource with id "+curationId);
			CurationResource resource= (CurationResource) CurationContext.getPortTypeContext().getWSHome().find(CurationContext.getPortTypeContext().makeKey(curationId));
			return resource.getEPR();
		}catch(Exception re){ 
			logger.error("error opening curation resource", re);
			throw new GCUBEFault(re,"error opening curation resource");
		}
	}
	
	
	/**
	 * return the possible values for that field in that dimension 
	 * 
	 * @param the request is the dimension id and the fieldName 
	 * @return all possible value for the choices dimension and the choices field name
	 */
	public StringArray getFieldValues(GetFieldValuesRequest request) throws GCUBEFault{
		StringArray toReturn= new StringArray();
		DBSession session=null;
		try {
			session= DBSession.connect();
			
			CodeList codelist = CodeList.get(request.getDimensionId());
			
			Select fieldValuesQuery= DBSession.getImplementation(Select.class);
			fieldValuesQuery.setAttributes(new SimpleAttribute(request.getFieldName()));
			fieldValuesQuery.setTables(new Table(codelist.getTable().getTableName()));
			ResultSet fieldValuesQueryRes= fieldValuesQuery.getResults(session);
			
			
			List<String> valuesList= new ArrayList<String>();
			
			while(fieldValuesQueryRes.next())
				valuesList.add(fieldValuesQueryRes.getString(1));
			toReturn.setStrings(valuesList.toArray(new String[0]));
		} catch (Exception e) {
			logger.error("failed getting values",e);
			throw new GCUBEFault(e);
		}finally{
			session.release();
		}
		return toReturn;				
	}
	
	/**
	 * return all the items
	 * 
	 * @param a string representing the user name
	 * @return A list of curation Items
	 * @throws GCUBEFault -
	 */
	public CurationItemsArray getCuratingItems(String user) throws GCUBEFault{
		logger.debug("getCuratingItems");
			
		CurationItemsArray response= new CurationItemsArray();
		try {
			
			Iterator<Curation> it = ObjectPersistency.get(Curation.class).getObjectByField("creator", user).iterator();
			
			List<CurationItem> listcurations= new ArrayList<CurationItem>();
			while (it.hasNext()){
				Curation cur= it.next();
				if(ServiceContext.getContext().getScope().isEnclosedIn(GCUBEScope.getScope(cur.getScope())))
							listcurations.add(createCurationItem(cur));
			}
			response.setItems(listcurations.toArray(new CurationItem[listcurations.size()]));
			
		}catch(Exception e){
			logger.error("error retrieving Time Series under curation",e);
			throw new GCUBEFault(e,"error retrieving Time Series under curation");
		}
		return response;
	}
	
	/**
	 * return an item stored as curation with the id in the request
	 * 
	 * @param request GetItemRequest
	 * @return a Curation Item
	 * @throws GCUBEFault -
	 */
	public CurationItem getItem(GetItemRequest req) throws GCUBEFault{
		logger.debug("getCuratingItems");
		CurationItem item=null;
		try {
			item= createCurationItem(ObjectPersistency.get(Curation.class).getByKey(req.getId()));
		}catch(Exception e){
			logger.error("error retrieving Time Series under curation",e);
			throw new GCUBEFault(e,"error retrieving Time Series under curation");
		}
		return item;
	}
	
	/**
	 * remove a curation resource
	 * 
	 * @param id
	 * @return
	 * @throws GCUBEFault -
	 */
	public VOID removeCuration(String id) throws GCUBEFault{
		try {
			ObjectPersistency.get(Curation.class).deleteByKey(id);
			CurationContext.getPortTypeContext().getWSHome().remove(
					CurationContext.getPortTypeContext().makeKey(id));
		} catch (Exception e) {
			logger.error("error removing curation with id "+id);
			throw new GCUBEUnrecoverableFault(e);
		}
		return new VOID();
	}
	
	
	public String saveRule(SaveRuleRequest request) throws GCUBEFault{
		try{
			Rule rule = new Rule(request.getName(), request.getDescription(), FilterExplorer.generateRulesDescription(request.getFilter()).toString(), request.getFilter(), Util.mapJavaToSql(request.getType()));
			if (!rule.store()) throw new Exception("error storing the rule");
			return rule.getId();
		}catch(Exception e){
			logger.error("error saving rule",e);
			throw new GCUBEFault("error saving the rule");
		}
	}
	
	public boolean removeRule(String id) throws GCUBEFault{
		try{
			Rule.destroy(id);
			return true;
		}catch(Exception e){
			logger.error("error getting rules",e);
			return false;
		}
	}
	
	public RulesArray getRules(VOID request) throws GCUBEFault{
		try{
			List<RuleItem> listRules = new ArrayList<RuleItem>();
			Iterator<Rule> rulesIterator = Rule.getAll();
			while (rulesIterator.hasNext()){
				Rule rule= rulesIterator.next();
				listRules.add(new RuleItem(rule.getDescription(), rule.getFilter(), rule.getHumanReadable(), rule.getId(), rule.getName(), Util.mapSqlToJava(rule.getType())));
			}
			return new RulesArray(listRules.toArray(new RuleItem[listRules.size()]));
		}catch(Exception e){
			logger.error("error getting rules",e);
			throw new GCUBEFault("error getting rules");
		}
	}
	
	private CurationItem createCurationItem(Curation item) throws Exception{
		CurationItem tempItem= new CurationItem();
		tempItem.setCreator(item.getCreator());
		tempItem.setDate(Util.dateFormatter.format(new Date(item.getDate().getTime())));
		tempItem.setDescription(item.getDescription());
		tempItem.setId(item.getId());
		tempItem.setLenght((int)item.getLength());
		tempItem.setPublisher(item.getPublisher());
		tempItem.setRights(item.getRights());
		tempItem.setSourceId(item.getSourceId());
		tempItem.setSourceName(item.getSourceName());
		tempItem.setTitle(item.getTitle());
		tempItem.setType(item.getType());
		
		return tempItem;
	}
	
	
	
}
