package org.gcube.contentmanagement.timeseriesservice.impl.curation;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.gcube.common.core.faults.GCUBEFault;
import org.gcube.common.core.types.VOID;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.common.dbinterface.Limit;
import org.gcube.common.dbinterface.Order.OrderType;
import org.gcube.common.dbinterface.attributes.SimpleAttribute;
import org.gcube.common.dbinterface.pool.DBSession;
import org.gcube.common.dbinterface.queries.Select;
import org.gcube.common.dbinterface.types.Type;
import org.gcube.contentmanagement.timeseriesservice.impl.context.CurationContext;
import org.gcube.contentmanagement.timeseriesservice.impl.curation.rules.Rule;
import org.gcube.contentmanagement.timeseriesservice.impl.curation.state.CurationResource;
import org.gcube.contentmanagement.timeseriesservice.impl.curation.state.CurationResourceHome;
import org.gcube.contentmanagement.timeseriesservice.impl.utils.Util;
import org.gcube.contentmanagement.timeseriesservice.stubs.CheckDimensionRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.CheckRulesRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.CheckTypeChangeRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.EditColumnRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.EditDimensionRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.ErrorPair;
import org.gcube.contentmanagement.timeseriesservice.stubs.ErrorPairArray;
import org.gcube.contentmanagement.timeseriesservice.stubs.GetDataAsJsonRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.ModifyEntryByIdRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.ModifyEntryByValueRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.ReplaceEntryIdRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.ReplaceEntryValueRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.ReplaceEntryValueWithIdRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.SetColumnAsRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.SetLabelRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.ApplyRulesRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.SetCurationProperties;  
import org.gcube.contentmanagement.timeseriesservice.stubs.types.ColumnDefinition;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.ColumnDefinitionArray;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.GuessDimensionArray;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.RuleItem;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.RulesArray;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.Status;
import org.globus.wsrf.ResourceException;


/**
 * 
 * @author lucio
 *
 */
public class CurationManager {

	static private GCUBELog logger= new GCUBELog(CurationManager.class);
	
	private CurationResource getResource() throws ResourceException{
		CurationResourceHome home=(CurationResourceHome) CurationContext.getPortTypeContext().getWSHome();
		return (CurationResource) home.find();
	}
	
	/**
	 * returns the number of the entries with errors
	 * 
	 * @param void request
	 * @return number of entries 
	 * @throws GCUBEFault -
	 */
	public long errorCount(VOID request) throws GCUBEFault{
		try{
			return getResource().getCount(true);
		}catch (Exception e){
			logger.error("error in count", e);
			throw new GCUBEFault(e);}
	}
	
	
	public void setProperties(SetCurationProperties properties) throws GCUBEFault{
		try{
			getResource().setProperties(properties.getTitle(), properties.getDescription(), properties.getPublisher(), properties.getRights());
		}catch (Exception e){
			logger.error("error setting properties", e);
			throw new GCUBEFault(e);}
	}
	
	/**
	 * return the entries count in the curation resource
	 * 
	 * @param void request
	 * @return the entries count
	 * @throws GCUBEFault -
	 */
	public long count(VOID request) throws GCUBEFault{
		try{
			return getResource().getCount(false);
		}catch (Exception e){ logger.error("error in count", e); throw new GCUBEFault(e);}
	}
	
	
	/**
	 * allows the user to enter in edit mode over a field setting a particular dimension
	 *  
	 * @param request contains the fieldId, DimensionId, KeyName  
	 * @return void
	 * @throws GCUBEFault -
	 */
	public VOID editDimension(EditDimensionRequest request) throws GCUBEFault{
		logger.info("edit dimension");
		try{
			logger.trace("keyId="+request.getKeyId());
			getResource().editDimension(request.getFieldId(), request.getDimensionId(), request.getKeyId());
			getResource().store();
		}catch (Exception e){logger.error("error entering in edit mode",e);throw new GCUBEFault(e);}
		return new VOID();
	}
	
	public VOID editColumn(EditColumnRequest request) throws GCUBEFault{
		logger.info("edit column");
		try{
			getResource().editColumn(request.getFieldId(), new Type(Util.mapJavaToSql(request.getType())));
			getResource().store();
		}catch (Exception e){logger.error("error entering in edit mode",e);throw new GCUBEFault(e);}
		return new VOID();
	}
	
	public VOID editRules(String fieldId) throws GCUBEFault{
		logger.info("edit rules");
		try{
			getResource().editRules(fieldId);
			getResource().store();
		}catch (Exception e){logger.error("error entering in edit mode",e);throw new GCUBEFault(e);}
		return new VOID();
	}
	
	/**
	 * 
	 * allows user to exit from the last edit mode open
	 * 
	 * @param request void
	 * @return void
	 * @throws GCUBEFault -
	 */
	public VOID closeEditing(VOID request) throws GCUBEFault{
		logger.info("closeEditing");
		try{
			getResource().closeEditing();
			getResource().store();
		}catch (Exception e){throw new GCUBEFault(e);}
		return new VOID();
	}
	
	
	/**
	 * returns the resource entries in JSon format
	 * 
	 * @param request a SelectQuery
	 * @return the data as Json
	 * @throws GCUBEFault -
	 */
	public String getDataAsJson(GetDataAsJsonRequest request) throws GCUBEFault{
		long start = System.currentTimeMillis();
		try{
			Select selectQuery= DBSession.getImplementation(Select.class);
			if (request.getQuery().getLimits()!=null) selectQuery.setLimit(new Limit(request.getQuery().getLimits().getLowerLimit(),request.getQuery().getLimits().getUpperLimit())); 
			if (request.getQuery().getOrders()!=null) {
				org.gcube.common.dbinterface.Order[] orders= new org.gcube.common.dbinterface.Order[request.getQuery().getOrders().length];
				for (int i=0; i<request.getQuery().getOrders().length; i++)
					orders[i]= new org.gcube.common.dbinterface.Order(request.getQuery().getOrders()[i].getOrder()==org.gcube.contentmanagement.timeseriesservice.stubs.types.OrderType.Ascending?OrderType.ASC:OrderType.DESC,new SimpleAttribute(request.getQuery().getOrders()[i].getField()));
					
				selectQuery.setOrders(orders);
			}
			logger.trace("requested query is "+selectQuery.getExpression());
 			logger.trace("getDataAsJson took "+(start-System.currentTimeMillis()));
			return getResource().getDataAsJson(selectQuery, request.isOnlyErrors());
		}catch (Exception e){
			logger.error("error getting data",e);
			throw new GCUBEFault(e);}
	}
	
	/**
	 * allows user to modify an entry passing a new value
	 * 
	 * 
	 * @param request ModifyEntryByValueRequest
	 * @return void
	 * @throws GCUBEFault -
	 */
	public VOID modifyEntryByValue(ModifyEntryByValueRequest request) throws GCUBEFault{
		logger.info("modifying value "+request.getFieldId()+" "+request.getNewValue());
		try {
			getResource().modifyEntryValue(request.getFieldId(), request.getNewValue(), request.getRowId());
		}catch (Exception e){
			logger.error("error modifying value",e);
			throw new GCUBEFault(e, "error modifying value");}
		return new VOID();
	}
	
	
	/**
	 *  
	 * allows user to modify a entries setting the new id (this can be apllyed only for a dimension)
	 * 
	 * @param request ModifyEntryByIdRequest
	 * @return void 
	 * @throws GCUBEFault -
	 */
	public VOID modifyEntryById(ModifyEntryByIdRequest request) throws GCUBEFault{
		logger.info("modifying id: "+request.getFieldId()+" "+request.getNewId());
		try {
			getResource().modifyEntryId(request.getFieldId(),request.getNewId(), request.getRowId());
		}catch (Exception e){
			logger.error("error modifying value",e);
			throw new GCUBEFault(e, "error modifying value");}
		return new VOID();
	}
	
	
	/**
	 * allows user to replace a value with another value
	 * 
	 * @param request ReplaceEntryValueRequest
	 * @return void
	 * @throws GCUBEFault -
	 */
	public VOID replaceEntryValue(ReplaceEntryValueRequest request) throws GCUBEFault{
		logger.info("modifying value "+request.getFieldId()+" "+request.getNewValue());
		try {
			getResource().replaceByValue(request.getOldValue(), request.getNewValue(), request.getFieldId());
		}catch (Exception e){
			logger.error("error replacing value",e);
			throw new GCUBEFault(e, "error replacing value");}
		return new VOID();
	}
	
	
	/**
	 * allows user to replace ids for a field with a new one
	 * 
	 * @param request ReplaceEntryIdRequest
	 * @return void 
	 * @throws GCUBEFault -
	 */
	public VOID replaceEntryId(ReplaceEntryIdRequest request) throws GCUBEFault{
		logger.info("modifying by id: "+request.getFieldId()+" "+request.getNewId());
		try {
			getResource().replaceById(request.getOldId(), request.getNewId(), request.getFieldId());
		}catch (Exception e){
			logger.error("error replacing value",e);
			throw new GCUBEFault(e, "error replacing value");}
		return new VOID();
	}
	
	
	/**
	 * returns the columns definition for the current curation  
	 * 
	 * @param request void
	 * @return ColumnDefinitionArray
	 * @throws GCUBEFault -
	 */
	public ColumnDefinitionArray getDimensions(VOID request) throws GCUBEFault{
		long start = System.currentTimeMillis();
		try{
			ColumnDefinitionArray toReturn = new ColumnDefinitionArray(getResource().getColumnDefinition());
			logger.trace("getColumnDefinition took "+(start-System.currentTimeMillis()));
			return toReturn;
		}catch (Exception e){
			logger.error("error getting Dimensions",e);
			throw new GCUBEFault(e);}
	}
	
	
	/**
	 * 
	 * @param request void
	 * @return void
	 * @throws GCUBEFault -
	 */
	public VOID saveColumnDefinition(VOID request) throws GCUBEFault{
		try {
			getResource().saveColumnDefinition();
			getResource().store();
		}catch (GCUBEFault e){
			logger.error(e);
			throw e;
		}catch (Exception e){
			logger.error("error saving column",e);
			throw new GCUBEFault(e);
		}
		return new VOID();
	}
	
	
	/**
	 * 
	 * @param request SetLabelRequest
	 * @return void
	 * @throws GCUBEFault -
	 */
	public VOID setLabelName(SetLabelRequest request) throws GCUBEFault{
		try {
			getResource().setLabel(request.getFieldId(), request.getNewLabel());
		}catch (Exception e){
			logger.error("error modifying value",e);
			throw new GCUBEFault(e, "error setting attribute lable for "+request.getFieldId());}
		return new VOID();		
	}
	
	
	/**
	 * 
	 * @param request void
	 * @return void
	 * @throws GCUBEFault -
	 */
	public VOID removeAllErrors(VOID request) throws GCUBEFault{
		try {
			getResource().removeAllErrors();
		}catch (Exception e){
			logger.error("error modifying value",e);
			throw new GCUBEFault(e, "error removing all errors");}
		return new VOID();
	}
	
	
	/**
	 * 
	 * @param rowId long
	 * @return void
	 * @throws GCUBEFault -
	 */
	public VOID removeSingleError(long rowId) throws GCUBEFault{
		try {
			getResource().removeError(rowId);
		}catch (Exception e){
			logger.error("error modifying value",e);
			throw new GCUBEFault(e, "error removing all errors");}
		return new VOID();
	}
	
	
	/**
	 * 
	 * @param request void
	 * @return void
	 * @throws GCUBEFault -
	 */
	public VOID closeEdit(VOID request) throws GCUBEFault{
		try {
			getResource().closeEditing();
		}catch (Exception e){
			logger.error("error modifying value",e);
			throw new GCUBEFault(e, "error closing edit mode");}
		return new VOID();
	}
	
	
	/**
	 * 
	 * @param request void
	 * @return boolean
	 * @throws GCUBEFault -
	 */
	public boolean isInEditMode(VOID request) throws GCUBEFault{
		try {
			return getResource().isUnderEdit();
		}catch (Exception e){
			logger.error("error modifying value",e);
			throw new GCUBEFault(e, "error verifying edit mode");}
	}
	
	
	/**
	 * 
	 * @param request void
	 * @return State
	 * @throws GCUBEFault -
	 */
	public Status isUnderInitialization(VOID request) throws GCUBEFault{
		try {
			return getResource().getUnderCreationState();
		}catch (Exception e){
			logger.error("error verifying initialization",e);
			throw new GCUBEFault(e, "error verifying initialization");}
	}
	
	
	/**
	 * 
	 * @param request SetColumnAsRequest
	 * @return void
	 * @throws GCUBEFault -
	 */
	public VOID setColumnAs(SetColumnAsRequest request) throws GCUBEFault{
		try {
			getResource().setColumn(request.getColumnType(),request.getFieldId());
			getResource().store();
		}catch (Exception e){
			logger.error("error setting attribute type",e);
			throw new GCUBEFault(e, "error setting attribute type");}
		return new VOID();
	}
	
	
	/**
	 * 
	 * @param request GuessRequest
	 * @return DimensionArray
	 * @throws GCUBEFault -
	 */
	public GuessDimensionArray guessDimension(String fieldId) throws GCUBEFault{
		try {
			CurationResource resource= getResource();
			return resource.guess(fieldId);
		}catch (Exception e){
			logger.error("error in guess Dimension",e);
			throw new GCUBEFault(e, "error in guess Dimension");}
	}
	
	
	
	/**
	 * 
	 * @param req void
	 * @return State
	 * @throws GCUBEFault -
	 */
	public Status initializeEditingState(VOID req) throws GCUBEFault{
		try {
			return getResource().getInitializeEditingState();
		}catch (Exception e){
			logger.error("error in guess Dimension",e);
			throw new GCUBEFault(e, "error in guess Dimension");}
	}
	
	
	/**
	 * 
	 * @param req void
	 * @return ColumnDefinition
	 * @throws GCUBEFault -
	 */
	public ColumnDefinition columnInEditMode(VOID req) throws GCUBEFault{
		try {
			return getResource().columnInEditMode();
		}catch (Exception e){
			logger.error("error in columnInEditMode",e);
			throw new GCUBEFault(e, "error in columnInEditMode");}
	}
	
	
	
	/**
	 * 
	 * @param fieldId String
	 * @return void
	 * @throws GCUBEFault -
	 */
	public VOID removeColumn(String fieldId) throws GCUBEFault{
		try {
			getResource().removeColumn(fieldId);
			getResource().store();
		}catch (Exception e){
			logger.error("error in remove column",e);
			throw new GCUBEFault(e, "error in remove column");}
		return new VOID();
	}
	
	/**
	 * 
	 * @param request ReplaceEntryValueWithIdRequest
	 * @return void
	 * @throws GCUBEFault -
	 */
	public VOID replaceEntryValueWithId(ReplaceEntryValueWithIdRequest request) throws GCUBEFault{
		logger.info("modifying value with id: "+request.getFieldId()+" "+request.getNewId());
		try {
			if (getResource().getFieldEditor()==null) throw new GCUBEFault("the Service is not in edit mode");
			getResource().getFieldEditor().replaceEntryValueWithId(request.getOldValue(), request.getNewId(), request.getFieldId());
		}catch (Exception e){
			logger.error("error replacing value",e);
			throw new GCUBEFault(e, "error replacing value");}
		return new VOID();
	}
	
	public String getPossibleValues(String word) throws GCUBEFault{
		try {
			return getResource().getPossibleValues(word);
		}catch (Exception e){
			logger.error("error guessing word "+word,e);
			throw new GCUBEFault(e, "error guessing word "+word);}
	}
	
	public ErrorPairArray getDistinctErrors(VOID req) throws GCUBEFault{
		try {
			if (getResource().getFieldEditor()==null) throw new GCUBEFault("the Service is not in edit mode");
			else return new ErrorPairArray(getResource().getFieldEditor().getDistinctErrors().toArray(new ErrorPair[0]));
		}catch (Exception e){
			logger.error("error getting distinct errors",e);
			throw new GCUBEFault(e,"error getting distinct errors");}
	}
	
	/**
	 * 
	 * @param request
	 * @return
	 * @throws GCUBEFault
	 */
	public VOID applyRules(ApplyRulesRequest request) throws GCUBEFault{
		try {
			if (request.getRules()==null || request.getRules().getItems()==null) return new VOID();
			getResource().applyRules(request.getRules().getItems(), request.getFieldId());
		}catch (Exception e){
			logger.error("error applying rules",e);
			throw new GCUBEFault(e,"error applying rules");}
		return new VOID();
	}
	
	/**
	 * 
	 * @param fieldID
	 * @return
	 * @throws GCUBEFault
	 */
	public RulesArray getApplyedRules(String fieldID) throws GCUBEFault{
		try {
			List<RuleItem> listRule = new ArrayList<RuleItem>();
			Iterator<Rule> ruleIt= getResource().applyedRules(fieldID);
			while(ruleIt.hasNext()){
				Rule rule = ruleIt.next();
				listRule.add(new RuleItem(rule.getDescription(), rule.getFilter(), rule.getHumanReadable(), rule.getId(), rule.getName(), Util.mapSqlToJava(rule.getType())));
			}
			
			return new RulesArray(listRule.toArray(new RuleItem[listRule.size()]));
		}catch (Exception e){
			logger.error("error getting rules",e);
			throw new GCUBEFault(e,"error getting rules");}
	}
	
	/**
	 * 
	 * @param fieldID
	 * @return
	 * @throws GCUBEFault
	 */
	public VOID removeApplyedRules(String fieldId) throws GCUBEFault{
		try {
			getResource().removeAllRules(fieldId);			
			return new VOID();
		}catch (Exception e){
			logger.error("error getting rules",e);
			throw new GCUBEFault(e,"error getting rules");}
	}
	
	/**
	 * 
	 * @param fieldID
	 * @return
	 * @throws GCUBEFault
	 */
	public RulesArray getApplyableRules(String fieldID) throws GCUBEFault{
		try {
			List<RuleItem> listRule = new ArrayList<RuleItem>();
			
			Iterator<Rule> ruleIt= getResource().applyableRules(fieldID);
			while(ruleIt.hasNext()){
				Rule rule = ruleIt.next();
				listRule.add(new RuleItem(rule.getDescription(), rule.getFilter(), rule.getHumanReadable(), rule.getId(), rule.getName(), Util.mapSqlToJava(rule.getType())));
			}
			
			return new RulesArray(listRule.toArray(new RuleItem[listRule.size()]));
		}catch (Exception e){
			logger.error("error getting rules",e);
			throw new GCUBEFault(e,"error getting rules");}
	}
	
	/**
	 * check how many errors there will be setting a particular dimension for a selected field
	 * 
	 * @param req CheckDimensionRequest
	 * @return the error number
	 * @throws GCUBEFault -
	 */
	public long checkDimension(CheckDimensionRequest req) throws GCUBEFault{
		try{
			return getResource().checkDimension(req.getCodeListId(), req.getFieldId(), req.getKeyId());
		}catch (Exception e){
			logger.error("error checking dimension",e);
			throw new GCUBEFault(e);}
	}
	
	/**
	 * check how many errors there will be setting a particular dimension for a selected field
	 * 
	 * @param req CheckDimensionRequest
	 * @return the error number
	 * @throws GCUBEFault -
	 */
	public long checkRules(CheckRulesRequest req) throws GCUBEFault{
		try{
			return getResource().checkRules(req.getFieldId(), req.getFilters());
		}catch (Exception e){
			logger.error("error checking rules",e);
			throw new GCUBEFault(e);}
	}
	
	/**
	 * check how many errors there will be setting a particular dimension for a selected field
	 * 
	 * @param req CheckDimensionRequest
	 * @return the error number
	 * @throws GCUBEFault -
	 */
	public long checkTypeChange(CheckTypeChangeRequest req) throws GCUBEFault{
		try{
			return getResource().checkTypeChange(req.getFieldId(), new Type(Util.mapJavaToSql(req.getType())));
		}catch (Exception e){
			logger.error("error checking dimension",e);
			throw new GCUBEFault(e);}
	}

	
}
