package org.gcube.contentmanagement.timeseriesservice.impl.codelist;

import java.io.File;
import java.net.URI;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import org.gcube.common.core.faults.GCUBEFault;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.common.dbinterface.Limit;
import org.gcube.common.dbinterface.persistence.ObjectNotFoundException;
import org.gcube.common.dbinterface.pool.DBSession;
import org.gcube.common.dbinterface.queries.Select;
import org.gcube.common.dbinterface.tables.Table;
import org.gcube.contentmanagement.codelistmanager.entities.TableField;
import org.gcube.contentmanagement.codelistmanager.managers.CodeListImport;
import org.gcube.contentmanagement.timeseriesservice.calls.RSWrapper;
import org.gcube.contentmanagement.timeseriesservice.impl.codelist.wrappers.CLImportItemWrapper;
import org.gcube.contentmanagement.timeseriesservice.impl.context.ServiceContext;
import org.gcube.contentmanagement.timeseriesservice.impl.utils.Util;
import org.gcube.contentmanagement.timeseriesservice.stubs.CLImportedItem;
import org.gcube.contentmanagement.timeseriesservice.stubs.CLImportedItemsArray;
import org.gcube.contentmanagement.timeseriesservice.stubs.ImportFromCsvRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.ImportFromSdmxRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.ImportStatusResponse;
import org.gcube.contentmanagement.timeseriesservice.stubs.codelist.types.CodelistColumnDefinition;
import org.gcube.contentmanagement.timeseriesservice.stubs.codelist.types.CodelistColumnType;
import org.gcube.contentmanagement.timeseriesservice.stubs.codelist.types.CodelistColumnsDefinitionArray;
import org.gcube.contentmanagement.timeseriesservice.stubs.codelist.types.CodelistIdentifier;
import org.gcube.contentmanagement.timeseriesservice.stubs.codelist.types.DataAsJsonRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.AccessFault;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.ItemNotFoundFault;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.Status;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.TrasmissionFault;

public class CodelistImportManager {

	
	private static GCUBELog logger = new GCUBELog(CodelistImportManager.class);
	
	
	/**
	 * 
	 * @param request
	 * @return
	 * @throws GCUBEFault
	 */
	public String importFromCsv(final ImportFromCsvRequest request) throws GCUBEFault{
		
		final File file;
		try{
			file = RSWrapper.getStreamFromLocator(new URI(request.getRsLocator()));	
		}catch (Exception e) {
			throw new TrasmissionFault();
		}
		
		if (file ==null ){
			logger.error("file not found");
			throw new TrasmissionFault();
		}
				
		try{
			final CodeListImport importer = new CodeListImport();
			importer.setDescription(request.getDescription());
			importer.setName(request.getName());
			importer.store();
			Thread importThread = new Thread(){
				public void run(){
					importer.importCSV(file, request.getEncoding(), request.isHasHeader(), request.getDelimiter().charAt(0), true);
				}
			};
			ServiceContext.getContext().useServiceCredentials(importThread);
			importThread.start();
			new CLImportItemWrapper(request.getOwner(), importer.getId(), ServiceContext.getContext().getScope()).store();
			logger.debug("import finished");
			return importer.getId();

		}catch (Exception e) {
			logger.error("error importing from csv",e);
			throw new GCUBEFault(e);
		}
	}
	
	
	/**
	 * 
	 * @param request
	 * @return
	 * @throws GCUBEFault
	 */
	public String importFromSdmx(final ImportFromSdmxRequest request) throws GCUBEFault{
		
				
		try{
			final CodeListImport importer = new CodeListImport();
			logger.trace("the importer has been stored?"+importer.store());
			if (request.getSdmxUri()!=null){
				final URI uri = new URI(request.getSdmxUri());
				Thread importThread = new Thread(){
					public void run(){
						importer.importSDMX(uri);
					}
				};
				ServiceContext.getContext().useServiceCredentials(importThread);
				importThread.start();
			}else{
				final File sdmxFile = RSWrapper.getStreamFromLocator(new URI(request.getRsLocator()));
				Thread importThread = new Thread(){
					public void run(){
						importer.importSDMX(sdmxFile);
						sdmxFile.delete();
					}
				};
				ServiceContext.getContext().useServiceCredentials(importThread);
				importThread.start();
			}
			
			new CLImportItemWrapper(request.getOwner(), importer.getId(), ServiceContext.getContext().getScope()).store();
			logger.debug("import finished");
			return importer.getId();
		}catch (Exception e) {
			logger.error("error importing from sdmx",e);
			throw new GCUBEFault();
		}
		
	}
	
	/**
	 * 
	 * @param user
	 * @return
	 * @throws GCUBEFault
	 */
	public CLImportedItemsArray getImportedCodelists(String user) throws GCUBEFault{
		List<CLImportedItem> clItems = new ArrayList<CLImportedItem>();
		Iterator<CLImportItemWrapper> clImportWrapperIterator;
		try {
			clImportWrapperIterator = CLImportItemWrapper.getByUser(user);
			
			while (clImportWrapperIterator.hasNext()){
				CLImportItemWrapper wrapperItem = clImportWrapperIterator.next();
				if (wrapperItem.getScope().equals(ServiceContext.getContext().getScope())){
					try{
						CodeListImport item = wrapperItem.getCodelist();
						clItems.add(new CLImportedItem(item.getAgencyId(), item.getProgress(), item.getLastUpdate(),item.getDescription(), item.getId(), item.getName(), wrapperItem.getOwner(), wrapperItem.getStatus(), item.getVersion()));
					}catch (ObjectNotFoundException e) {
						logger.warn("object with id "+wrapperItem.getCodelistId()+" not found");
						wrapperItem.remove();
					}
				}
			}
		} catch (Exception e) {
			logger.error("error getting imported items",e);
			throw new GCUBEFault("error retrieving codelist imported items");
		}
		return new CLImportedItemsArray(clItems.toArray(new CLImportedItem[clItems.size()]));
	
	}
	
	/**
	 * 
	 * @param ide
	 * @return
	 * @throws GCUBEFault
	 */
	public ImportStatusResponse getImportStatus(CodelistIdentifier ide) throws GCUBEFault{
		try{
			ImportStatusResponse status = new ImportStatusResponse();
			CLImportItemWrapper clImportWrapper = CLImportItemWrapper.get(ide.getId());
			
	 		if (!clImportWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !ide.getUser().equals(clImportWrapper.getOwner())) throw new AccessFault();
			CodeListImport clImport = clImportWrapper.getCodelist(); 
			logger.debug(" status: "+clImport.getStatus()+" totalLine: "+clImport.getTotalLines()+"   linecount:"+clImport.getProgress() );
			
			switch (clImport.getStatus() ) {
			case Initialized:
				status.setStatus(Status.Initializing);
				break;
			case InProgress: 
				status.setStatus(Status.Open); 
				break;
			case Complete:
				status.setStatus(Status.Close); 
				break;
			case Failed:
				status.setStatus(Status.Error);
				break;
			default:
				status.setStatus(Status.Open);
			} 
			
			status.setTotalLine(clImport.getTotalLines());
			status.setLineCount(clImport.getProgress());
			
			return status;
		}catch (ObjectNotFoundException e) {
			logger.error("item not found", e);
			throw new ItemNotFoundFault();
		}catch (Exception e1) {
			logger.error("error retrieving import status",e1);
			throw new GCUBEFault(e1);
		}
	}
	
	/**
	 * 
	 * @param request
	 * @return
	 * @throws GCUBEFault
	 */
	public String getDataAsJson(DataAsJsonRequest request) throws GCUBEFault{
		try {
			CLImportItemWrapper clImportWrapper = CLImportItemWrapper.get(request.getIdentifier().getId());
			if (!clImportWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !request.getIdentifier().getUser().equals(clImportWrapper.getOwner())) throw new AccessFault();
			Select selectQuery= DBSession.getImplementation(Select.class);
			selectQuery.setTables(new Table(clImportWrapper.getCodelist().getTableName()));
			selectQuery.setLimit(new Limit(request.getLimit().getLowerLimit(), request.getLimit().getUpperLimit()));
			//logger.trace(selectQuery.getExpression());
			return selectQuery.getResultAsJSon(true);
		}catch (ObjectNotFoundException e) {
			logger.error("item not found", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error getting data as json",e1);
			throw new GCUBEFault(e1);
		}
		
	}
	
	/**
	 * 
	 * @param identifier
	 * @return
	 * @throws GCUBEFault
	 */
	public CodelistColumnsDefinitionArray getColumnsDefinition(CodelistIdentifier identifier) throws GCUBEFault{
		List<CodelistColumnDefinition> columnDefinitionList= new ArrayList<CodelistColumnDefinition>();
		try {
			CLImportItemWrapper clImportWrapper = CLImportItemWrapper.get(identifier.getId());
			if (!clImportWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !identifier.getUser().equals(clImportWrapper.getOwner())) throw new AccessFault();
			CodeListImport clImport = clImportWrapper.getCodelist(); 
			TableField[] returnList = clImport.getFieldsName().values().toArray(new TableField[0]);
			Arrays.sort(returnList);
			for (TableField cd :returnList)
				columnDefinitionList.add(new CodelistColumnDefinition(CodelistColumnType.fromString(cd.getColumnReference().getType().toString()), Util.mapSqlToJava(cd.getDataType()),  cd.getId(), cd.getFieldName(),cd.getColumnReference().getCodelistReferenceId()));
			return new CodelistColumnsDefinitionArray(columnDefinitionList.toArray(new CodelistColumnDefinition[columnDefinitionList.size()]));
		}catch (ObjectNotFoundException e) {
			logger.error("item not found", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error getting column definition",e1);
			throw new GCUBEFault(e1);
		}
		
	}
	
	/**
	 * 
	 * @param identifier
	 * @return
	 * @throws GCUBEFault
	 */
	public boolean remove(CodelistIdentifier identifier) throws GCUBEFault{
		try {
			CLImportItemWrapper clImportWrapper = CLImportItemWrapper.get(identifier.getId());
			if (!clImportWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !identifier.getUser().equals(clImportWrapper.getOwner())) throw new AccessFault();
			return clImportWrapper.remove();
		}catch (ObjectNotFoundException e) {
			logger.error("item not found", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error removing imported codelist",e1);
			throw new GCUBEFault(e1);
		}
	}
	
	/**
	 * 
	 * @param identifier
	 * @return
	 * @throws GCUBEFault
	 */
	public CLImportedItem getCLImportItem(CodelistIdentifier identifier) throws GCUBEFault{
		try {
			CLImportItemWrapper clImportWrapper = CLImportItemWrapper.get(identifier.getId());
			if (!clImportWrapper.getScope().toString().equals(ServiceContext.getContext().getScope().toString()) || !identifier.getUser().equals(clImportWrapper.getOwner())) throw new AccessFault();
			CodeListImport clImport = clImportWrapper.getCodelist();
			return new CLImportedItem(clImport.getAgencyId(),clImport.getProgress(),clImport.getLastUpdate(), clImport.getDescription(), clImport.getId(), clImport.getName(), clImportWrapper.getOwner(), clImportWrapper.getStatus(), clImport.getVersion());
		}catch (ObjectNotFoundException e) {
			logger.error("item not found", e);
			throw new ItemNotFoundFault();
		} catch (Exception e1) {
			logger.error("error getting imported item with id "+identifier.getId(),e1);
			throw new GCUBEFault(e1);
		}
	}
	
}	
