package org.gcube.contentmanagement.timeseriesservice.impl.timeseries;

import org.gcube.contentmanagement.timeseriesservice.stubs.AddFAOAreasRequest;
import java.util.Calendar;
import java.util.LinkedList;
import org.gcube.common.core.faults.GCUBEFault;
import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.scope.GCUBEScope.Type;
import org.gcube.common.core.state.GCUBEWSResourceKey;
import org.gcube.common.core.types.StringArray;
import org.gcube.common.core.types.VOID;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.contentmanagement.timeseries.geotools.filters.AFilter;
import org.gcube.contentmanagement.timeseries.geotools.filters.SpaceFilter;
import org.gcube.contentmanagement.timeseries.geotools.filters.SpeciesFilter;
import org.gcube.contentmanagement.timeseries.geotools.filters.custom.CountryYearFilter;
import org.gcube.contentmanagement.timeseries.geotools.filters.custom.FAOAreaFilter;
import org.gcube.contentmanagement.timeseriesservice.stubs.AggregateRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.ChangeColumnRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.ChartDataRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.CompatibleColumnsRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.DistinctValuesRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.GetAreaTransformationResultRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.GetSpaceTransformationResultRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.GetSpaceSpeciesTransformationResultRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.GetTimeCountriesTransformationResultRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.IsUnionAllowedRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.CurrentState;
import org.gcube.contentmanagement.timeseriesservice.stubs.DenormalizeRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.GetAllDataAsJsonRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.GroupRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.OperationsListArray;
import org.gcube.contentmanagement.timeseriesservice.stubs.EnrichRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.TsFilterRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.UnionRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.GetFishingMontlyEffortRequest;
import org.gcube.contentmanagement.timeseriesservice.impl.context.CurationContext;
import org.gcube.contentmanagement.timeseriesservice.impl.context.ImportContext;
import org.gcube.contentmanagement.timeseriesservice.impl.context.ServiceContext;
import org.gcube.contentmanagement.timeseriesservice.impl.context.TimeSeriesContext;
import org.gcube.contentmanagement.timeseriesservice.impl.curation.state.CurationResource;
import org.gcube.contentmanagement.timeseriesservice.impl.history.CurationHistory;
import org.gcube.contentmanagement.timeseriesservice.impl.history.TSHistory;
import org.gcube.contentmanagement.timeseriesservice.impl.history.TSHistoryItem;
import org.gcube.contentmanagement.timeseriesservice.impl.history.TSHistory.ParentType;
import org.gcube.contentmanagement.timeseriesservice.impl.timeseries.operations.AddFAOAreas;
import org.gcube.contentmanagement.timeseriesservice.impl.timeseries.operations.Aggregation;
import org.gcube.contentmanagement.timeseriesservice.impl.timeseries.operations.ColumnChange;
import org.gcube.contentmanagement.timeseriesservice.impl.timeseries.operations.Denormalization;
import org.gcube.contentmanagement.timeseriesservice.impl.timeseries.operations.Enrichment;
import org.gcube.contentmanagement.timeseriesservice.impl.timeseries.operations.Filter;
import org.gcube.contentmanagement.timeseriesservice.impl.timeseries.operations.Grouping;
import org.gcube.contentmanagement.timeseriesservice.impl.timeseries.operations.Union;
import org.gcube.contentmanagement.timeseriesservice.impl.timeseries.state.TimeSeriesResource;
import org.globus.wsrf.ResourceException;
import org.gcube.contentmanagement.timeseriesservice.stubs.GetTransformationStatusResponse;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.ColumnDefinitionArray;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.CompatibleColumnsMappingArray;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.CurationHistoryItem;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.CurationHistoryOps;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.EntryType;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.ExportRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.FilterCondition;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.OperationType;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.PublishingLevel;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.Status;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.TimeseriesHistory;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.TimeseriesHistoryItem;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.TsHistoryOps;


/**
 * 
 * @author lucio
 *
 */
public class TimeSeriesManager {

	private static  GCUBELog logger = new GCUBELog(TimeSeriesManager.class);
	
	protected TimeSeriesResource getResource() throws ResourceException{
		return (TimeSeriesResource)TimeSeriesContext.getPortTypeContext().getWSHome().find();
	}
	
	
	/**
	 * 
	 * @param request
	 * @return
	 * @throws GCUBEFault
	 */
	public String getAllDataAsJson(GetAllDataAsJsonRequest request) throws GCUBEFault{
		try {
			return getResource().getAllDataAsJSon(request.getLimit(), request.getOrder(), request.getUser());
		} catch (Exception e) {
			logger.error("error getting data",e);
			throw new GCUBEFault("error getting data");
		}
	}
	
	/**
	 * 
	 * @param req
	 * @return
	 * @throws GCUBEFault
	 */
	public Status getInitializationState(VOID req) throws GCUBEFault{
		try {
			return getResource().getInitializationState();
		} catch (Exception e) {
			logger.error("error getting initialization state",e);
			throw new GCUBEFault("error getting initialization state");
		}
	}
	
	/**
	 * 
	 * @param req
	 * @return
	 */
	public String exportAsCsv(ExportRequest req) throws GCUBEFault{
		try {
			return getResource().exportAsCsv(req);
		} catch (Exception e) {
			logger.error("error exporting TS as CSV",e);
			throw new GCUBEFault("error exporting TS as CSV");
		}
	}
	
	
	/**
	 * 
	 * @param req
	 * @return
	 * @throws GCUBEFault
	 */
	public ColumnDefinitionArray getColumnsDefinition(String user) throws GCUBEFault{
		try {
			return new ColumnDefinitionArray(getResource().getCurrentColumnsDefinition(user));
		} catch (Exception e) {
			logger.error("error in getColumnsDefinition",e);
			throw new GCUBEFault("error exporting TS as CSV");
		}
	}
			
	
	/**
	 * 
	 * @param user
	 * @return
	 * @throws GCUBEFault
	 */
	public long getElementsCount(String user) throws GCUBEFault{
		try {
			return getResource().getElementCount(user);
		} catch (Exception e) {
			logger.error("error getting count",e);
			throw new GCUBEFault("error getting count");
		}
	}
	
	
	/**
	 * 
	 * @param request
	 * @return
	 * @throws GCUBEFault
	 */
	public VOID tsFilter(TsFilterRequest request) throws GCUBEFault{
		try {
			getResource().performOperation(request.getUser(), Filter.class, request.getFilter());
			getResource().store();
			return new VOID();
		} catch (Exception e) {
			logger.error("error in tsFilter",e);
			throw new GCUBEFault("error filtering TS");
		}
	}
	
	
	/**
	 * 
	 * @param request
	 * @return
	 * @throws GCUBEFault
	 */
	public VOID union(UnionRequest request) throws GCUBEFault{
		logger.debug("union operation requested");
		try {
			getResource().performOperation(request.getUser(), Union.class, request.getId(), request.getFieldMappingList()); 
			getResource().store();
			return new VOID();
		} catch (Exception e) {
			logger.error("error in union operation",e);
			throw new GCUBEFault("error in union operation");
		}
	}
	
	/**
	 * 
	 * @param request
	 * @return
	 * @throws GCUBEFault
	 */
	public VOID changeColumn(ChangeColumnRequest request) throws GCUBEFault{
		logger.debug("union operation requested");
		try {
			getResource().performOperation(request.getUser(), ColumnChange.class, request.getFieldId(), request.getKeyId()); 
			getResource().store();
			return new VOID();
		} catch (Exception e) {
			logger.error("error in union operation",e);
			throw new GCUBEFault("error in union operation");
		}
	}
	
	
	/**
	 * 
	 * @param request
	 * @return
	 * @throws GCUBEFault
	 */
	public VOID enrich(EnrichRequest request) throws GCUBEFault{
		logger.debug("enrichment operation requested");
		try {
			getResource().performOperation(request.getUser(), Enrichment.class, request); 
			getResource().store();
			return new VOID();
		} catch (Exception e) {
			logger.error("error in enrichment operation",e);
			throw new GCUBEFault("error in enrichment operation");
		}
	}
	
	/**
	 * 
	 * @param request
	 * @return
	 * @throws GCUBEFault
	 */
	public VOID addFAOAreas(AddFAOAreasRequest request) throws GCUBEFault{
		logger.debug("enrichment operation requested");
		try {
			getResource().performOperation(request.getUser(), AddFAOAreas.class, request); 
			getResource().store();
			return new VOID();
		} catch (Exception e) {
			logger.error("error in enrichment operation",e);
			throw new GCUBEFault("error in enrichment operation");
		}
	}
	
	/**
	 * 
	 * @param request AggregateRequest
	 * @return
	 * @throws GCUBEFault
	 */
	public VOID aggregate(AggregateRequest request) throws GCUBEFault{
		logger.debug("aggregate operation requested with "+request.getFieldList().toString()+" as field list");
		TimeSeriesResource res=null;
		
		try {
			res= getResource();
		}catch(Exception e){
			logger.error("error getting the resource", e);
			new GCUBEFault(e,"error getting the resource");
		}
		
		try {
			for (String fieldId: request.getFieldList())
				if (res.getColumnDefinitionReference(fieldId).getColumnType()!=EntryType.Dimension)
					throw new Exception("the parameters are not correct");
		}catch(Exception e){ 
			logger.error("aggragate: the parameters are not correct",e);
			throw new GCUBEFault(e,"aggragate: the parameters are not correct");}
		
		try{
			getResource().performOperation(request.getUser(), Aggregation.class, request); 
			getResource().store();
			return new VOID();
		} catch (Exception e) {
			logger.error("error in aggregate operation",e);
			throw new GCUBEFault("error in aggregate operation");
		}
	}
	
	
	/**
	 * 
	 * @param request GroupRequest
	 * @return
	 * @throws GCUBEFault
	 */
	public VOID group(GroupRequest request) throws GCUBEFault{
		logger.debug("union operation requested");
		try {
			getResource().performOperation(request.getUser(), Grouping.class, request); 
			getResource().store();
			return new VOID();
		} catch (Exception e) {
			logger.error("error in group operation",e);
			throw new GCUBEFault("error in group operation");
		}
	}
	
	/**
	 * 
	 * @param req
	 * @return
	 * @throws Exception
	 */
	public VOID denormalize(DenormalizeRequest request) throws GCUBEFault{
		logger.debug("denormalize operation requested");
		try {
			if (getResource().getColumnDefinitionReference(request.getAttributeFieldId()).getColumnType()!=EntryType.Attribute 
					||  getResource().getColumnDefinitionReference(request.getValueFieldId()).getColumnType()!=EntryType.Value)
				throw new GCUBEFault("the parameters are not correct");
			getResource().performOperation(request.getUser(), Denormalization.class, request.getAttributeFieldId(), request.getValueFieldId()); 
			getResource().store();
			return new VOID();
		} catch (Exception e) {
			logger.error("error denormalizing",e);
			throw new GCUBEFault("error denormalizing the time series");
		}
	}
	
	
	
	/**
	 * 
	 * @param user
	 * @return
	 * @throws GCUBEFault
	 */
	public VOID discardCurrentOperation(String user)throws GCUBEFault{
		logger.trace("discarding operation");
		try {
			getResource().discardOperation(user);
			getResource().store();
			return new VOID();
		} catch (Exception e) {
			logger.error("error discarding operation",e);
			throw new GCUBEFault("error discarding operation");
		}
	}
	
	/**
	 * 
	 * @param user
	 * @return
	 * @throws GCUBEFault
	 */
	public VOID discardAllOperations(String user)throws GCUBEFault{
		try {
			getResource().discardAllOperations(user);
			getResource().store();
			return new VOID();
		} catch (Exception e) {
			logger.error("error discarding operation",e);
			throw new GCUBEFault("error discarding operation");
		}
	}
	
	/**
	 * 
	 * @param user
	 * @return
	 * @throws GCUBEFault
	 */
	public FilterCondition getFilter(String user)throws GCUBEFault{
		try {
			return ((Filter)getResource().getCurrentOperation(user)).getFilterCondition();
		} catch (Exception e) {
			logger.error("error getCurrentOperation",e);
			throw new GCUBEFault("error getCurrentOperation");
		}
		
	}
	
	/**
	 * 
	 * @param user
	 * @return
	 * @throws GCUBEFault
	 */
	public CurrentState getCurrentState(String user) throws GCUBEFault{
		try {
			if (getResource().getCurrentOperation(user)==null)
				return CurrentState.Closed;
			else return getResource().getCurrentOperation(user).getCurrentState();
				
		} catch (Exception e) {
			logger.error("error getCurrentOperation",e);
			throw new GCUBEFault("error getCurrentOperation");
		}
	}
	
	/**
	 * 
	 * @param user
	 * @return
	 * @throws GCUBEFault
	 */
	public OperationType getCurrentOperation(String user) throws GCUBEFault{
		try {
			if (getResource().getCurrentOperation(user)==null)
				return OperationType.None;
			else return getResource().getCurrentOperation(user).getType();
				
		} catch (Exception e) {
			logger.error("error getCurrentOperation",e);
			throw new GCUBEFault("error getCurrentOperation");
		}
	}
	
	/**
	 * 
	 * @param user
	 * @return
	 * @throws GCUBEFault
	 */
	public OperationsListArray getOperationsList(String user) throws GCUBEFault{
		try {
			if (getResource().getOperationList(user)==null)
				return new OperationsListArray(new OperationType[0]);
			else return new OperationsListArray(getResource().getOperationList(user));
				
		} catch (Exception e) {
			logger.error("error getOperationsList",e);
			throw new GCUBEFault("error getOperationsList");
		}
	}
	
	/**
	 * 
	 * @param resourceId
	 * @return
	 * @throws GCUBEFault
	 */
	public boolean isUnionAllowed(IsUnionAllowedRequest request) throws GCUBEFault{
		logger.debug("check compatibility with "+request.getId());
		try{
			TimeSeriesResource sourceResource=getResource();
			TimeSeriesResource destinationResource= (TimeSeriesResource)TimeSeriesContext.getPortTypeContext().getWSHome().find(TimeSeriesContext.getPortTypeContext().makeKey(request.getId()));
			boolean returnValue= Union.checkCompatibility(sourceResource.getCurrentColumnsDefinition(request.getUser()), destinationResource.getCurrentColumnsDefinition(request.getUser()));
			logger.debug("the TimeSeries "+(returnValue?"is compatible":"is not compatible"));
			return returnValue;
		}catch(Exception e){
			logger.error("error verifying union",e);
			return false;			
		}
	}
	
	/**
	 * 
	 * @param req void
	 * @return
	 * @throws Exception -
	 */
	public VOID publish(PublishingLevel level) throws GCUBEFault{
		try{
			if (level==PublishingLevel.VRE)
				getResource().setTSPublished();
			else{
				GCUBEScope scope= ServiceContext.getContext().getScope();
				GCUBEWSResourceKey resourceKey = getResource().getID();
				if (scope.getType()== Type.VRE)
					ServiceContext.getContext().setScope(scope.getEnclosingScope());
				TimeSeriesContext.getPortTypeContext().getWSHome().create(resourceKey);
				getResource().publishAtVOLevel(ServiceContext.getContext().getScope());
				ServiceContext.getContext().setScope(scope);
			}
		}catch (Exception e) {
			logger.error("error publishing the ts",e);
			throw new GCUBEFault("error publishing the ts");
		}
		return new VOID();
	}
	
	/**
	 * 
	 * @param request
	 * @return
	 * @throws GCUBEFault
	 */
	public TimeseriesHistory getHistory(VOID request) throws GCUBEFault{
		TSHistory tsHistory;
		
		//starting retrieving timeseries history
		LinkedList<TimeseriesHistoryItem> historyItemsList= new LinkedList<TimeseriesHistoryItem>();
		try{
			tsHistory= getResource().getHistory();
		}catch(Exception e){logger.error("impossible to find the resource",e); throw new GCUBEFault(e);}
		while (tsHistory.getParentType()==ParentType.TS){
			TsHistoryOps[] tsOperations= new TsHistoryOps[tsHistory.getHistoryList().size()];
			for (int i=0; i<tsHistory.getHistoryList().size(); i++){
				TSHistoryItem tempItem=tsHistory.getHistoryList().get(i);
				Calendar tsItemCalendar= Calendar.getInstance();
				tsItemCalendar.setTime(tempItem.getDate());
				tsOperations[i]=new TsHistoryOps(tsItemCalendar,tempItem.getHumanDescription(),tempItem.getOpType());		
			}
			boolean parentAlive= true;
			TimeSeriesResource tsResource=null;
			try{
				tsResource=(TimeSeriesResource)TimeSeriesContext.getPortTypeContext().getWSHome().find(TimeSeriesContext.getPortTypeContext().makeKey(tsHistory.getParentId()));
			}catch(ResourceException re){logger.debug("the parent TS with id "+tsHistory.getParentId()+" is not alive!!");parentAlive=false;}
			
			Calendar tsCalendar= Calendar.getInstance();
			tsCalendar.setTime(tsHistory.getHistoryCreationDate());
			historyItemsList.add(new TimeseriesHistoryItem(tsHistory.getAuthor(),tsCalendar,tsOperations,parentAlive,tsHistory.getParentId(),tsHistory.getParentName(), org.gcube.contentmanagement.timeseriesservice.stubs.types.ParentType.fromString(tsHistory.getParentType().toString())));
			try {
				tsHistory=parentAlive?tsResource.getHistory():TSHistory.retrieveHistory(tsHistory.getParentId());
			} catch (Exception e) {
				logger.error("unexpected error preparing history response",e);
				throw new GCUBEFault("unexpected error preparing history response");
			}
		}
		
		CurationHistoryItem curationHistoryItem=null;
		try{
			CurationResource curResource=null;
			boolean parentCurAlive=true;
			try{
				String parentId= tsHistory.getParentId();
				logger.trace("the curation Id is "+parentId);
				curResource=(CurationResource)CurationContext.getPortTypeContext().getWSHome().find(CurationContext.getPortTypeContext().makeKey(parentId));
			}catch(ResourceException re){logger.debug("the parent Curation with id "+tsHistory.getParentId()+" is not alive!!");parentCurAlive=false;}	

			Calendar tsCalendar= Calendar.getInstance();
			tsCalendar.setTime(tsHistory.getHistoryCreationDate());
			historyItemsList.add(new TimeseriesHistoryItem(tsHistory.getAuthor(),tsCalendar, new TsHistoryOps[0],parentCurAlive,tsHistory.getParentId(),tsHistory.getParentName(), org.gcube.contentmanagement.timeseriesservice.stubs.types.ParentType.fromString(tsHistory.getParentType().toString())));

			//starting retrieving curation history
			CurationHistory curationHistory;
			try{
				curationHistory=parentCurAlive?curResource.getHistory():CurationHistory.retrieveHistory(tsHistory.getParentId());
			} catch (Exception e) {
				logger.error("unexpected error preparing history response",e);
				throw new GCUBEFault("unexpected error preparing history response");
			}
			CurationHistoryOps[] curationItemList=new CurationHistoryOps[tsHistory.getItemLimit()];
			for (int i=0; i<tsHistory.getItemLimit(); i++){
				org.gcube.contentmanagement.timeseriesservice.impl.history.CurationHistoryItem curationItem=curationHistory.getItems().get(i);
				Calendar curItemCalendar= Calendar.getInstance();
				curItemCalendar.setTime(curationItem.getDate());
				curationItemList[i]=new CurationHistoryOps(curItemCalendar,curationItem.getHumanDescription(), curationItem.getOpType().toString());
			}

			boolean parentImporterAlive=true;
			try{
				ImportContext.getPortTypeContext().getWSHome().find(ImportContext.getPortTypeContext().makeKey(curationHistory.getCsvParentId()));
			}catch(ResourceException re){logger.debug("the parent csv with id "+curationHistory.getCsvParentId()+" is not alive!!");parentImporterAlive=false;}	

			Calendar curCalendar= Calendar.getInstance();
			curCalendar.setTime(curationHistory.getDate());

			curationHistoryItem= new CurationHistoryItem(curationHistory.getCreator(),curCalendar,curationItemList,parentImporterAlive,curationHistory.getCsvParentId(),curationHistory.getParentTitle());

			//TODO: add csv History
		}catch (Exception e) {
			logger.error("error retrieve curation history",e);
			throw new GCUBEFault(e);
		}
		return new TimeseriesHistory(null, curationHistoryItem, historyItemsList.toArray(new TimeseriesHistoryItem[0]));
	}

	/**
	 * 
	 * @param request
	 * @return
	 * @throws GCUBEFault
	 */
	public CompatibleColumnsMappingArray getCompatibleColumns(CompatibleColumnsRequest request) throws GCUBEFault{
		logger.debug("call to getCompatibleColumns with request: "+request.getId()+" and user "+request.getUser());		
		TimeSeriesResource sourceResource;
		TimeSeriesResource destinationResource;
		try{
			sourceResource=getResource();
			destinationResource= (TimeSeriesResource)TimeSeriesContext.getPortTypeContext().getWSHome().find(TimeSeriesContext.getPortTypeContext().makeKey(request.getId()));
		}catch(Exception e){
			logger.error("impossible to retrieve resource",e);
			throw new GCUBEFault(e,"error retrieving resource");			
		}
		//check compatibility between the selected timeseries
		if (!Union.checkCompatibility(sourceResource.getCurrentColumnsDefinition(request.getUser()), destinationResource.getCurrentColumnsDefinition(request.getUser()))) throw new GCUBEFault("Selected TimeSeries are not compatible");
		try{
			return new CompatibleColumnsMappingArray(Union.getCompatibileColumns(sourceResource.getCurrentColumnsDefinition(request.getUser()), destinationResource.getCurrentColumnsDefinition(request.getUser())));
		}catch (Exception e){
			logger.error("error getting compatible columns",e);
			throw new GCUBEFault(e,"error getting compatible columns");
		}
	}
	
	public StringArray getDistinctValues(DistinctValuesRequest request) throws GCUBEFault{
		try{
			int startElement=-1;
			int offset = -1;
			if (request.getLimit()!=null){
				startElement= request.getLimit().getLowerLimit();
				offset = request.getLimit().getUpperLimit();
			}
			StringArray arrayString= new StringArray(getResource().getDistinctValues(request.getFieldId(), startElement, offset));
			logger.trace("returning "+arrayString.getItems().length+" distinct values");
			return arrayString;
		}catch (Exception e) {
			logger.error("error getting distinct values",e);
			throw new GCUBEFault("error getting distinct values");
		}
	}
	
	public String getChartData(ChartDataRequest req) throws GCUBEFault {
		try{
			return getResource().getChartData(req.getMaxDimension(), req.getXDimension(), req.getYDimension(),req.getGroupDimension(), req.getAttributeColumn(), req.getFilters().getItems(), req.getAreas().getItems());
		}catch (Exception e) {
			logger.error("error getting chart data",e);
			throw new GCUBEFault("error getting chart data");
		}
	}
	
	public String getAreaTransformationResult(GetAreaTransformationResultRequest request) throws GCUBEFault{
		try{
			AFilter filter = new FAOAreaFilter(getResource().getTable().getTableName() ,request.getAggregationColumn(), request.getQuantitiesColumn());
			return getResource().transform(filter , request.getUser());
		}catch (Exception e) {
			logger.error("error transforming timeseries",e);
			throw new GCUBEFault("error transforming timeseries");
		}
	}
	
	public String getSpaceTransformationResult(GetSpaceTransformationResultRequest request) throws GCUBEFault{
		try{
			AFilter filter =  new SpaceFilter(getResource().getTable().getTableName() ,request.getAggregationColumn(), request.getQuantitiesColumn());
			return getResource().transform(filter , request.getUser());
		}catch (Exception e) {
			logger.error("error transforming timeseries",e);
			throw new GCUBEFault("error transforming timeseries");
		}
	}
	
	public String getTimeCountriesTransformationResult(GetTimeCountriesTransformationResultRequest request) throws GCUBEFault{
		try{
			AFilter filter = new CountryYearFilter(getResource().getTable().getTableName() ,request.getAggregationColumn(), request.getInformationColumn(), request.getQuantitiesColumn());
			return getResource().transform(filter, request.getUser());
		}catch (Exception e) {
			logger.error("error transforming timeseries",e);
			throw new GCUBEFault("error transforming timeseries");
		}
	}
	
	public String getSpaceSpeciesTransformationResult(GetSpaceSpeciesTransformationResultRequest request) throws GCUBEFault{
		try{
			AFilter filter =  new SpeciesFilter(getResource().getTable().getTableName() ,request.getAggregationColumn(), request.getInformationColumn(), request.getQuantitiesColumn());
			return getResource().transform(filter , request.getUser());
		}catch (Exception e) {
			logger.error("error transforming timeseries",e);
			throw new GCUBEFault("error transforming timeseries");
		}
	}
	
	public String getFishingMontlyEffort(GetFishingMontlyEffortRequest request) throws GCUBEFault{
		try{
			return getResource().getFishingMontlyEffort(request);
		}catch (Exception e) {
			logger.error("error transforming timeseries",e);
			throw new GCUBEFault("error transforming timeseries");
		}
	}
	
	public GetTransformationStatusResponse getTransformationStatus(String groupName) throws GCUBEFault{
		try{
			return getResource().transformationStatus(groupName);
		}catch (Exception e) {
			logger.error("error transformation status for groupName "+groupName,e);
			throw new GCUBEFault("error transformation status for groupName "+groupName);
		}
	}
	
	
}
