package org.gcube.contentmanagement.timeseriesservice.impl.importer;

import java.io.Serializable;
import java.sql.Timestamp;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.gcube.common.core.scope.GCUBEScope;
import org.gcube.common.core.state.GCUBEWSResourceKey;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.common.dbinterface.Specification;
import org.gcube.common.dbinterface.persistence.ObjectPersistency;
import org.gcube.common.dbinterface.persistence.annotations.FieldDefinition;
import org.gcube.common.dbinterface.persistence.annotations.TableRootDefinition;
import org.gcube.common.dbinterface.pool.DBSession;
import org.gcube.common.dbinterface.queries.DropTable;
import org.gcube.common.dbinterface.queries.Select;
import org.gcube.common.dbinterface.tables.SimpleTable;
import org.gcube.contentmanagement.timeseriesservice.impl.context.ImportContext;
import org.gcube.contentmanagement.timeseriesservice.impl.context.ServiceContext;
import org.gcube.contentmanagement.timeseriesservice.impl.thread.InsertDenomalizedT;
import org.gcube.contentmanagement.timeseriesservice.impl.thread.InsertNormalizedT;
import org.gcube.contentmanagement.timeseriesservice.impl.utils.Constants;
import org.gcube.contentmanagement.timeseriesservice.stubs.DenormalizedImportRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.ImportRequest;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.ColumnDefinition;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.DataType;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.EntryType;
import org.gcube.contentmanagement.timeseriesservice.stubs.types.Status;

/**
 * 
 * @author lucio
 *
 */
@TableRootDefinition
public class ImporterItem implements Serializable {

	private static GCUBELog logger= new GCUBELog(ImporterItem.class);
	
	private GCUBEWSResourceKey resourceKey;
	
	@FieldDefinition(specifications={Specification.NOT_NULL}, precision={10})
	private Status state= Status.Open;

	@FieldDefinition(specifications={Specification.NOT_NULL}, precision={150})
	private String title;
	
	@FieldDefinition(specifications={Specification.NOT_NULL, Specification.PRIMARY_KEY}, precision={50})
	private String id;
	
	@FieldDefinition(specifications={Specification.NOT_NULL}, precision={150})
	private String creator;
	
	@FieldDefinition(specifications={Specification.NOT_NULL})
	private String description;
	
	@FieldDefinition(specifications={Specification.NOT_NULL}, precision={150})
	private String publisher;
	
	@FieldDefinition(specifications={Specification.NOT_NULL}, precision={150})
	private String type;
	
	@FieldDefinition(specifications={Specification.NOT_NULL}, precision={255})
	private String source;
	
	@FieldDefinition(specifications={Specification.NOT_NULL})
	private String rights;
	
	@FieldDefinition()
	private List<String> fieldNames=null;
	
	@FieldDefinition()
	private SimpleTable table;
	
	@FieldDefinition(specifications={Specification.NOT_NULL})
	private Timestamp date;
	
	@FieldDefinition(specifications={Specification.NOT_NULL}, precision={50})
	private String encoding="";
	
	private int exstimatedLines=0;
	private int importProgress=0;
	
	@FieldDefinition(specifications={Specification.NOT_NULL}, precision={10})
	private int totalLines;
	
	@FieldDefinition(specifications={Specification.NOT_NULL}, precision={150})
	private String scope;
	
	public ImporterItem(GCUBEWSResourceKey key, GCUBEScope scope){
		this.date= new Timestamp(System.currentTimeMillis());
		this.scope = scope.toString();
		this.resourceKey= key;
	}
	
	@SuppressWarnings("unused")
	private ImporterItem(){}

	public GCUBEWSResourceKey getResourceKey(){
		if (resourceKey==null)
			this.resourceKey = ImportContext.getPortTypeContext().makeKey(this.id);
		return this.resourceKey;
	}
	
	/**
	 * 
	 */
	private static final long serialVersionUID = 1L;

	/**
	 * 
	 * @param state
	 */
	public void setStatus(Status state) {
		this.state = state;
	}
	
	/**
	 * 
	 * @return
	 */
	public Status getStatus(){
		return this.state;
	}

	
	/**
	 * 
	 * @return
	 */
	public String getId() {
		return id;
	}

	/**
	 * 
	 * @param id
	 */
	public void setId(String id) {
		this.id = id;
	}

	
	/**
	 * 
	 * @return
	 */
	public String getDescription() {
		return description;
	}

	
	/**
	 * 
	 * @param description
	 */
	public void setDescription(String description) {
		this.description = description;
	}

	
	
	/**
	 * @return the encoding
	 */
	public String getEncoding() {
		return encoding;
	}

	/**
	 * @param encoding the encoding to set
	 */
	public void setEncoding(String encoding) {
		this.encoding = encoding;
	}

	/**
	 * 
	 * @return
	 */
	public ColumnDefinition[] getColumnsDefinition() {
		ColumnDefinition[] columnsDefinition = new ColumnDefinition[this.fieldNames.size()];
		int i =0;
		for (String field: this.fieldNames){
			ColumnDefinition columnDefinition= new ColumnDefinition(EntryType.Undefined, null,null, "field"+i,  null, field.compareTo("")==0?"field"+i:field, DataType.Text);
			columnsDefinition[i++]= columnDefinition;
		}
		return columnsDefinition;
	}
		
	/**
	 * 
	 * @throws Exception
	 */
	public void store() throws Exception{
		ObjectPersistency<ImporterItem> persistency = ObjectPersistency.get(ImporterItem.class);
		if (!persistency.existsKey(id))
			persistency.insert(this);
		else persistency.update(this);
	}
	
	/**
	 * 
	 * @param refLocator
	 * @param numElementsPerRow
	 * @throws Exception
	 */
	public void storeData(ImportRequest request) throws Exception{
		if (this.getStatus()==Status.Close){
			logger.warn("cannot store data, the possibility to import data is CLOSED");
			throw new Exception("cannot store data, the possibility to import data is CLOSED");
		}
		//storing data
		InsertNormalizedT idt=new InsertNormalizedT(request.getRsLocator(),Constants.getImportTableDataName(this.getId()),request.isHasHeader(), request.getDelimiter().charAt(0), request.getEncoding(), request.getFieldsMask(), this);
		ServiceContext.getContext().setScope(idt, ServiceContext.getContext().getScope());
		idt.start();
	}
	
	
	/**
	 * 
	 * @param refLocator
	 * @param numElementsPerRow
	 * @throws Exception
	 */
	public void storeDenormalizedData(DenormalizedImportRequest request) throws Exception{
		if (this.getStatus()==Status.Close){
			logger.warn("cannot store data, the possibility to import data is CLOSED");
			throw new Exception("cannot store data, the possibility to import data is CLOSED");
		}
		//storing data
		InsertDenomalizedT idt=new InsertDenomalizedT(request.getRsLocator(),Constants.getImportTableDataName(this.getId()), request.isHasHeader(), request.getDelimiter().charAt(0), request.getEncoding(), request.getFieldsMask(), request.getFieldsAttributeMask(), request.getAttributeLabel(), request.getValueLabel(), this);
		ServiceContext.getContext().setScope(idt, ServiceContext.getContext().getScope());
		idt.start();
	}
	
	/**
	 * closes the possibility to import data
	 */
	public void setClosed(){
		this.setStatus(Status.Close);
	}
	
	public void setErrorState(){
		this.setStatus(Status.Error);
	}
	
	 /**
	 * 
	 * 
	 * @return
	 * @throws Exception
	 */
	public String getDataAsJson(Select query) throws Exception{
		return query.getResultAsJSon(true);
	}
	
	
	 /**
	 * 
	 * @param lowerLimit
	 * @param upperLimit
	 * @return
	 * @throws Exception
	 */
	public String getData() throws Exception{
		/*
		ResultSet resultSet=session.executeQuery("SELECT * FROM "+Constants.getImportTableDataName(this.id)+";");
		String toReturn= Util.toJSon(resultSet);
		session.close();
		return toReturn;*/
		return null;
	}
	
	/**
	 * 
	 * @throws Exception
	 */
	public void remove() throws Exception{
		DBSession session= DBSession.connect();
		try{
			ObjectPersistency.get(ImporterItem.class).deleteByKey(this.getId());
		}catch(Exception e1){logger.error("error dropping entry eith id "+id,e1);}	
		try{
			DropTable drop= DBSession.getImplementation(DropTable.class);
			drop.setTableName(Constants.getImportTableDataName(id));
			drop.execute(session);
		}catch(Exception e){ logger.error("error dropping table "+Constants.getImportTableDataName(id));}
		session.release();
	}
	
	public int count() throws Exception{
		SimpleTable tempResourceTable= new SimpleTable(Constants.getImportTableDataName(id));
		tempResourceTable.initializeCount();
		return tempResourceTable.getCount();
	}

	public List<String> getFieldNames() {
		return fieldNames;
	}

	public void setFieldNames(List<String> fieldNames) {
		this.fieldNames = fieldNames;
	}

	public Map<String, String> getFieldMapping(){
		if (fieldNames!=null){
			Map<String, String> mapping= new HashMap<String, String>();
			for (int i =0; i<fieldNames.size(); i++)
				mapping.put(fieldNames.get(i), "field"+i);
			return mapping;
		} else return null;
	}

	public String getTitle() {
		return title;
	}

	public void setTitle(String title) {
		this.title = title;
	}

	public String getCreator() {
		return creator;
	}

	public void setCreator(String creator) {
		this.creator = creator;
	}

	public String getPublisher() {
		return publisher;
	}

	public void setPublisher(String publisher) {
		this.publisher = publisher;
	}

	public String getType() {
		return type;
	}

	public void setType(String type) {
		this.type = type;
	}

	public String getSource() {
		return source;
	}

	public void setSource(String source) {
		this.source = source;
	}

	public String getRights() {
		return rights;
	}

	public void setRights(String rights) {
		this.rights = rights;
	}

	public int getImportProgress() {
		return importProgress;
	}

	public void setImportProgress(int importProgress) {
		this.importProgress = importProgress;
	}

	public int getTotalLines() {
		return totalLines;
	}

	public void setTotalLines(int totalLines) {
		this.totalLines = totalLines;
	}

	public int getExstimatedLines() {
		return exstimatedLines;
	}

	public void setExstimatedLines(int exstimatedLine) {
		this.exstimatedLines = exstimatedLine;
	}

	public SimpleTable getTable() {
		return table;
	}

	public void setTable(SimpleTable table) {
		this.table = table;
	}

	public Timestamp getDate() {
		return date;
	}

	public void setDate(Timestamp date) {
		this.date = date;
	}

	public String getScope() {
		return scope;
	}

	public void setScope(String scope) {
		this.scope = scope;
	}
}
