package org.gcube.contentmanagement.codelistmanager.managers.handlers;

import java.sql.ResultSet;
import java.util.ArrayList;
import java.util.Hashtable;

import org.gcube.common.dbinterface.CastObject;
import org.gcube.common.dbinterface.Condition;
import org.gcube.common.dbinterface.attributes.AggregatedAttribute;
import org.gcube.common.dbinterface.attributes.AggregationFunctions;
import org.gcube.common.dbinterface.attributes.BooleanAttribute;
import org.gcube.common.dbinterface.attributes.SimpleAttribute;
import org.gcube.common.dbinterface.conditions.ListSelect;
import org.gcube.common.dbinterface.conditions.NotInOperator;
import org.gcube.common.dbinterface.conditions.ORCondition;
import org.gcube.common.dbinterface.conditions.OperatorCondition;
import org.gcube.common.dbinterface.pool.DBSession;
import org.gcube.common.dbinterface.queries.Select;
import org.gcube.common.dbinterface.queries.Update;
import org.gcube.common.dbinterface.queries.alters.ModifyColumnType;
import org.gcube.common.dbinterface.tables.SimpleTable;
import org.gcube.common.dbinterface.types.Type;
import org.gcube.common.dbinterface.utils.Utility;
import org.gcube.contentmanagement.codelistmanager.entities.CodeList;
import org.gcube.contentmanagement.codelistmanager.entities.TableField;
import org.gcube.contentmanagement.codelistmanager.entities.TableField.ColumnType;
import org.gcube.contentmanagement.codelistmanager.exception.ColumnTypeNotSelectableException;
import org.gcube.contentmanagement.codelistmanager.exception.NullValuesOnCastException;
import org.gcube.contentmanagement.codelistmanager.exception.ValueNotFoundException;
import org.gcube.contentmanagement.codelistmanager.util.ColumnReference;
import org.gcube.contentmanagement.codelistmanager.util.Constants;
import org.gcube.contentmanagement.codelistmanager.util.csv.ImportUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HierarchicalCodeListHandler implements CodeListCurationHandler{

	
	private static final Logger logger = LoggerFactory.getLogger(HierarchicalCodeListHandler.class);
	
	private static HierarchicalCodeListHandler codelistHandler = new HierarchicalCodeListHandler();
		
	public static HierarchicalCodeListHandler get(){
		return codelistHandler;
	}


	public Integer[] checkInvalidValues(String fieldId, ColumnReference columnReference, SimpleTable table,
			Hashtable<String, TableField> fieldLabelMapping) throws Exception{
		DBSession session=null;
		logger.debug("checking rows for columnReference "+columnReference.getCodelistReferenceId()+" "+columnReference.getCodelistReferenceName()+"  "+columnReference.getType().toString());
		if (columnReference.getType()!=ColumnType.HLParentCode && 
				columnReference.getType()!=ColumnType.HLChildCode) throw new Exception(); //TODO new Exception
		try{
			
			CodeList refCodelist = CodeList.get(columnReference.getCodelistReferenceId());
			
			Type type = new Type(refCodelist.getLabelFieldMapping().get(refCodelist.getCodeColumnId()).getDataType());
			
			session = DBSession.connect();
			Select select = DBSession.getImplementation(Select.class);
			select.setAttributes(new SimpleAttribute(Constants.ID_LABEL));
			CastObject cast= Utility.getCast(new SimpleAttribute(fieldId), type);
			cast.setUseCastFunction(true);
			
			Condition cond = new OperatorCondition<CastObject, Object>(cast, null," is ");
							
			Select opSelect= DBSession.getImplementation(Select.class);
			opSelect.setTables(refCodelist.getTable());
			opSelect.setAttributes(new SimpleAttribute(refCodelist.getCodeColumnId()));
			cond = new ORCondition(cond, new NotInOperator(new SimpleAttribute(cast.getCast()), new ListSelect(opSelect)));
						
			select.setFilter(cond);
			select.setTables(table);
			logger.debug(select.getExpression());
			ResultSet result = select.getResults(session);
			ArrayList<Integer> toReturn= new ArrayList<Integer>();
			while (result.next())
				toReturn.add(result.getInt(1));
			return toReturn.toArray(new Integer[toReturn.size()]);
		}catch (Exception e) {
			logger.error("error evaluating the query",e);
			throw e;
		}finally{
			if (session!=null) session.release();
		}
	}

	public Integer[] checkInvalidValues(String fieldId, Type type,
			SimpleTable table, Hashtable<String, TableField> fieldLabelMapping)
			throws Exception {
		DBSession session=null;
		try{
			
			session = DBSession.connect();
			Select select = DBSession.getImplementation(Select.class);
			select.setAttributes(new SimpleAttribute(Constants.ID_LABEL));
			CastObject cast= Utility.getCast(new SimpleAttribute(fieldId), type);
			cast.setUseCastFunction(true);
			
			Condition cond = new OperatorCondition<CastObject, Object>(cast, null," is ");
			
			if (fieldLabelMapping.get(fieldId).getColumnReference().getType()==ColumnType.HLParentCode || 
					fieldLabelMapping.get(fieldId).getColumnReference().getType()==ColumnType.HLChildCode){
				CodeList refCodelist = CodeList.get(fieldLabelMapping.get(fieldId).getColumnReference().getCodelistReferenceId());
				Type externalType = new Type(refCodelist.getLabelFieldMapping().get(refCodelist.getCodeColumnId()).getDataType());
				Select opSelect= DBSession.getImplementation(Select.class);
				opSelect.setTables(refCodelist.getTable());
				opSelect.setAttributes(new SimpleAttribute(refCodelist.getCodeColumnId()));
				cast= Utility.getCast(new SimpleAttribute(fieldId), externalType);
				cast.setUseCastFunction(true);
				cond = new ORCondition(cond, new NotInOperator(new SimpleAttribute(cast.getCast()), new ListSelect(opSelect)));
			}
			
			select.setFilter(cond);
			select.setTables(table);
			logger.debug(select.getExpression());
			ResultSet result = select.getResults(session);
			ArrayList<Integer> toReturn= new ArrayList<Integer>();
			while (result.next())
				toReturn.add(result.getInt(1));
			return toReturn.toArray(new Integer[toReturn.size()]);
		}catch (Exception e) {
			logger.error("error evaluating the query",e);
			throw e;
		}finally{
			if (session!=null) session.release();
		}
	}
	
	
	private boolean isRelationField(String fieldId, Hashtable<String, TableField> fieldLabelMapping){
		return fieldLabelMapping.get(fieldId).getColumnReference().getType()==ColumnType.HLParentCode || fieldLabelMapping.get(fieldId).getColumnReference().getType()==ColumnType.HLChildCode;
	}
	
	public boolean replaceValue(String fieldId, String newValue, int rowId,
			SimpleTable table, Hashtable<String, TableField> fieldLabelMapping)
			throws Exception {
		if (isRelationField(fieldId, fieldLabelMapping)) return setParent(fieldId, newValue, rowId, table, fieldLabelMapping);
		else {
			Type type = table.getFieldsMapping().get(fieldId);
			Select select = DBSession.getImplementation(Select.class);
			CastObject cast= Utility.getCast(newValue, type);
			cast.setUseCastFunction(true);
			select.setAttributes(new BooleanAttribute("field", new OperatorCondition<CastObject, Object>(cast, null," IS NOT ") ));
			DBSession session = null;
			try{
				session = DBSession.connect();
				ResultSet rs = select.getResults(session, false);
				rs.next();
				if (!rs.getBoolean(1)) return false;
				
				int[] tempLength = fieldLabelMapping.get(fieldId).getLength();
				if (newValue.length()>tempLength[0] || ImportUtil.getAfterDotLength(newValue)>tempLength[1]){
					if (newValue.length()>tempLength[0]) tempLength[0] = newValue.length();
					if (ImportUtil.getAfterDotLength(newValue)>tempLength[1]) tempLength[1] = ImportUtil.getAfterDotLength(newValue);
					type.setPrecision(tempLength);			
					changeColumnDataType(fieldId, type, table, fieldLabelMapping);
				}
				
				Update update = DBSession.getImplementation(Update.class);
				update.setTable(table);
				update.setOperators(new OperatorCondition<SimpleAttribute, CastObject>(new SimpleAttribute(fieldId), cast, "="));
				update.setFilter(new OperatorCondition<SimpleAttribute, Integer>(new SimpleAttribute(Constants.ID_LABEL), rowId, "="));
				update.execute(session);
				return update.getAffectedLines()>0;
			}catch (Exception e) {
				logger.error("an error occuring trying to replace value",e);
				return false;
			}finally{
				if (session!=null)session.release();
			}
		}
	}

	private boolean setParent(String fieldId, String newValue, int rowId,
			SimpleTable table, Hashtable<String, TableField> fieldLabelMapping) throws ColumnTypeNotSelectableException, ValueNotFoundException, Exception{
	
		CodeList refCodelist = CodeList.get(fieldLabelMapping.get(fieldId).getColumnReference().getCodelistReferenceId());
		
		Select select = DBSession.getImplementation(Select.class);
		select.setTables(table);
		select.setAttributes(new AggregatedAttribute("*",AggregationFunctions.COUNT));
		CastObject castObject= Utility.getCast(newValue, refCodelist.getTable().getFieldsMapping().get(refCodelist.getCodeColumnId()));
		select.setFilter(new OperatorCondition<SimpleAttribute, CastObject>(new SimpleAttribute(refCodelist.getCodeColumnId()),castObject, "="));
		DBSession session = null;
		try{
			session = DBSession.connect();
			ResultSet result = select.getResults(session);
			result.next();
			if (result.getInt(1)==0) throw new ValueNotFoundException();
			
			//updating the table
			Update update= DBSession.getImplementation(Update.class);
			update.setTable(table);
			update.setOperators(new OperatorCondition<SimpleAttribute, CastObject >(new SimpleAttribute(fieldId),castObject,"="));
			update.setFilter(new OperatorCondition<SimpleAttribute, Integer>(new SimpleAttribute(Constants.ID_LABEL), rowId,"="));
			update.execute(session);
			
			if (update.getAffectedLines()>0) return true;
			else return false;
		}finally{
			if (session!=null) session.release();
		}
		
	}


	/**
	 * 
	 * @param fieldId
	 * @param columnType
	 * @throws ParentNotSelectableException
	 */
	public void changeColumnType(String fieldId, ColumnType columnType, SimpleTable table,
			Hashtable<String, TableField> fieldLabelMapping, String ... relatedCLId ) throws ColumnTypeNotSelectableException, Exception{
		logger.debug("selected column type is "+columnType.toString());
		if (columnType == ColumnType.HLChildCode || columnType == ColumnType.HLParentCode ) {
			if(relatedCLId.length==0) throw new ColumnTypeNotSelectableException();
			logger.debug("related codelist selected is"+relatedCLId[0]);
			CodeList referenceCL = CodeList.get(relatedCLId[0]);
			fieldLabelMapping.get(fieldId).setColumnReference(new ColumnReference(columnType, referenceCL.getName(), referenceCL.getId()));
		} else if (columnType == ColumnType.Undefined){
			fieldLabelMapping.get(fieldId).setColumnReference(new ColumnReference(columnType));
		}else throw new ColumnTypeNotSelectableException();
	}

	
	public boolean changeColumnDataType(String fieldId, Type type,
			SimpleTable table, Hashtable<String, TableField> fieldLabelMapping)
			throws NullValuesOnCastException {
		DBSession session=null;
		int nullValues;
		try{
			nullValues= checkInvalidValues(fieldId, type, table, fieldLabelMapping).length;
		}catch (Exception e) {
			return false;
		}
		if (nullValues>0) throw new NullValuesOnCastException(); 
		try{
			session = DBSession.connect();
			ModifyColumnType changeColumnType= DBSession.getImplementation(ModifyColumnType.class);
			changeColumnType.setColumn(new SimpleAttribute(fieldId));
			changeColumnType.setNewType(type);
			changeColumnType.setTable(table);
			changeColumnType.setUseCast(true);
			//System.out.println(changeColumnType.getExpression());
			changeColumnType.execute(session);
			fieldLabelMapping.get(fieldId).setDataType(type.getType());
			fieldLabelMapping.get(fieldId).setLength(type.getPrecisionArray());
			table.initializeFieldMapping(session);
		}catch (Exception e) {
			logger.error("error changeing column type ",e);
			return false;
		}finally{
			if (session!=null) session.release();
		}
		return true;
	}


	public boolean checkRelationMatch(String fieldId, SimpleTable table,
			Hashtable<String, TableField> fieldLabelMapping) {
		DBSession session= null;
		try{
			TableField relationCode= null;
		
			relationCode = fieldLabelMapping.get(fieldId);
			
			if (relationCode==null) return true;
			
			session = DBSession.connect();
			Select select = DBSession.getImplementation(Select.class);
			select.setAttributes(new AggregatedAttribute("*", AggregationFunctions.COUNT));
			
			CodeList refCodelist = CodeList.get(fieldLabelMapping.get(fieldId).getColumnReference().getCodelistReferenceId());
			
			Type externalType = new Type(refCodelist.getLabelFieldMapping().get(refCodelist.getCodeColumnId()).getDataType());
			
			Select subSelect = DBSession.getImplementation(Select.class);
			
			subSelect.setTables(refCodelist.getTable());
			subSelect.setAttributes(new SimpleAttribute(Utility.getCast(new SimpleAttribute(refCodelist.getCodeColumnId()), externalType).getCast()));
			
			refCodelist.getLabelFieldMapping().get(refCodelist.getCodeColumnId());
			CastObject cast= Utility.getCast(new SimpleAttribute(relationCode.getId()), externalType);
			cast.setUseCastFunction(true);
			
			select.setTables(table);
			select.setFilter(new NotInOperator(new SimpleAttribute(cast.getCast()),new ListSelect(subSelect)));
		
			ResultSet rs =select.getResults(session, false);
			rs.next();
			if (rs.getInt(1)>0) return false;
		}catch (Exception e) {
			
			logger.debug("error matching parents",e);
			return false;
		}finally{
			if (session!=null) session.release();
		}
		return true;
	}

	public boolean isMappingFinished(SimpleTable table,
			Hashtable<String, TableField> fieldLabelMapping) {
		for (TableField tf : fieldLabelMapping.values())
			if (!((tf.getColumnReference().getType()== ColumnType.HLParentCode || tf.getColumnReference().getType()== ColumnType.HLChildCode) &&
				checkRelationMatch(tf.getId(), table, fieldLabelMapping))) return false;
		return true;
	}

}
