package org.gcube.common.storagehub.client.proxies;

import java.io.InputStream;
import java.net.URL;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.ws.rs.client.Entity;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.MultivaluedHashMap;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.core.Response;

import org.gcube.common.clients.Call;
import org.gcube.common.clients.delegates.ProxyDelegate;
import org.gcube.common.gxrest.request.GXWebTargetAdapterRequest;
import org.gcube.common.gxrest.response.inbound.GXInboundResponse;
import org.gcube.common.storagehub.client.StreamDescriptor;
import org.gcube.common.storagehub.model.Metadata;
import org.gcube.common.storagehub.model.acls.ACL;
import org.gcube.common.storagehub.model.acls.AccessType;
import org.gcube.common.storagehub.model.annotations.RootNode;
import org.gcube.common.storagehub.model.exceptions.BackendGenericError;
import org.gcube.common.storagehub.model.exceptions.StorageHubException;
import org.gcube.common.storagehub.model.items.GCubeItem;
import org.gcube.common.storagehub.model.items.Item;
import org.gcube.common.storagehub.model.service.ItemList;
import org.gcube.common.storagehub.model.service.ItemWrapper;
import org.gcube.common.storagehub.model.service.Version;
import org.gcube.common.storagehub.model.service.VersionList;
import org.gcube.common.storagehub.model.types.ACLList;
import org.glassfish.jersey.media.multipart.FormDataMultiPart;
import org.glassfish.jersey.media.multipart.MultiPartFeature;

public class DefaultItemManager implements ItemManagerClient {

	private final ProxyDelegate<GXWebTargetAdapterRequest> delegate;

	private static final String CONTENT_DISPOSITION= "Content-Disposition";
	private static final String CONTENT_TYPE= "Content-Type";

	public DefaultItemManager(ProxyDelegate<GXWebTargetAdapterRequest> config){
		this.delegate = config;
	}


	@Override
	public StreamDescriptor resolvePublicLink(String identifier) throws StorageHubException{
		Call<GXWebTargetAdapterRequest, StreamDescriptor> call = new Call<GXWebTargetAdapterRequest, StreamDescriptor>() {
			@Override
			public StreamDescriptor call(GXWebTargetAdapterRequest manager) throws StorageHubException, Exception {
				Objects.requireNonNull(identifier, "id cannot be null");

				GXWebTargetAdapterRequest myManager = manager.path("publiclink").path(identifier);
				Map<String, Object[]> params = new HashMap<>();

				GXInboundResponse response = myManager.queryParams(params).get();

				if (response.isErrorResponse()) {
					if (response.hasException()) 
						throw response.getException();
					else 
						throw new BackendGenericError();
				}

				Response resp = response.getSource();

				return createStreamDescriptor(resp);
			}

		};
		try {
			StreamDescriptor result = delegate.make(call);
			return result;
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}

	@Override
	@Deprecated
	public List<? extends Item> getChildren(String id, Class<? extends Item> onlyOfType, String ... excludeNodes) throws StorageHubException {
		return getChildren(id,  onlyOfType, false, excludeNodes);
	}

	@Override
	public List<? extends Item> getChildren(String id, Class<? extends Item> onlyOfType, boolean showHidden, String ... excludeNodes) throws StorageHubException{
		Call<GXWebTargetAdapterRequest, ItemList> call = new Call<GXWebTargetAdapterRequest, ItemList>() {
			@Override
			public ItemList call(GXWebTargetAdapterRequest manager) throws StorageHubException, Exception {
				Objects.requireNonNull(id, "id cannot be null");
				GXWebTargetAdapterRequest myManager = manager.path(id).path("children");
				Map<String, Object[]> params = new HashMap<>();
				if (excludeNodes !=null && excludeNodes.length>0)
					params.put("exclude",excludeNodes);

				if (onlyOfType!=null)
					params.put("onlyType", new Object[] {resolveNodeType(onlyOfType)});
				params.put("showHidden", new Object[] {showHidden});

				GXInboundResponse response = myManager.queryParams(params).get();

				if (response.isErrorResponse()) {
					if (response.hasException()) 
						throw response.getException();
					else 
						throw new BackendGenericError();
				}

				ItemList items = response.getSource().readEntity(ItemList.class);

				return items;
			}
		};
		try {
			ItemList result = delegate.make(call);
			return result.getItemlist();
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}


	@Override
	public List<? extends Item> getChildren(String id, int start, int limit, boolean showHidden, Class<? extends Item> onlyOfType, String... excludeNodes) throws StorageHubException{
		Call<GXWebTargetAdapterRequest, ItemList> call = new Call<GXWebTargetAdapterRequest, ItemList>() {
			@Override
			public ItemList call(GXWebTargetAdapterRequest manager) throws Exception {
				Objects.requireNonNull(id, "id cannot be null");
				try {
					GXWebTargetAdapterRequest myManager = manager.path(id).path("children").path("paged");
					Map<String, Object[]> params = new HashMap<>();


					if (excludeNodes !=null && excludeNodes.length>0)
						params.put("exclude",excludeNodes);

					if (onlyOfType!=null)
						params.put("onlyType", new Object[] {resolveNodeType(onlyOfType)});

					params.put("start", new Object[] {start});
					params.put("limit", new Object[] {limit});
					params.put("showHidden", new Object[] {showHidden});

					GXInboundResponse response = myManager.queryParams(params).get();

					if (response.isErrorResponse()) {
						if (response.hasException()) 
							throw response.getException();
						else 
							throw new BackendGenericError();
					}

					ItemList items = response.getSource().readEntity(ItemList.class);

					return items;
				}catch(Exception e) {
					e.printStackTrace();
					throw e;
				}
			}
		};
		try {
			ItemList result = delegate.make(call);
			return result.getItemlist();
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}



	@Override
	public List<? extends Item> getChildren(String id, int start, int limit, boolean showHidden,
			String... excludeNodes) throws StorageHubException {
		return getChildren(id, start, limit, showHidden, null,  excludeNodes);
	}

	@Override
	@Deprecated
	public List<? extends Item> getChildren(String id, String ... excludeNodes) throws StorageHubException{
		return getChildren(id, false,excludeNodes);
	}

	@Override
	public List<? extends Item> getChildren(String id, boolean showHidden, String ... excludeNodes) throws StorageHubException{
		return getChildren(id,  null, showHidden,  excludeNodes);
	}

	@Override
	@Deprecated
	public Integer childrenCount(String id ,  Class<? extends Item> onlyOfType) throws StorageHubException{
		return childrenCount(id ,  false, onlyOfType);
	}

	@Override
	public Integer childrenCount(String id ,  boolean showHidden, Class<? extends Item> onlyOfType) throws StorageHubException {
		Call<GXWebTargetAdapterRequest, Integer> call = new Call<GXWebTargetAdapterRequest, Integer>() {
			@Override
			public Integer call(GXWebTargetAdapterRequest manager) throws Exception {
				Objects.requireNonNull(id, "id cannot be null");
				GXWebTargetAdapterRequest myManager = manager.path(id).path("children").path("count");
				Map<String, Object[]> params = new HashMap<>();
				if (onlyOfType!=null)
					params.put("onlyType", new Object[] {resolveNodeType(onlyOfType)});

				params.put("showHidden", new Object[] {showHidden});

				GXInboundResponse response = myManager.queryParams(params).get();

				if (response.isErrorResponse()) {
					if (response.hasException()) 
						throw response.getException();
					else 
						throw new BackendGenericError();
				}

				return response.getSource().readEntity(Integer.class);
			}
		};
		try {
			Integer result = delegate.make(call);
			return result;
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}

	@Override
	@Deprecated
	public Integer childrenCount(String id) throws StorageHubException{
		return childrenCount(id, false);
	}

	@Override
	public Integer childrenCount(String id, boolean showHidden) throws StorageHubException {
		return childrenCount(id, showHidden, null);
	}

	private String resolveNodeType(Class<? extends Item> itemClass) throws StorageHubException {
		if (!itemClass.isAnnotationPresent(RootNode.class)) return null;
		String nodeType=  itemClass.getAnnotation(RootNode.class).value();
		return nodeType;
	}


	@Override
	public StreamDescriptor download(String id, String... excludeNodes) throws StorageHubException {
		Call<GXWebTargetAdapterRequest, StreamDescriptor> call = new Call<GXWebTargetAdapterRequest, StreamDescriptor>() {
			@Override
			public StreamDescriptor call(GXWebTargetAdapterRequest manager) throws Exception {
				Objects.requireNonNull(id, "id cannot be null");
				GXWebTargetAdapterRequest myManager = manager.path(id).path("download");
				Map<String, Object[]> params = new HashMap<>();

				if (excludeNodes !=null && excludeNodes.length>0)
					params.put("exclude",excludeNodes);

				GXInboundResponse response = myManager.queryParams(params).get();

				if (response.isErrorResponse()) {
					if (response.hasException()) 
						throw response.getException();
					else 
						throw new BackendGenericError();
				}

				Response resp = response.getSource();

				return createStreamDescriptor(resp);
			}
		};
		try {
			StreamDescriptor result = delegate.make(call);
			return result;
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}

	@Override
	public StreamDescriptor downloadSpecificVersion(String id, String version) throws StorageHubException {
		Call<GXWebTargetAdapterRequest, StreamDescriptor> call = new Call<GXWebTargetAdapterRequest, StreamDescriptor>() {
			@Override
			public StreamDescriptor call(GXWebTargetAdapterRequest manager) throws Exception {
				Objects.requireNonNull(id, "id cannot be null");
				Objects.requireNonNull(version, "version cannot be null");

				GXWebTargetAdapterRequest myManager = manager.path(id).path("versions").path(version).path("download");

				GXInboundResponse response = myManager.get();

				if (response.isErrorResponse()) {
					if (response.hasException()) 
						throw response.getException();
					else 
						throw new BackendGenericError();
				}

				Response resp = response.getSource();

				return createStreamDescriptor(resp);
			}
		};
		try {
			StreamDescriptor result = delegate.make(call);
			return result;
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}


	@Override
	public List<Version> getFileVersions(String id) throws StorageHubException {
		Call<GXWebTargetAdapterRequest, VersionList> call = new Call<GXWebTargetAdapterRequest, VersionList>() {
			@Override
			public VersionList call(GXWebTargetAdapterRequest manager) throws Exception {
				Objects.requireNonNull(id, "id cannot be null");
				GXWebTargetAdapterRequest myManager = manager.path(id).path("versions");

				GXInboundResponse response = myManager.get();

				if (response.isErrorResponse()) {
					if (response.hasException()) 
						throw response.getException();
					else 
						throw new BackendGenericError();
				}

				VersionList versions = response.getSource().readEntity(VersionList.class);

				return versions;
			}
		};
		try {
			VersionList result = delegate.make(call);
			return result.getItemlist();
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}

	@Override
	public Item get(String id, String... excludeNodes) throws StorageHubException {
		Call<GXWebTargetAdapterRequest, ItemWrapper<Item>> call = new Call<GXWebTargetAdapterRequest, ItemWrapper<Item>>() {
			@Override
			public ItemWrapper<Item> call(GXWebTargetAdapterRequest manager) throws Exception {
				Objects.requireNonNull(id, "id cannot be null");
				GXWebTargetAdapterRequest myManager = manager.path(id);
				Map<String, Object[]> params = new HashMap<>();

				if (excludeNodes !=null && excludeNodes.length>0)
					params.put("exclude",excludeNodes);

				GXInboundResponse response = myManager.queryParams(params).get();

				if (response.isErrorResponse()) {
					if (response.hasException()) 
						throw response.getException();
					else 
						throw new BackendGenericError("error response from the service with code: "+response.getHTTPCode());
				}

				ItemWrapper<Item> item = response.getSource().readEntity(ItemWrapper.class);


				return item;
			}
		};
		try {
			ItemWrapper<Item> result = delegate.make(call);
			return result.getItem();
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}

	@Override
	public Item getRootSharedFolder(String id) throws StorageHubException {
		Call<GXWebTargetAdapterRequest, ItemWrapper<Item>> call = new Call<GXWebTargetAdapterRequest, ItemWrapper<Item>>() {
			@Override
			public ItemWrapper<Item> call(GXWebTargetAdapterRequest manager) throws Exception {
				Objects.requireNonNull(id, "id cannot be null");
				GXWebTargetAdapterRequest myManager = manager.path(id).path("rootSharedFolder");
				GXInboundResponse response = myManager.get();

				if (response.isErrorResponse()) {
					if (response.hasException()) 
						throw response.getException();
					else 
						throw new BackendGenericError();
				}

				ItemWrapper<Item> item = response.getSource().readEntity(ItemWrapper.class);


				return item;
			}
		};
		try {
			ItemWrapper<Item> result = delegate.make(call);
			return result.getItem();
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}


	@Override
	public URL getPublickLink(String id) throws StorageHubException {
		Call<GXWebTargetAdapterRequest, URL> call = new Call<GXWebTargetAdapterRequest, URL>() {
			@Override
			public URL call(GXWebTargetAdapterRequest manager) throws Exception {
				Objects.requireNonNull(id, "id cannot be null");
				GXWebTargetAdapterRequest myManager = manager.path(id).path("publiclink");
				GXInboundResponse response = myManager.get();

				if (response.isErrorResponse()) {
					if (response.hasException()) 
						throw response.getException();
					else 
						throw new BackendGenericError();
				}

				URL item = response.getSource().readEntity(URL.class);

				return item;
			}
		};
		try {
			URL result = delegate.make(call);
			return result;
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}

	@Override
	public URL getPublickLink(String id, String version) throws StorageHubException {
		Call<GXWebTargetAdapterRequest, URL> call = new Call<GXWebTargetAdapterRequest, URL>() {
			@Override
			public URL call(GXWebTargetAdapterRequest manager) throws Exception {
				Objects.requireNonNull(id, "id cannot be null");
				Objects.requireNonNull(version, "version cannot be null");

				GXWebTargetAdapterRequest myManager = manager.path(id).path("publiclink");
				Map<String, Object[]> params = new HashMap<>();
				params.put("version",new Object[] {version});
				GXInboundResponse response = myManager.queryParams(params).get();

				if (response.isErrorResponse()) {
					if (response.hasException()) 
						throw response.getException();
					else 
						throw new BackendGenericError();
				}

				URL item = response.getSource().readEntity(URL.class);

				return item;
			}
		};
		try {
			URL result = delegate.make(call);
			return result;
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}


	@Override
	public List<? extends Item> findChildrenByNamePattern(String id, String name, String... excludeNodes) throws StorageHubException {
		Call<GXWebTargetAdapterRequest, ItemList> call = new Call<GXWebTargetAdapterRequest, ItemList>() {
			@Override
			public ItemList call(GXWebTargetAdapterRequest manager) throws Exception {
				Objects.requireNonNull(id, "id cannot be null");
				Objects.requireNonNull(id, "name cannot be null");

				GXWebTargetAdapterRequest myManager = manager.path(id).path("items").path(name);
				Map<String, Object[]> params = new HashMap<>();

				if (excludeNodes !=null && excludeNodes.length>0)
					params.put("exclude",excludeNodes);

				System.out.println("calling "+ myManager.queryParams(params).toString());				
				GXInboundResponse response = myManager.queryParams(params).get();

				if (response.isErrorResponse()) {
					if (response.hasException()) 
						throw response.getException();
					else 
						throw new BackendGenericError();
				}

				ItemList items = response.getSource().readEntity(ItemList.class);

				return items;
			}
		};
		try {
			ItemList result = delegate.make(call);
			return result.getItemlist();
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}


	@Override
	public String uploadFile(InputStream stream, String parentId, String fileName, String description) throws StorageHubException {
		Call<GXWebTargetAdapterRequest, String> call = new Call<GXWebTargetAdapterRequest, String>() {
			@Override
			public String call(GXWebTargetAdapterRequest manager) throws Exception {
				Objects.requireNonNull(stream, "stream cannot be null");
				Objects.requireNonNull(parentId, "parentId cannot be null");
				Objects.requireNonNull(fileName, "parentId cannot be null");
				Objects.requireNonNull(description, "parentId cannot be null");

				GXWebTargetAdapterRequest myManager = manager.register(MultiPartFeature.class).path(parentId)
						.path("create").path("FILE");

				GXInboundResponse response =null;
				FormDataMultiPart multipart = new FormDataMultiPart();
				
				multipart.field("name", fileName);
				multipart.field("description", description);
				multipart.field("file", stream, MediaType.APPLICATION_OCTET_STREAM_TYPE);
				
				Entity<FormDataMultiPart> entity = Entity.entity(multipart, MediaType.MULTIPART_FORM_DATA);
				
				response = myManager.post(entity);
				
				if (response.isErrorResponse()) {
					if (response.hasException()) {
						throw response.getException();
					}else 
						throw new BackendGenericError();
				}

				return response.getSource().readEntity(String.class);
			}
		};
		try {
			return delegate.make(call);
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}

	@Override
	public String uploadArchive(InputStream stream, String parentId, String extractionFolderName) throws StorageHubException {
		Call<GXWebTargetAdapterRequest, String> call = new Call<GXWebTargetAdapterRequest, String>() {
			@Override
			public String call(GXWebTargetAdapterRequest manager) throws Exception {
				Objects.requireNonNull(stream, "stream cannot be null");
				Objects.requireNonNull(parentId, "parentId cannot be null");
				Objects.requireNonNull(extractionFolderName, "extraction folder name cannot be null");

				GXWebTargetAdapterRequest myManager = manager.register(MultiPartFeature.class).path(parentId)
						.path("create").path("ARCHIVE");

				FormDataMultiPart multipart = new FormDataMultiPart();
				multipart.field("parentFolderName", extractionFolderName);
				multipart.field("file", stream, MediaType.APPLICATION_OCTET_STREAM_TYPE);

				GXInboundResponse response = myManager.post(Entity.entity(multipart, MediaType.MULTIPART_FORM_DATA_TYPE));

				if (response.isErrorResponse()) {
					if (response.hasException()) 
						throw response.getException();
					else 
						throw new BackendGenericError();
				}

				return response.getSource().readEntity(String.class);

			}
		};
		try {
			return delegate.make(call);
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}

	@Override
	@Deprecated
	public String createFolder(String parentId, String name, String description) throws StorageHubException {
		Call<GXWebTargetAdapterRequest, String> call = new Call<GXWebTargetAdapterRequest, String>() {
			@Override
			public String call(GXWebTargetAdapterRequest manager) throws Exception {
				return createFolder(parentId, name, description, false);
			}
		};
		try {
			return delegate.make(call);
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}


	@Override
	public String createFolder(String parentId, String name, String description, boolean hidden) throws StorageHubException {
		Call<GXWebTargetAdapterRequest, String> call = new Call<GXWebTargetAdapterRequest, String>() {
			@Override
			public String call(GXWebTargetAdapterRequest manager) throws Exception {
				System.out.println("call done");
				Objects.requireNonNull(parentId, "parentId cannot be null");
				Objects.requireNonNull(name, "folder name cannot be null");
				Objects.requireNonNull(description, "parentId cannot be null");


				GXWebTargetAdapterRequest myManager = manager.path(parentId)
						.path("create").path("FOLDER");

				MultivaluedMap<String, String> formData = new MultivaluedHashMap<String, String>();
				formData.add("name", name);
				formData.add("description", description);
				formData.add("hidden", Boolean.toString(hidden));


				GXInboundResponse response = myManager.post(Entity.entity(formData, MediaType.APPLICATION_FORM_URLENCODED));

				if (response.isErrorResponse()) {
					if (response.hasException()) 
						throw response.getException();
					else 
						throw new BackendGenericError();
				}

				return response.getSource().readEntity(String.class);
			}
		};
		try {
			return delegate.make(call);
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}
	
	
	@Override
	public String createURL(String parentId, String name, String description,URL url) throws StorageHubException {
		Call<GXWebTargetAdapterRequest, String> call = new Call<GXWebTargetAdapterRequest, String>() {
			@Override
			public String call(GXWebTargetAdapterRequest manager) throws Exception {
				Objects.requireNonNull(parentId, "parentId cannot be null");
				Objects.requireNonNull(name, "folder name cannot be null");
				Objects.requireNonNull(description, "parentId cannot be null");


				GXWebTargetAdapterRequest myManager = manager.path(parentId)
						.path("create").path("URL");

				MultivaluedMap<String, String> formData = new MultivaluedHashMap<String, String>();
				formData.add("name", name);
				formData.add("description", description);
				formData.add("value", url.toString());


				GXInboundResponse response = myManager.post(Entity.entity(formData, MediaType.APPLICATION_FORM_URLENCODED));

				if (response.isErrorResponse()) {
					if (response.hasException()) 
						throw response.getException();
					else 
						throw new BackendGenericError();
				}

				return response.getSource().readEntity(String.class);
			}
		};
		try {
			return delegate.make(call);
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}


	@Override
	public String createGcubeItem(String parentId, GCubeItem item) throws StorageHubException {
		Call<GXWebTargetAdapterRequest, String> call = new Call<GXWebTargetAdapterRequest, String>() {
			@Override
			public String call(GXWebTargetAdapterRequest manager) throws Exception {

				Objects.requireNonNull(parentId, "parentId cannot be null");
				Objects.requireNonNull(item, "item cannot be null");

				GXWebTargetAdapterRequest myManager = manager.path(parentId)
						.path("create").path("GCUBEITEM");


				GXInboundResponse response = myManager.post(Entity.json(item));

				if (response.isErrorResponse()) {
					if (response.hasException()) 
						throw response.getException();
					else 
						throw new BackendGenericError();
				}

				return response.getSource().readEntity(String.class);
			}
		};
		try {
			return delegate.make(call);
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}

	@Override
	public String shareFolder(String id, Set<String> users, AccessType accessType) throws StorageHubException {
		Call<GXWebTargetAdapterRequest, String> call = new Call<GXWebTargetAdapterRequest, String>() {
			@Override
			public String call(GXWebTargetAdapterRequest manager) throws Exception {

				Objects.requireNonNull(id, "id cannot be null");
				Objects.requireNonNull(users, "users list cannot be null");
				Objects.requireNonNull(accessType, "access type cannot be null");

				GXWebTargetAdapterRequest myManager = manager.register(MultiPartFeature.class).path(id)
						.path("share");

				try (FormDataMultiPart multipart = new FormDataMultiPart()){
					multipart.field("defaultAccessType", accessType, MediaType.APPLICATION_JSON_TYPE);
					multipart.field("users", users, MediaType.APPLICATION_JSON_TYPE);

					GXInboundResponse response = myManager.put(Entity.entity(multipart, MediaType.MULTIPART_FORM_DATA_TYPE));

					if (response.isErrorResponse()) {
						if (response.hasException()) 
							throw response.getException();
						else 
							throw new BackendGenericError();
					}

					return response.getSource().readEntity(String.class);
				}
			}
		};
		try {
			return delegate.make(call);
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}

	@Override
	public String unshareFolder(String id, Set<String> users) throws StorageHubException {
		Call<GXWebTargetAdapterRequest, String> call = new Call<GXWebTargetAdapterRequest, String>() {
			@Override
			public String call(GXWebTargetAdapterRequest manager) throws Exception {
				Objects.requireNonNull(id, "id cannot be null");
				Objects.requireNonNull(users, "users list cannot be null");

				GXWebTargetAdapterRequest myManager = manager.register(MultiPartFeature.class).path(id)
						.path("unshare");

				try (FormDataMultiPart multipart = new FormDataMultiPart()){
					multipart.field("users", users, MediaType.APPLICATION_JSON_TYPE);
					GXInboundResponse response = myManager.put(Entity.entity(multipart, MediaType.MULTIPART_FORM_DATA_TYPE));

					if (response.isErrorResponse()) {
						if (response.hasException()) 
							throw response.getException();
						else 
							throw new BackendGenericError();
					}

					return response.getSource().readEntity(String.class);
				}
			}
		};
		try {
			return delegate.make(call);
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}

	@Override
	@Deprecated
	public void delete(String id) throws StorageHubException {
		delete(id, false);
	}
	
	@Override
	public void delete(String id, boolean force) throws StorageHubException {
		Call<GXWebTargetAdapterRequest, Void> call = new Call<GXWebTargetAdapterRequest, Void>() {
			@Override
			public Void call(GXWebTargetAdapterRequest manager) throws Exception {
				Objects.requireNonNull(id, "id cannot be null");
				
				HashMap<String, Object[]> queryParam = new HashMap<String, Object[]>();
				queryParam.put("force", new Object[] {force});
				
				GXWebTargetAdapterRequest myManager = manager.path(id);
				GXInboundResponse response = myManager.queryParams(queryParam).delete();
				
				if (response.isErrorResponse()) {
					if (response.hasException()) 
						throw response.getException();
					else 
						throw new BackendGenericError("error response returned from server "+response.getHTTPCode());

				}

				return null;
			}
		};
		try {
			delegate.make(call);
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}

	@Override
	public List<? extends Item> getAnchestors(String id, String... excludeNodes) throws StorageHubException {
		Call<GXWebTargetAdapterRequest, ItemList> call = new Call<GXWebTargetAdapterRequest, ItemList>() {
			@Override
			public ItemList call(GXWebTargetAdapterRequest manager) throws Exception {
				Objects.requireNonNull(id, "id cannot be null");

				GXWebTargetAdapterRequest myManager = manager.path(id).path("anchestors");
				Map<String, Object[]> params = new HashMap<>();

				if (excludeNodes !=null && excludeNodes.length>0)
					params.put("exclude",excludeNodes);

				GXInboundResponse response = myManager.queryParams(params).get();

				if (response.isErrorResponse()) {
					if (response.hasException()) 
						throw response.getException();
					else 
						throw new BackendGenericError();
				}

				return response.getSource().readEntity(ItemList.class);
			}
		};
		try {
			ItemList result = delegate.make(call);
			return result.getItemlist();
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}

	@Override
	public List<ACL> getACL(String id) throws StorageHubException {
		Call<GXWebTargetAdapterRequest, ACLList> call = new Call<GXWebTargetAdapterRequest, ACLList>() {
			@Override
			public ACLList call(GXWebTargetAdapterRequest manager) throws Exception {
				Objects.requireNonNull(id, "id cannot be null");

				GXWebTargetAdapterRequest myManager = manager.path(id).path("acls");
				GXInboundResponse response = myManager.get();

				if (response.isErrorResponse()) {
					if (response.hasException()) 
						throw response.getException();
					else 
						throw new BackendGenericError();
				}

				return response.getSource().readEntity(ACLList.class);
			}
		};
		try {
			return delegate.make(call).getAcls();
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}

	
	
	@Override
	public boolean canWriteInto(String id) throws StorageHubException {
		Call<GXWebTargetAdapterRequest, Boolean> call = new Call<GXWebTargetAdapterRequest, Boolean>() {
			@Override
			public Boolean call(GXWebTargetAdapterRequest manager) throws Exception {
				Objects.requireNonNull(id, "id cannot be null");
				
				GXWebTargetAdapterRequest myManager = manager.path(id).path("acls").path("write");
				
				
				GXInboundResponse response = myManager.get();
				
				if (response.isErrorResponse()) {
					if (response.hasException()) 
						throw response.getException();
					else 
						throw new BackendGenericError();
				}
				
				
				return response.getSource().readEntity(Boolean.class);
				

			}
		};
		try {
			return delegate.make(call);
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}
	
	
	
	@Override
	public String changeACL(String id, String user, AccessType accessType) throws StorageHubException {
		Call<GXWebTargetAdapterRequest, String> call = new Call<GXWebTargetAdapterRequest, String>() {
			@Override
			public String call(GXWebTargetAdapterRequest manager) throws Exception {
				Objects.requireNonNull(id, "id cannot be null");
				Objects.requireNonNull(user, "user cannot be null");
				
				
				GXWebTargetAdapterRequest myManager = manager.path(id).path("acls");
				
				try (FormDataMultiPart multipart = new FormDataMultiPart()){
					multipart.field("accessType", accessType, MediaType.APPLICATION_JSON_TYPE);
					multipart.field("user", user);

					GXInboundResponse response = myManager.put(Entity.entity(multipart, MediaType.MULTIPART_FORM_DATA_TYPE));

					if (response.isErrorResponse()) {
						if (response.hasException()) 
							throw response.getException();
						else 
							throw new BackendGenericError();
					}

					return response.getSource().readEntity(String.class);
				}

			}
		};
		try {
			return delegate.make(call);
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}

	@Override
	public String copy(String id, String destinationFolderId, String newFilename) throws StorageHubException {
		Call<GXWebTargetAdapterRequest, String> call = new Call<GXWebTargetAdapterRequest, String>() {
			@Override
			public String call(GXWebTargetAdapterRequest manager) throws Exception {

				Objects.requireNonNull(id, "id cannot be null");
				Objects.requireNonNull(newFilename, "new file name cannot be null");

				GXWebTargetAdapterRequest myManager = manager.path(id)
						.path("copy");

				MultivaluedMap<String, String> formData = new MultivaluedHashMap<String, String>();
				formData.add("destinationId", destinationFolderId);
				formData.add("fileName", newFilename);


				GXInboundResponse response = myManager.put(Entity.form(formData));

				if (response.isErrorResponse()) {
					if (response.hasException()) 
						throw response.getException();
					else 
						throw new BackendGenericError();
				}

				return response.getSource().readEntity(String.class);
			}
		};
		try {
			return delegate.make(call);
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}

	
	
	
	@Override
	public String setPublic(String id, boolean publish) throws StorageHubException {
		Call<GXWebTargetAdapterRequest, String> call = new Call<GXWebTargetAdapterRequest, String>() {
			@Override
			public String call(GXWebTargetAdapterRequest manager) throws Exception {
				Objects.requireNonNull(id, "id cannot be null");
				
				GXWebTargetAdapterRequest myManager = manager.path(id)
						.path("publish");

				MultivaluedMap<String, String> formData = new MultivaluedHashMap<String, String>();
				formData.add("publish", Boolean.toString(publish));


				GXInboundResponse response = myManager.put(Entity.form(formData));

				if (response.isErrorResponse()) {
					if (response.hasException()) 
						throw response.getException();
					else 
						throw new BackendGenericError();
				}

				return response.getSource().readEntity(String.class);
			}
		};
		try {
			return delegate.make(call);
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}


	@Override
	public String move(String id, String destinationFolderId) throws StorageHubException {
		Call<GXWebTargetAdapterRequest, String> call = new Call<GXWebTargetAdapterRequest, String>() {
			@Override
			public String call(GXWebTargetAdapterRequest manager) throws Exception {
				Objects.requireNonNull(id, "id cannot be null");
				Objects.requireNonNull(destinationFolderId, "destination folder cannot be null");


				GXWebTargetAdapterRequest myManager = manager.path(id)
						.path("move");

				MultivaluedMap<String, String> formData = new MultivaluedHashMap<String, String>();
				formData.add("destinationId", destinationFolderId);


				GXInboundResponse response = myManager.put(Entity.form(formData));

				if (response.isErrorResponse()) {
					if (response.hasException()) 
						throw response.getException();
					else 
						throw new BackendGenericError();
				}

				return response.getSource().readEntity(String.class);
			}
		};
		try {
			return delegate.make(call);
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}

	@Override
	public String rename(String id, String newName) throws StorageHubException {
		Call<GXWebTargetAdapterRequest, String> call = new Call<GXWebTargetAdapterRequest, String>() {
			@Override
			public String call(GXWebTargetAdapterRequest manager) throws Exception {
				Objects.requireNonNull(id, "id cannot be null");
				Objects.requireNonNull(newName, "new name cannot be null");

				GXWebTargetAdapterRequest myManager = manager.path(id)
						.path("rename");

				MultivaluedMap<String, String> formData = new MultivaluedHashMap<String, String>();
				formData.add("newName", newName);


				GXInboundResponse response = myManager.put(Entity.form(formData));

				if (response.isErrorResponse()) {
					if (response.hasException()) 
						throw response.getException();
					else 
						throw new BackendGenericError();
				}

				return response.getSource().readEntity(String.class);
			}
		};
		try {
			return delegate.make(call);
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}


	@Override
	public String setMetadata(String id, Metadata metadata) throws StorageHubException {
		Call<GXWebTargetAdapterRequest, String> call = new Call<GXWebTargetAdapterRequest, String>() {
			@Override
			public String call(GXWebTargetAdapterRequest manager) throws Exception {
				Objects.requireNonNull(id, "id cannot be null");
				Objects.requireNonNull(metadata, "metadata cannot be null");


				GXWebTargetAdapterRequest myManager = manager.path(id)
						.path("metadata");


				GXInboundResponse response = myManager.put(Entity.json(metadata));

				if (response.isErrorResponse()) {
					if (response.hasException()) 
						throw response.getException();
					else 
						throw new BackendGenericError();
				}

				return response.getSource().readEntity(String.class);
			}
		};
		try {
			return delegate.make(call);
		}catch(StorageHubException e) {
			throw e;
		}catch(Exception e1) {
			throw new RuntimeException(e1);
		}
	}


	private StreamDescriptor createStreamDescriptor(Response resp) {
		InputStream stream = resp.readEntity(InputStream.class);
		String disposition = resp.getHeaderString(CONTENT_DISPOSITION);
		String fileName = disposition.replaceFirst("attachment; filename = ([^/s]+)?", "$1");
		String contentType = resp.getHeaderString(CONTENT_TYPE);
		return new StreamDescriptor(stream, fileName, contentType);
	}
}
