/*
 * Decompiled with CFR 0.152.
 */
package marytts.unitselection.data;

import java.io.ByteArrayInputStream;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.io.UTFDataFormatException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Vector;
import marytts.exceptions.MaryConfigurationException;
import marytts.unitselection.data.Unit;
import marytts.util.MaryUtils;
import marytts.util.Pair;
import marytts.util.data.Datagram;
import marytts.util.data.MaryHeader;
import marytts.util.io.StreamUtils;

public class TimelineReader {
    protected MaryHeader maryHdr = null;
    protected ProcHeader procHdr = null;
    protected Index idx = null;
    protected int sampleRate = 0;
    protected long numDatagrams = 0L;
    protected long totalDuration = -1L;
    protected int datagramsBytePos = 0;
    protected int timeIdxBytePos = 0;
    private MappedByteBuffer mappedBB = null;
    private FileChannel fileChannel = null;

    public TimelineReader(String fileName) throws MaryConfigurationException {
        this(fileName, true);
    }

    public TimelineReader(String fileName, boolean tryMemoryMapping) throws MaryConfigurationException {
        if (fileName == null) {
            throw new NullPointerException("Filename is null");
        }
        try {
            this.load(fileName, tryMemoryMapping);
        }
        catch (Exception e) {
            throw new MaryConfigurationException("Cannot load timeline file from " + fileName, e);
        }
    }

    protected TimelineReader() {
    }

    protected void load(String fileName) throws IOException, BufferUnderflowException, MaryConfigurationException, NullPointerException {
        this.load(fileName, true);
    }

    protected void load(String fileName, boolean tryMemoryMapping) throws IOException, BufferUnderflowException, MaryConfigurationException, NullPointerException {
        assert (fileName != null) : "filename is null";
        RandomAccessFile file = new RandomAccessFile(fileName, "r");
        FileChannel fc = file.getChannel();
        ByteBuffer headerBB = ByteBuffer.allocate(65536);
        fc.read(headerBB);
        headerBB.limit(headerBB.position());
        headerBB.position(0);
        this.maryHdr = new MaryHeader(headerBB);
        if (this.maryHdr.getType() != 500) {
            throw new MaryConfigurationException("File is not a valid timeline file.");
        }
        this.procHdr = new ProcHeader(headerBB);
        this.sampleRate = headerBB.getInt();
        this.numDatagrams = headerBB.getLong();
        if (this.sampleRate <= 0 || this.numDatagrams < 0L) {
            throw new MaryConfigurationException("Illegal values in timeline file.");
        }
        this.datagramsBytePos = (int)headerBB.getLong();
        this.timeIdxBytePos = (int)headerBB.getLong();
        if (this.timeIdxBytePos < this.datagramsBytePos) {
            throw new MaryConfigurationException("File seems corrupt: index is expected after data, not before");
        }
        fc.position(this.timeIdxBytePos);
        ByteBuffer indexBB = ByteBuffer.allocate((int)(fc.size() - (long)this.timeIdxBytePos));
        fc.read(indexBB);
        indexBB.limit(indexBB.position());
        indexBB.position(0);
        this.idx = new Index(indexBB);
        if (tryMemoryMapping) {
            try {
                this.mappedBB = fc.map(FileChannel.MapMode.READ_ONLY, this.datagramsBytePos, this.timeIdxBytePos - this.datagramsBytePos);
                file.close();
            }
            catch (IOException ome) {
                MaryUtils.getLogger("Timeline").warn("Cannot use memory mapping for timeline file '" + fileName + "' -- falling back to piecewise reading");
            }
        }
        if (!tryMemoryMapping || this.mappedBB == null) {
            this.fileChannel = fc;
            assert (this.fileChannel != null);
        }
        assert (this.idx != null);
        assert (this.procHdr != null);
        assert (this.fileChannel == null && this.mappedBB != null || this.fileChannel != null && this.mappedBB == null);
    }

    public String getProcHeaderContents() {
        return this.procHdr.getString();
    }

    public long getNumDatagrams() {
        assert (this.numDatagrams >= 0L);
        return this.numDatagrams;
    }

    protected long getDatagramsBytePos() {
        return this.datagramsBytePos;
    }

    public int getSampleRate() {
        assert (this.sampleRate > 0);
        return this.sampleRate;
    }

    public long getTotalDuration() throws MaryConfigurationException {
        if (this.totalDuration == -1L) {
            this.computeTotalDuration();
        }
        assert (this.totalDuration >= 0L);
        return this.totalDuration;
    }

    protected void computeTotalDuration() throws MaryConfigurationException {
        long time = 0L;
        long nRead = 0L;
        boolean haveReadAll = false;
        try {
            Pair<ByteBuffer, Long> p = this.getByteBufferAtTime(0L);
            ByteBuffer bb = p.getFirst();
            assert (p.getSecond() == 0L);
            while (!haveReadAll) {
                Datagram dat = this.getNextDatagram(bb);
                if (dat == null) {
                    p = this.getByteBufferAtTime(time);
                    bb = p.getFirst();
                    assert (p.getSecond() == time);
                    dat = this.getNextDatagram(bb);
                    if (dat == null) break;
                }
                assert (dat != null);
                time += dat.getDuration();
                if (++nRead != this.numDatagrams) continue;
                haveReadAll = true;
            }
        }
        catch (Exception e) {
            throw new MaryConfigurationException("Could not compute total duration", e);
        }
        if (!haveReadAll) {
            throw new MaryConfigurationException("Could not read all datagrams to compute total duration");
        }
        this.totalDuration = time;
    }

    public Index getIndex() {
        assert (this.idx != null);
        return this.idx;
    }

    protected long scaleTime(int reqSampleRate, long targetTimeInSamples) {
        if (reqSampleRate == this.sampleRate) {
            return targetTimeInSamples;
        }
        return Math.round((double)reqSampleRate * (double)targetTimeInSamples / (double)this.sampleRate);
    }

    protected long unScaleTime(int reqSampleRate, long timelineTimeInSamples) {
        if (reqSampleRate == this.sampleRate) {
            return timelineTimeInSamples;
        }
        return Math.round((double)this.sampleRate * (double)timelineTimeInSamples / (double)reqSampleRate);
    }

    protected long skipNextDatagram(ByteBuffer bb) throws IOException {
        long datagramDuration = bb.getLong();
        int datagramSize = bb.getInt();
        if (bb.position() + datagramSize > bb.limit()) {
            throw new IOException("cannot skip datagram: it is not fully contained in byte buffer");
        }
        bb.position(bb.position() + datagramSize);
        return datagramDuration;
    }

    protected Datagram getNextDatagram(ByteBuffer bb) {
        assert (bb != null);
        if (bb.position() == bb.limit()) {
            return null;
        }
        try {
            return new Datagram(bb);
        }
        catch (IOException ioe) {
            return null;
        }
    }

    protected long hopToTime(ByteBuffer bb, long currentTimeInSamples, long targetTimeInSamples) throws IOException, IllegalArgumentException {
        assert (bb != null);
        assert (currentTimeInSamples >= 0L);
        assert (targetTimeInSamples >= currentTimeInSamples) : "Cannot hop back from time " + currentTimeInSamples + " to time " + targetTimeInSamples;
        if (currentTimeInSamples == targetTimeInSamples) {
            return currentTimeInSamples;
        }
        int byteBefore = bb.position();
        long timeBefore = currentTimeInSamples;
        while (currentTimeInSamples <= targetTimeInSamples) {
            timeBefore = currentTimeInSamples;
            byteBefore = bb.position();
            long skippedDuration = this.skipNextDatagram(bb);
            currentTimeInSamples += skippedDuration;
        }
        bb.position(byteBefore);
        return timeBefore;
    }

    protected Pair<ByteBuffer, Long> getByteBufferAtTime(long targetTimeInSamples) throws IOException, BufferUnderflowException {
        if (this.mappedBB != null) {
            return this.getMappedByteBufferAtTime(targetTimeInSamples);
        }
        return this.loadByteBufferAtTime(targetTimeInSamples);
    }

    protected Pair<ByteBuffer, Long> getMappedByteBufferAtTime(long targetTimeInSamples) throws IllegalArgumentException, IOException {
        assert (this.mappedBB != null);
        IdxField idxFieldBefore = this.idx.getIdxFieldBefore(targetTimeInSamples);
        long time = idxFieldBefore.timePtr;
        int bytePos = (int)(idxFieldBefore.bytePtr - (long)this.datagramsBytePos);
        ByteBuffer bb = this.mappedBB.duplicate();
        bb.position(bytePos);
        time = this.hopToTime(bb, time, targetTimeInSamples);
        return new Pair<ByteBuffer, Long>(bb, time);
    }

    protected Pair<ByteBuffer, Long> loadByteBufferAtTime(long targetTimeInSamples) throws IOException {
        assert (this.fileChannel != null);
        int bufSize = 65536;
        IdxField idxFieldBefore = this.idx.getIdxFieldBefore(targetTimeInSamples);
        long time = idxFieldBefore.timePtr;
        long bytePos = idxFieldBefore.bytePtr;
        if (bytePos + (long)bufSize > (long)this.timeIdxBytePos) {
            bufSize = (int)((long)this.timeIdxBytePos - bytePos);
        }
        ByteBuffer bb = this.loadByteBuffer(bytePos, bufSize);
        while (true) {
            if (!this.canReadDatagramHeader(bb)) {
                bb = this.loadByteBuffer(bytePos, bufSize);
                assert (this.canReadDatagramHeader(bb));
            }
            int posBefore = bb.position();
            Datagram d = new Datagram(bb, false);
            if (time + d.getDuration() > targetTimeInSamples) {
                bb.position(posBefore);
                int datagramNumBytes = 12 + d.getLength();
                if (!this.canReadAmount(bb, datagramNumBytes)) {
                    bb = this.loadByteBuffer(bytePos, Math.max(datagramNumBytes, bufSize));
                }
                assert (this.canReadAmount(bb, datagramNumBytes));
                break;
            }
            time += d.getDuration();
            if (this.canReadAmount(bb, d.getLength())) {
                bb.position(bb.position() + d.getLength());
                continue;
            }
            bytePos += (long)bb.position();
            bb = this.loadByteBuffer(bytePos += (long)d.getLength(), bufSize);
        }
        return new Pair<ByteBuffer, Long>(bb, time);
    }

    private ByteBuffer loadByteBuffer(long bytePos, int bufSize) throws IOException {
        ByteBuffer bb = ByteBuffer.allocate(bufSize);
        this.fileChannel.read(bb, bytePos);
        bb.limit(bb.position());
        bb.position(0);
        return bb;
    }

    private boolean canReadDatagramHeader(ByteBuffer bb) {
        return this.canReadAmount(bb, 12);
    }

    private boolean canReadAmount(ByteBuffer bb, int amount) {
        return bb.limit() - bb.position() >= amount;
    }

    public Datagram getDatagram(long targetTimeInSamples) throws IOException {
        Pair<ByteBuffer, Long> p = this.getByteBufferAtTime(targetTimeInSamples);
        ByteBuffer bb = p.getFirst();
        return this.getNextDatagram(bb);
    }

    public Datagram getDatagram(long targetTimeInSamples, int reqSampleRate) throws IOException {
        long scaledTargetTime = this.scaleTime(reqSampleRate, targetTimeInSamples);
        Datagram dat = this.getDatagram(scaledTargetTime);
        if (dat == null) {
            return null;
        }
        if (reqSampleRate != this.sampleRate) {
            dat.setDuration(this.unScaleTime(reqSampleRate, dat.getDuration()));
        }
        return dat;
    }

    private Datagram[] getDatagrams(long targetTimeInSamples, int nDatagrams, long timeSpanInSamples, int reqSampleRate, long[] returnOffset) throws IllegalArgumentException, IOException {
        boolean byNumber;
        if (targetTimeInSamples < 0L) {
            throw new IllegalArgumentException("Can't get a datagram from a negative time position (given time position was [" + targetTimeInSamples + "]).");
        }
        if (reqSampleRate <= 0) {
            throw new IllegalArgumentException("sample rate must be positive, but is " + reqSampleRate);
        }
        if (timeSpanInSamples > 0L) {
            byNumber = false;
        } else {
            byNumber = true;
            if (nDatagrams <= 0) {
                nDatagrams = 1;
            }
        }
        long scaledTargetTime = this.scaleTime(reqSampleRate, targetTimeInSamples);
        Pair<ByteBuffer, Long> p = this.getByteBufferAtTime(scaledTargetTime);
        ByteBuffer bb = p.getFirst();
        long time = p.getSecond();
        if (returnOffset != null) {
            if (returnOffset.length == 0) {
                throw new IllegalArgumentException("If returnOffset is given, it must have length of at least 1");
            }
            returnOffset[0] = this.unScaleTime(reqSampleRate, scaledTargetTime - time);
        }
        ArrayList<Datagram> datagrams = new ArrayList<Datagram>(byNumber ? nDatagrams : 10);
        long endTime = byNumber ? -1L : this.scaleTime(reqSampleRate, targetTimeInSamples + timeSpanInSamples);
        int nRead = 0;
        boolean haveReadAll = false;
        while (!haveReadAll) {
            Datagram dat = this.getNextDatagram(bb);
            if (dat == null) {
                try {
                    p = this.getByteBufferAtTime(time);
                }
                catch (Exception ioe) {
                    break;
                }
                bb = p.getFirst();
                dat = this.getNextDatagram(bb);
                if (dat == null) break;
            }
            assert (dat != null);
            time += dat.getDuration();
            ++nRead;
            if (reqSampleRate != this.sampleRate) {
                dat.setDuration(this.unScaleTime(reqSampleRate, dat.getDuration()));
            }
            datagrams.add(dat);
            if ((!byNumber || nRead != nDatagrams) && (byNumber || time < endTime)) continue;
            haveReadAll = true;
        }
        return datagrams.toArray(new Datagram[0]);
    }

    public Datagram[] getDatagrams(long targetTimeInSamples, long timeSpanInSamples, int reqSampleRate, long[] returnOffset) throws IOException {
        return this.getDatagrams(targetTimeInSamples, -1, timeSpanInSamples, reqSampleRate, returnOffset);
    }

    public Datagram[] getDatagrams(long targetTimeInSamples, long timeSpanInSamples, int reqSampleRate) throws IOException {
        return this.getDatagrams(targetTimeInSamples, timeSpanInSamples, reqSampleRate, null);
    }

    public Datagram[] getDatagrams(long targetTimeInSamples, long timeSpanInSamples) throws IOException {
        return this.getDatagrams(targetTimeInSamples, timeSpanInSamples, this.sampleRate, null);
    }

    public Datagram[] getDatagrams(long targetTimeInSamples, int number, int reqSampleRate, long[] returnOffset) throws IOException {
        return this.getDatagrams(targetTimeInSamples, number, -1L, reqSampleRate, returnOffset);
    }

    public Datagram[] getDatagrams(Unit unit, int reqSampleRate, long[] returnOffset) throws IOException {
        return this.getDatagrams(unit.startTime, (long)unit.duration, reqSampleRate, returnOffset);
    }

    public Datagram[] getDatagrams(Unit unit, int reqSampleRate) throws IOException {
        return this.getDatagrams(unit, reqSampleRate, null);
    }

    public static class ProcHeader {
        private String procHeader = null;

        private ProcHeader(RandomAccessFile raf) throws IOException {
            this.loadProcHeader(raf);
        }

        private ProcHeader(ByteBuffer bb) throws BufferUnderflowException, UTFDataFormatException {
            this.loadProcHeader(bb);
        }

        public ProcHeader(String procStr) {
            if (procStr == null) {
                throw new NullPointerException("null argument");
            }
            this.procHeader = procStr;
        }

        public int getCharSize() {
            assert (this.procHeader != null);
            return this.procHeader.length();
        }

        public String getString() {
            assert (this.procHeader != null);
            return this.procHeader;
        }

        private void loadProcHeader(RandomAccessFile rafIn) throws IOException {
            assert (rafIn != null) : "null argument";
            this.procHeader = rafIn.readUTF();
            assert (this.procHeader != null);
        }

        private void loadProcHeader(ByteBuffer bb) throws BufferUnderflowException, UTFDataFormatException {
            this.procHeader = StreamUtils.readUTF(bb);
            assert (this.procHeader != null);
        }

        public long dump(RandomAccessFile rafIn) throws IOException {
            long before = rafIn.getFilePointer();
            rafIn.writeUTF(this.procHeader);
            long after = rafIn.getFilePointer();
            return after - before;
        }
    }

    public static class IdxField {
        public long bytePtr = 0L;
        public long timePtr = 0L;

        public IdxField() {
            this.bytePtr = 0L;
            this.timePtr = 0L;
        }

        public IdxField(long setBytePtr, long setTimePtr) {
            this.bytePtr = setBytePtr;
            this.timePtr = setTimePtr;
        }
    }

    public static class Index {
        private int idxInterval = 0;
        private long[] bytePtrs;
        private long[] timePtrs;

        private Index(DataInput raf) throws IOException, MaryConfigurationException {
            assert (raf != null) : "null argument";
            this.load(raf);
        }

        private Index(ByteBuffer bb) throws BufferUnderflowException, MaryConfigurationException {
            assert (bb != null) : "null argument";
            this.load(bb);
        }

        public Index(int idxInterval, Vector<IdxField> indexFields) throws IllegalArgumentException, NullPointerException {
            if (idxInterval <= 0) {
                throw new IllegalArgumentException("got index interval <= 0");
            }
            if (indexFields == null) {
                throw new NullPointerException("null argument");
            }
            this.idxInterval = idxInterval;
            this.bytePtrs = new long[indexFields.size()];
            this.timePtrs = new long[indexFields.size()];
            for (int i = 0; i < this.bytePtrs.length; ++i) {
                IdxField f = indexFields.get(i);
                this.bytePtrs[i] = f.bytePtr;
                this.timePtrs[i] = f.timePtr;
                if (i <= 0 || this.bytePtrs[i] >= this.bytePtrs[i - 1] && this.timePtrs[i] >= this.timePtrs[i - 1]) continue;
                throw new IllegalArgumentException("Pointer positions in index fields must be strictly monotonously rising");
            }
        }

        public void load(DataInput rafIn) throws IOException, MaryConfigurationException {
            int numIdx = rafIn.readInt();
            this.idxInterval = rafIn.readInt();
            if (this.idxInterval <= 0) {
                throw new MaryConfigurationException("read negative index interval -- file seems corrupt");
            }
            this.bytePtrs = new long[numIdx];
            this.timePtrs = new long[numIdx];
            int numBytesToRead = 16 * numIdx + 16;
            byte[] data = new byte[numBytesToRead];
            rafIn.readFully(data);
            DataInputStream bufIn = new DataInputStream(new ByteArrayInputStream(data));
            for (int i = 0; i < numIdx; ++i) {
                this.bytePtrs[i] = bufIn.readLong();
                this.timePtrs[i] = bufIn.readLong();
                if (i <= 0 || this.bytePtrs[i] >= this.bytePtrs[i - 1] && this.timePtrs[i] >= this.timePtrs[i - 1]) continue;
                throw new MaryConfigurationException("File seems corrupt: Pointer positions in index fields are not strictly monotonously rising");
            }
            bufIn.readLong();
            bufIn.readLong();
        }

        private void load(ByteBuffer bb) throws BufferUnderflowException, MaryConfigurationException {
            int numIdx = bb.getInt();
            this.idxInterval = bb.getInt();
            if (this.idxInterval <= 0) {
                throw new MaryConfigurationException("read negative index interval -- file seems corrupt");
            }
            this.bytePtrs = new long[numIdx];
            this.timePtrs = new long[numIdx];
            for (int i = 0; i < numIdx; ++i) {
                this.bytePtrs[i] = bb.getLong();
                this.timePtrs[i] = bb.getLong();
                if (i <= 0 || this.bytePtrs[i] >= this.bytePtrs[i - 1] && this.timePtrs[i] >= this.timePtrs[i - 1]) continue;
                throw new MaryConfigurationException("File seems corrupt: Pointer positions in index fields are not strictly monotonously rising");
            }
            bb.getLong();
            bb.getLong();
        }

        public long dump(RandomAccessFile rafIn) throws IOException {
            long nBytes = 0L;
            int numIdx = this.getNumIdx();
            rafIn.writeInt(numIdx);
            nBytes += 4L;
            rafIn.writeInt(this.idxInterval);
            nBytes += 4L;
            for (int i = 0; i < numIdx; ++i) {
                rafIn.writeLong(this.bytePtrs[i]);
                nBytes += 8L;
                rafIn.writeLong(this.timePtrs[i]);
                nBytes += 8L;
            }
            rafIn.writeLong(0L);
            rafIn.writeLong(0L);
            return nBytes += 16L;
        }

        public void print() {
            System.out.println("<INDEX>");
            int numIdx = this.getNumIdx();
            System.out.println("interval = " + this.idxInterval);
            System.out.println("numIdx = " + numIdx);
            for (int i = 0; i < numIdx; ++i) {
                System.out.println("( " + this.bytePtrs[i] + " , " + this.timePtrs[i] + " )");
            }
            System.out.println("</INDEX>");
        }

        public int getNumIdx() {
            return this.bytePtrs.length;
        }

        public int getIdxInterval() {
            return this.idxInterval;
        }

        public IdxField getIdxField(int i) {
            if (i < 0) {
                throw new IndexOutOfBoundsException("Negative index.");
            }
            if (i >= this.bytePtrs.length) {
                throw new IndexOutOfBoundsException("Requested index no. " + i + ", but highest is " + this.bytePtrs.length);
            }
            return new IdxField(this.bytePtrs[i], this.timePtrs[i]);
        }

        public IdxField getIdxFieldBefore(long timePosition) {
            if (timePosition < 0L) {
                throw new IllegalArgumentException("Negative time given");
            }
            int index = (int)(timePosition / (long)this.idxInterval);
            if (index < 0) {
                throw new RuntimeException("Negative index field: [" + index + "] encountered when getting index before time=[" + timePosition + "] (idxInterval=[" + this.idxInterval + "]).");
            }
            if (index >= this.bytePtrs.length) {
                index = this.bytePtrs.length - 1;
            }
            return new IdxField(this.bytePtrs[index], this.timePtrs[index]);
        }
    }
}

