package org.gcube.common.homelibrary.jcr.workspace.trash;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.gcube.common.homelibary.model.items.ItemDelegate;
import org.gcube.common.homelibary.model.items.type.NodeProperty;
import org.gcube.common.homelibary.model.items.type.WorkspaceItemType;
import org.gcube.common.homelibrary.home.exceptions.InternalErrorException;
import org.gcube.common.homelibrary.home.workspace.WorkspaceItem;
import org.gcube.common.homelibrary.home.workspace.exceptions.InsufficientPrivilegesException;
import org.gcube.common.homelibrary.home.workspace.exceptions.ItemAlreadyExistException;
import org.gcube.common.homelibrary.home.workspace.exceptions.ItemNotFoundException;
import org.gcube.common.homelibrary.home.workspace.exceptions.WorkspaceFolderNotFoundException;
import org.gcube.common.homelibrary.home.workspace.exceptions.WrongDestinationException;
import org.gcube.common.homelibrary.home.workspace.exceptions.WrongItemTypeException;
import org.gcube.common.homelibrary.home.workspace.trash.WorkspaceTrashItem;
import org.gcube.common.homelibrary.jcr.repository.JCRRepository;
import org.gcube.common.homelibrary.jcr.repository.external.GCUBEStorage;
import org.gcube.common.homelibrary.jcr.workspace.JCRWorkspace;
import org.gcube.common.homelibrary.jcr.workspace.JCRWorkspaceFolder;
import org.gcube.common.homelibrary.jcr.workspace.servlet.JCRServlets;
import org.gcube.common.homelibrary.jcr.workspace.servlet.wrapper.DelegateManager;
import org.gcube.contentmanagement.blobstorage.transport.backend.RemoteBackendException;

import com.thoughtworks.xstream.XStream;

public class JCRWorkspaceTrashItem extends JCRWorkspaceFolder implements WorkspaceTrashItem {

//	protected static final String CONTENT				= "jcr:content";
//	protected static final String DELETE_DATE			= "hl:deletedTime";
//	protected static final String DELETE_USER			= "hl:deletedBy";
//	protected static final String ORIGINAL_PARENT_ID	= "hl:originalParentId";
//	protected static final String ORIGINAL_PATH			= "hl:deletedFrom";
//	protected static final String NAME					= "hl:name";
//	protected static final String MIME_TYPE				= "hl:mimeType";
//	protected static final String LENGTH				= "hl:length";
//	protected static final String IS_FOLDER				= "hl:isFolder";


	private final JCRWorkspace workspace;

	private ItemDelegate delegate;

	public JCRWorkspaceTrashItem(JCRWorkspace workspace, ItemDelegate delegate) throws RepositoryException {
		super(workspace, delegate);

		this.delegate = delegate;
		this.workspace = workspace;
	}



	public JCRWorkspaceTrashItem(JCRWorkspace workspace, ItemDelegate delegate,
			String name, String description, Calendar date, String portalLogin,
			String originalParentId, String mimeType, long length, boolean isFolder, String originalPath) throws RepositoryException {
		super(workspace, delegate, name, description);

		this.delegate = delegate;
		this.workspace = workspace;

		delegate.setProperties(new HashMap<NodeProperty, String>());
		delegate.getProperties().put(NodeProperty.TRASH_ITEM_NAME, name);
		delegate.getProperties().put(NodeProperty.DELETE_DATE, new XStream().toXML(Calendar.getInstance()));
		delegate.getProperties().put(NodeProperty.DELETE_BY, portalLogin);
		delegate.getProperties().put(NodeProperty.ORIGINAL_PARENT_ID, originalParentId);
		delegate.getProperties().put(NodeProperty.DELETED_FROM, originalPath);
		delegate.getProperties().put(NodeProperty.TRASH_ITEM_MIME_TYPE, mimeType);
		delegate.getProperties().put(NodeProperty.LENGTH, new XStream().toXML(length));
		delegate.getProperties().put(NodeProperty.IS_FOLDER, new XStream().toXML(isFolder));
		
	}

	@Override
	public void restore() throws InternalErrorException {
		JCRServlets servlets = JCRRepository.getServlets();	
		ItemDelegate itemToRestore = null;
		try {
			
			DelegateManager wrap = new DelegateManager(delegate, workspace.getOwner().getPortalLogin());
			itemToRestore = wrap.getNode(getName());
			ItemDelegate originalParentNode = null;
			try{	
				originalParentNode = servlets.getItemById(getOriginalParentId());
				logger.info("Original Parent Item path: " + originalParentNode.getPath());
			}catch (Exception e) {
				logger.warn("The original Parent Node does not exist anymore, the item will be restored to the root");
			}


			ItemDelegate parentNode = null;

			if (originalParentNode != null) {
				boolean inTrash = workspace.isInTrash(originalParentNode);
				logger.info(originalParentNode.getPath() + " is In Trash? " + inTrash);

				if (inTrash){
					parentNode = servlets.getItemById(workspace.getRoot().getId());
					logger.info("The file " + getName()  + " will be restored in the root");
				}
				else{
					parentNode = originalParentNode;
					logger.info("The file " + getName()  + " will be restored to the original parent");
				}
			}else
				parentNode = servlets.getItemById(workspace.getRoot().getId());
			

			//if a file with the same name already exists to the destination path, 
			//the item will be restored using time stamp + original name
			//es. "2012_04_05_11400029_MyAwesomeFile.txt"
			try {
//				System.out.println("GET NAME " + getName());
//				System.out.println("parentNode.getId() " + parentNode.getPath());
				if	(workspace.exists(getName(), parentNode.getId())){

					logger.info(getName() + " already exists in " + parentNode.getPath());
					try{
						String time = timeStampRename();
						setName(itemToRestore, time + getName());
					}catch (Exception e1) {
						throw new InternalErrorException(e1);
					}
				}
			} catch (ItemNotFoundException e1) {
				// TODO Auto-generated catch block
				e1.printStackTrace();
			} catch (WrongItemTypeException e1) {
				// TODO Auto-generated catch block
				e1.printStackTrace();
			}

			try {
				workspace.moveItem(itemToRestore.getId(), parentNode.getId());
			} catch (ItemNotFoundException | WrongDestinationException
					| InsufficientPrivilegesException
					| ItemAlreadyExistException
					| WorkspaceFolderNotFoundException e) {
				throw new InternalErrorException(e);
			}


			//remove trash item (just the folder)
			ItemDelegate folderToRemove = servlets.getItemById(delegate.getId());
			logger.info("Remove folder: " + folderToRemove.getPath());
			DelegateManager wrap1 = new DelegateManager(folderToRemove, workspace.getOwner().getPortalLogin());
			wrap1.remove();

		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} catch (RemoteBackendException e) {
			throw new InternalErrorException(e);
		} catch (ItemNotFoundException e) {
			throw new InternalErrorException(e);
		}

	}


	/**
	 * Get Time Stamp to rename a file
	 * @return
	 */
	private static String timeStampRename() {
		//		"2012_04_05_11400029_MyAwesomeFile.txt"
		Date date = new Date();
		SimpleDateFormat sdf = new SimpleDateFormat("yyyy_MM_dd_HHmmssss_");
		String formattedDate = sdf.format(date);
		//		System.out.println(formattedDate);
		return formattedDate;

	}



	@Override
	public void deletePermanently() throws InternalErrorException {

		JCRServlets servlets = null;
		try {
			servlets = JCRRepository.getServlets();
			ItemDelegate node = servlets.getItemById(getId());
			WorkspaceItem item = workspace.getWorkspaceItem(node);
			logger.debug("deletePermanently node: " + item.getName());

			//remove the content of the trash item
			try{
				workspace.getStorage().removeRemoteFolder(node.getPath(), workspace.getOwner().getPortalLogin());
				//				removeRemoteContent(node.getNode(item.getName()), node.getNode(item.getName()).getType());
			}catch (Exception e) {
				logger.warn("Error removing " + node.getPath() + " from storage", e);
			}

			//remove the trash item
			DelegateManager wrap = new DelegateManager(node, workspace.getOwner().getPortalLogin());
			wrap.remove();
		} catch (RepositoryException e) {
			throw new InternalErrorException(e);
		} 
	}


	@Override
	public WorkspaceItemType getType() {
		return WorkspaceItemType.TRASH_ITEM;
	}



	public long getLength() throws InternalErrorException {
		return (Long) new XStream().fromXML(delegate.getProperties().get(NodeProperty.LENGTH));
	}

	@Override
	public String getMimeType() throws InternalErrorException {
		return delegate.getProperties().get(NodeProperty.TRASH_ITEM_MIME_TYPE);
	}


	@Override
	public String getDeletedFrom() {
		return delegate.getProperties().get(NodeProperty.DELETED_FROM);
	}


	//getter methods
	@Override
	public String getOriginalParentId() {
		return delegate.getProperties().get(NodeProperty.ORIGINAL_PARENT_ID);
	}

	@Override
	public Calendar getDeletedTime() {
		return (Calendar) new XStream().fromXML(delegate.getProperties().get(NodeProperty.DELETE_DATE));
	}

	@Override
	public String getDeletedBy() {
		return delegate.getProperties().get(NodeProperty.DELETE_BY);
	}

	@Override
	public boolean isFolder() {
		return (Boolean) new XStream().fromXML(delegate.getProperties().get(NodeProperty.IS_FOLDER));
	}


	@Override
	public String getName() throws InternalErrorException {	

		String name = "";
//		 
//		ItemDelegate node = null;
//		JCRServlets servlets = null;
		try {
//			servlets = JCRRepository.getServlets();
//			node = servlets.getItemById(delegate.getId());
			name =	delegate.getProperties().get(NodeProperty.TRASH_ITEM_NAME);
		} catch (Exception e) {
			throw new InternalErrorException(e);
		} 

		return name;
	}


	public void setName(ItemDelegate node, String name) throws RepositoryException {

		try {
			JCRServlets servlets = JCRRepository.getServlets();
			ItemDelegate parent = servlets.getItemById(node.getParentId());
			String newPath = parent.getPath() 
					+ workspace.getPathSeparator() + name;
			
			node.setTitle(name);
			DelegateManager wrap = new DelegateManager(node, workspace.getOwner().getPortalLogin());
			wrap.save();
			servlets.move(node.getPath(), newPath);
		} catch (Exception e) {
			logger.error("Error setting a new name to WorkspaceTrashItem: " + e);
		} 

	}



	@Override
	public void removeChild(WorkspaceItem child) throws InternalErrorException,
	InsufficientPrivilegesException {}




}
