package org.gcube.common.iam;

import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import org.gcube.com.fasterxml.jackson.annotation.JsonIgnore;
import org.gcube.com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import org.gcube.com.fasterxml.jackson.annotation.JsonProperty;
import org.gcube.com.fasterxml.jackson.annotation.JsonSetter;
import org.gcube.common.keycloak.model.AccessToken;

@JsonIgnoreProperties(ignoreUnknown = true)
public class D4ScienceJWTObject extends AccessToken {

    protected final static List<String> MINIMAL_ROLES = Arrays.asList("Member");

    private String clientId;

    // "name", as the client pretty name, e.g. Catalogue for gcat, Workspace for
    // storage-hub
    @JsonProperty("name")
    private String clientName;

    // username of the user who requested such client
    @JsonProperty("contact_person")
    private String contactPerson;

    // the name of the organisation / community using such client. D4Science will be
    // used for internal clients.
    @JsonProperty("contact_organisation")
    private String contactOrganisation;

    private static final String INTERNAL_CLIENT_ORGANISATION_NAME = "D4Science";

    public D4ScienceJWTObject() {
    }
    // Custom getters for specific claims, potentially overriding or adapting
    // behavior
    public String getUsername() {
        return super.getPreferredUsername();
    }

    public List<String> getContextRoles() {
        // Assuming context is 'aud' from IDToken, which is String[]
        // Need to convert AccessToken.Access roles to List<String>
        String currentContext = getRawContext();
        if (currentContext != null && getResourceAccess() != null) {
            AccessToken.Access access = getResourceAccess().get(currentContext);
            if (access != null && access.getRoles() != null) {
                return new ArrayList<>(access.getRoles());
            }
        }
        return MINIMAL_ROLES;
    }

    public List<String> getRoles() {
        return getContextRoles(); // Delegate to context-specific roles
    }

    public List<String> getRealmRoles() {
        if (getRealmAccess() != null && getRealmAccess().getRoles() != null) {
            return new ArrayList<>(getRealmAccess().getRoles());
        }
        return Collections.emptyList();
    }

    public String getRawContext() {
        String[] audience = getAudience();
        if (audience != null && audience.length > 0) {
            return audience[0];
        }
        return null;
    }

    public String getContext() {
        String raw_audience = getRawContext();
        if (raw_audience == null)
            return null;
        try {
            return URLDecoder.decode(raw_audience, StandardCharsets.UTF_8.toString());
        } catch (UnsupportedEncodingException e) {
            return raw_audience;
        }
    }

    @JsonIgnore
    public boolean isExternalService() {
        return contactOrganisation != null && contactOrganisation.equals(INTERNAL_CLIENT_ORGANISATION_NAME);
    }

    @JsonIgnore
    public boolean isApplication() {
        return clientId != null;
    }

    @JsonSetter("client_id")
    public void setClientId(String clientId) {
        this.clientId = clientId;
    }

    /**
     * Maintianing the backward compatibility with the old name
     *
     * @param clientId
     */
    @JsonSetter("clientId")
    private void setMediaAsset(String clientId) {
        if (this.clientId == null) {
            setClientId(clientId);
        }
    }

    public String getClientName() {
        // Use the custom clientName field if available, otherwise fallback to name from
        // IDToken or other claims
        if (this.clientName != null) {
            return this.clientName;
        }
        return super.getName(); // IDToken.name
    }

    public void setClientName(String clientName) {
        this.clientName = clientName;
    }

    public String getContactPerson() {
        return contactPerson;
    }

    public void setContactPerson(String contactPerson) {
        this.contactPerson = contactPerson;
    }

    public String getContactOrganisation() {
        return contactOrganisation;
    }

    public void setContactOrganisation(String contactOrganisation) {
        this.contactOrganisation = contactOrganisation;
    }
}
