package org.gcube.application.framework.harvesting.db;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.HashMap;
import javax.xml.transform.TransformerException;
import org.gcube.application.framework.harvesting.common.ElementGenerator;
import org.gcube.application.framework.harvesting.common.db.exceptions.DBConnectionException;
import org.gcube.application.framework.harvesting.common.db.exceptions.SourceIDNotFoundException;
import org.gcube.application.framework.harvesting.common.db.tools.SourcePropsTools;
import org.gcube.application.framework.harvesting.common.dbXMLObjects.DBSource;
import org.gcube.application.framework.harvesting.common.dbXMLObjects.Edge;
import org.gcube.application.framework.harvesting.common.dbXMLObjects.Table;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

/**
 * THIS CLASS SHOULD BE USED ONLY FOR TESTING PURPOSES. Use the DBDataStax instead !
 * @author nikolas
 *
 */
public class DBDataDom {

	private DBSource dbSource;
	private Connection conn;
	
//	private static enum DBTYPE { mysql, postgresql, sqlite }
	private static HashMap<String, String> dbNameToDriverClass;
	
	static{
		dbNameToDriverClass = new HashMap<String, String>();
		dbNameToDriverClass.put("mysql", "com.mysql.jdbc.Driver");
		dbNameToDriverClass.put("postgresql", "org.postgresql.Driver");
		dbNameToDriverClass.put("sqlite", "org.sqlite.JDBC");
		
	}
	
	public DBDataDom(String sourceID, String filepath){
		if(filepath==null) //when it is called from the servlet, it is null. When called from the testing class, the path string is provided. 
			filepath = System.getProperty("catalina.base")+"/shared/d4s/sqlDBMappings.xml";
		try {
			dbSource = SourcePropsTools.parseSourceProps(filepath,sourceID);
		} catch (SourceIDNotFoundException e) {
			e.printStackTrace();
		}
	}
	
	
	public DBSource getUserProps(){
		return dbSource;
	}
	
	public String getUserData() throws SQLException, DBConnectionException{
		Document doc = ElementGenerator.getDocument();
		String rootTable = "";
		ArrayList<Table> tables = dbSource.getTables();
		ArrayList<Edge> edges = dbSource.getEdges();
		
		//1. find the root table
		for(Table table : tables){
			boolean root = true;
			for(Edge edge : edges){
				if(table.getName().equalsIgnoreCase(edge.getChild()))
					root = false;
			}
			if(root){
				rootTable = table.getName();
				break;
			}
		}
//		System.out.println("root table is : "+rootTable);
		//2. initiate a connection to the database and retrieve the tuples of the root table 
		conn = connectToDatabaseOrDieTrying();
		
		Element root = doc.createElement("root");
		
		root.appendChild(formSubElement(rootTable, new HashMap<String, String>()));

//		System.out.println("Total sql queries: "+WriteFile.getTotalQueries());
		try {
//			WriteFile.writeOnFile(ElementGenerator.domToXML(root));
			return ElementGenerator.domToXML(root);
		} catch (TransformerException e) {
			return e.getMessage();
		}
		
	}
	
	/**
	 * Recursive function to provide the tree structure values
	 */
	public Element formSubElement(String tableName, HashMap<String, String> keysVals) throws SQLException{
		Document doc = ElementGenerator.getDocument();
		ResultSet rs = getFilteredTableResultSet(tableName, conn, keysVals);
		ResultSetMetaData fields = rs.getMetaData();
		Element elementList = doc.createElement(tableName+"_list");
		
		ArrayList<Edge> edges = SourcePropsTools.getEdges(dbSource, tableName);
		while(rs.next()){
			Element subTable = doc.createElement(tableName);

			for(int i=1;i<=fields.getColumnCount();i++){
				Element el = doc.createElement(fields.getColumnLabel(i));
				el.appendChild(doc.createTextNode(rs.getString(i)));
				subTable.appendChild(el);
			}
			for(Edge edge : edges){
				HashMap<String, String> kv = new HashMap<String,String>();
				String[] pkeys = edge.getPKeys().split(",");
				String[] ckeys = edge.getCKeys().split(",");
				for(int i=0;i<ckeys.length;i++)
					kv.put(ckeys[i], rs.getString(pkeys[i]));
				Element e = formSubElement(edge.getChild(), kv);
				if(e.hasChildNodes())
					subTable.appendChild(e);
			}
			elementList.appendChild(subTable);
		}	
		return elementList;
	}
	
	
	
	private ResultSet getDefaultTableResultSet(String table, Connection conn) throws SQLException {
		Statement st = conn.createStatement();
		return st.executeQuery(SourcePropsTools.getSqlOfTable(dbSource, table));

	}
	
	/**
	 * Key and value (within keysValues) will be appended on the end of the default sql query. (e.g. "where key=value")
	 */
	private ResultSet getFilteredTableResultSet(String table, Connection conn, HashMap<String,String> keysValues) throws SQLException {
		Statement st = conn.createStatement();
		String sql = SourcePropsTools.getSqlOfTable(dbSource, table);
		if(keysValues.isEmpty())
			return st.executeQuery(sql);
			
		if(sql.toLowerCase().contains(" where ")){ //means that it already has a 'where' filter, so we append our filter with an 'and' 
			for(String key : keysValues.keySet())
				sql += " and "+key+"='"+keysValues.get(key)+"'";
		}
		else{
			sql += " where ";
			for(String key : keysValues.keySet())
				sql += key+"='"+keysValues.get(key)+"' and ";
		}
		if(sql.endsWith(" and "))//remove trailing " and "
			sql = sql.substring(0, sql.length()-5); //5 characters
		return st.executeQuery(sql);

	}
	
	
	private Connection connectToDatabaseOrDieTrying() throws DBConnectionException {
		Connection conn = null;
		try {
			Class.forName(dbNameToDriverClass.get(dbSource.getDBType()));
			String url = "jdbc:"+dbSource.getDBType()+"://"+dbSource.getHost()+"/"+dbSource.getDBName();
			conn = DriverManager.getConnection(url, dbSource.getDBName(), dbSource.getDBPassword());
		} catch (ClassNotFoundException e) {
			throw new DBConnectionException("Could not find the jdbc connection class", e);
		} catch (SQLException e) {
			throw new DBConnectionException("Error connecting to the database through the jdbc class ", e);
		}
		return conn;
	}
	
}
