package org.gcube.application.geoportal.service.engine.providers;

import lombok.extern.slf4j.Slf4j;
import org.gcube.application.cms.plugins.InitializablePlugin;
import org.gcube.application.cms.plugins.Plugin;
import org.gcube.application.cms.plugins.faults.InitializationException;
import org.gcube.application.cms.plugins.faults.ShutDownException;
import org.gcube.application.cms.plugins.reports.InitializationReport;
import org.gcube.application.geoportal.common.utils.ContextUtils;
import org.gcube.application.geoportal.service.model.internal.faults.ConfigurationException;

import org.reflections.Reflections;
import org.reflections.util.ConfigurationBuilder;
import org.reflections.util.FilterBuilder;

import java.util.HashMap;
import java.util.Map;
import java.util.ServiceLoader;

@Slf4j
public class PluginManager extends AbstractScopedMap<Map<String, Plugin>>{

    Map<String,Plugin> implementations=new HashMap<>();


    public PluginManager(){
        super("Plugin Cache");
        // reflections

        Reflections reflections = new Reflections(
                new ConfigurationBuilder()
                        .forPackage("org.gcube.application.cms")
                        .filterInputsBy(new FilterBuilder().includePackage("org.gcube.application.cms")));

        reflections.getSubTypesOf(Plugin.class).iterator().forEachRemaining(pluginClass->{
            if(!pluginClass.isInterface()){
                try {
                    Plugin plugin = pluginClass.newInstance();
                    log.debug("Loading {} descriptiorn : ", plugin, plugin.getDescriptor());
                    implementations.put(plugin.getDescriptor().getId(), plugin);
                }catch (Throwable t){
                    log.warn("Unable to instantiate Plugin "+pluginClass,t);
                }
            }

            });

        log.info("Loaded {} plugins",implementations.keySet().size());

        // Init plugins
        implementations.forEach((id,p)->{
            if(p instanceof InitializablePlugin){
                log.info("INIT Plugin {}",id);
                try {
                    InitializablePlugin ip=(InitializablePlugin) p;
                    InitializationReport rep=ip.init();
                    log.info("INIT REPORT : {}",rep);
                }catch (InitializationException e){
                    log.error("Failed to initialize "+id,e);
                }catch(Throwable t){
                    log.error("Unable to initialize "+id,t);
                }
            }
        });
    }


    @Override
    protected Map<String, Plugin> retrieveObject() throws ConfigurationException {
        // Init plugins
        implementations.forEach((id,p)->{
            if(p instanceof InitializablePlugin){
                log.info("INIT Plugin {} in context {} ",id, ContextUtils.getCurrentScope());
                try {
                    InitializablePlugin ip=(InitializablePlugin) p;
                    InitializationReport rep=ip.initInContext();
                    log.info("INIT REPORT : {}",rep);
                }catch (InitializationException e){
                    log.error("Failed to initialize "+id,e);
                }catch(Throwable t){
                    log.error("Unable to initialize "+id,t);
                }
            }
        });
        return implementations;
    }

    @Override
    protected void dispose(Map<String, Plugin> toDispose) {
        // ShutDown plugins
        implementations.forEach((id,p)->{
            if(p instanceof InitializablePlugin){
                log.info("Shutting down Plugin {}",id);
                try {
                    InitializablePlugin ip=(InitializablePlugin) p;
                    ip.shutdown();
                }catch (ShutDownException e){
                    log.error("Failed to shutdown "+id,e);
                }catch(Throwable t){
                    log.error("Unable to shutdown "+id,t);
                }
            }
        });
    }

    @Override
    public void init() {

    }
}
