package org.gcube.application.geoportalcommon;

import java.io.Serializable;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;

import org.gcube.application.geoportal.common.model.legacy.AbstractRelazione;
import org.gcube.application.geoportal.common.model.legacy.AccessPolicy;
import org.gcube.application.geoportal.common.model.legacy.BBOX;
import org.gcube.application.geoportal.common.model.legacy.Concessione;
import org.gcube.application.geoportal.common.model.legacy.LayerConcessione;
import org.gcube.application.geoportal.common.model.legacy.OtherContent;
import org.gcube.application.geoportal.common.model.legacy.PersistedContent;
import org.gcube.application.geoportal.common.model.legacy.RelazioneScavo;
import org.gcube.application.geoportal.common.model.legacy.UploadedImage;
import org.gcube.application.geoportal.common.model.legacy.WorkspaceContent;
import org.gcube.application.geoportal.common.model.legacy.report.ValidationReport;
import org.gcube.application.geoportalcommon.shared.gis.BoundsMap;
import org.gcube.application.geoportalcommon.shared.products.BaseConcessioneDV;
import org.gcube.application.geoportalcommon.shared.products.ConcessioneDV;
import org.gcube.application.geoportalcommon.shared.products.content.OtherContentDV;
import org.gcube.application.geoportalcommon.shared.products.content.WorkspaceContentDV;
import org.gcube.application.geoportalcommon.shared.products.model.AbstractRelazioneScavoDV;
import org.gcube.application.geoportalcommon.shared.products.model.LayerConcessioneDV;
import org.gcube.application.geoportalcommon.shared.products.model.RelazioneScavoDV;
import org.gcube.application.geoportalcommon.shared.products.model.UploadedImageDV;
import org.gcube.application.geoportalcommon.shared.products.model.ValidationReportDV;
import org.gcube.application.geoportalcommon.shared.products.model.ValidationReportDV.ValidationStatus;
import org.gcube.application.geoportalcommon.util.DateUtils;
import org.gcube.application.geoportalcommon.util.URLParserUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The Class ConvertToGUIModel.
 *
 * @author Francesco Mangiacrapa at ISTI-CNR (francesco.mangiacrapa@isti.cnr.it)
 * 
 *         Nov 2, 2020
 */
public class ConvertToDataViewModel {

	private static final String NO_TIME = "T00:00";

	/** The Constant LOG. */
	private static final Logger LOG = LoggerFactory.getLogger(ConvertToDataViewModel.class);

	public static final String DATE_FORMAT = "dd-MM-yyyy";

	public static final String HOURS_MINUTES_SEPARATOR = ":";

	public static final String TIME_FORMAT = "HH" + HOURS_MINUTES_SEPARATOR + "mm";

	/**
	 * To concessione.
	 *
	 * @param concessione the concessione
	 * @return the concessione
	 * @throws Exception the exception
	 */
	public static ConcessioneDV toConcessione(Concessione concessione) throws Exception {
		LOG.debug("called toConcessione for: " + concessione);

		if (concessione == null)
			return null;

		try {

			ConcessioneDV theConcessione = new ConcessioneDV();
			theConcessione.setPolicy(toPolicy(concessione.getPolicy()));
			// theConcessione.setAuthors(concessione.getAuthors());
			theConcessione.setCentroidLat(concessione.getCentroidLat());
			theConcessione.setCentroidLong(concessione.getCentroidLong());
			theConcessione.setContributore(concessione.getContributore());
			theConcessione.setCreationTime(DateUtils.asDate(concessione.getCreationTime()));
			theConcessione.setCreationUser(toUser(concessione.getCreationUser()));
			theConcessione.setDataInizioProgetto(DateUtils.asDate(concessione.getDataInizioProgetto()));
			theConcessione.setDataFineProgetto(DateUtils.asDate(concessione.getDataFineProgetto()));
			theConcessione.setDescrizioneContenuto(concessione.getDescrizioneContenuto());
			theConcessione.setEditore(concessione.getEditore());
			theConcessione.setFontiFinanziamento(concessione.getFontiFinanziamento());
			theConcessione.setItemId(concessione.getMongo_id());
			theConcessione.setNome(concessione.getNome());

			theConcessione.setIntroduzione(concessione.getIntroduzione());
			theConcessione.setLastUpdateTime(DateUtils.asDate(concessione.getLastUpdateTime()));
			theConcessione.setLastUpdateUser(toUser(concessione.getLastUpdateUser()));
			theConcessione.setLicenzaID(concessione.getLicenzaID());
			theConcessione.setParoleChiaveICCD(concessione.getParoleChiaveICCD());
			theConcessione.setParoleChiaveLibere(concessione.getParoleChiaveLibere());

			theConcessione.setResponsabile(concessione.getResponsabile());
			theConcessione.setRisorseCorrelate(concessione.getRisorseCorrelate());
			theConcessione.setSoggetto(concessione.getSoggetto());
			theConcessione.setTitolariCopyright(concessione.getTitolareCopyright());
			theConcessione.setTitolariLicenza(concessione.getTitolareLicenza());
			theConcessione.setTitolari(concessione.getTitolari());
			theConcessione.setVersion(concessione.getVersion());

			theConcessione.setRecordType(concessione.getRecordType().name());

			theConcessione.setAbstractRelazioneScavo(toAbstractRelazioneScavo(concessione.getAbstractRelazione()));
			theConcessione.setRelazioneScavo(toRelazioneScavo(concessione.getRelazioneScavo()));

			if (concessione.getImmaginiRappresentative() != null) {
				List<UploadedImageDV> uploadedImagesDV = new ArrayList<UploadedImageDV>(
						concessione.getImmaginiRappresentative().size());
				for (UploadedImage ui : concessione.getImmaginiRappresentative()) {
					uploadedImagesDV.add(toUploadedImage(ui));
				}
				theConcessione.setImmaginiRappresentative(uploadedImagesDV);
			}

			if (concessione.getGenericContent() != null) {
				List<OtherContentDV> otherContentsDV = new ArrayList<OtherContentDV>(
						concessione.getGenericContent().size());
				for (OtherContent gc : concessione.getGenericContent()) {
					otherContentsDV.add(toOtherContentDV(gc));
				}
				theConcessione.setGenericContent(otherContentsDV);
			}

			if (concessione.getPianteFineScavo() != null) {
				List<LayerConcessioneDV> piantaScavoDV = new ArrayList<LayerConcessioneDV>(
						concessione.getPianteFineScavo().size());
				for (LayerConcessione lc : concessione.getPianteFineScavo()) {
					// TODO CHECK WHY I HAVE TO PASS theConcessione = null IN ORDER TO AVOID GWT
					// serialization error
					piantaScavoDV.add(toLayerConcessione(lc, null));
				}
				theConcessione.setPianteFineScavo(piantaScavoDV);
			}

			if (concessione.getPosizionamentoScavo() != null) {
				// TODO CHECK WHY I HAVE TO PASS theConcessione = null IN ORDER TO AVOID GWT
				// serialization error
				LayerConcessioneDV thePosizScavo = toLayerConcessione(concessione.getPosizionamentoScavo(), null);
				theConcessione.setPosizionamentoScavo(thePosizScavo);
			}

			LOG.debug("Returning concessione: " + theConcessione);
			return theConcessione;
		} catch (Exception e) {
			LOG.error("Error on converting concessione: " + concessione, e);
			return null;
		}

	}

	/**
	 * To metadata concessione.
	 *
	 * @param concessione             the concessione
	 * @param includeValidationReport the include validation report
	 * @return the concessione DV
	 * @throws Exception the exception
	 */
	public static ConcessioneDV toMetadataConcessione(Concessione concessione, boolean includeValidationReport)
			throws Exception {
		LOG.debug("called toConcessione for: " + concessione);

		if (concessione == null)
			return null;

		try {

			ConcessioneDV theConcessione = new ConcessioneDV();
			theConcessione.setPolicy(toPolicy(concessione.getPolicy()));
			theConcessione.setCentroidLat(concessione.getCentroidLat());
			theConcessione.setCentroidLong(concessione.getCentroidLong());
			theConcessione.setContributore(concessione.getContributore());
			theConcessione.setCreationTime(DateUtils.asDate(concessione.getCreationTime()));
			theConcessione.setCreationUser(toUser(concessione.getCreationUser()));
			theConcessione.setDataInizioProgetto(DateUtils.asDate(concessione.getDataInizioProgetto()));
			theConcessione.setDataFineProgetto(DateUtils.asDate(concessione.getDataFineProgetto()));
			theConcessione.setDescrizioneContenuto(concessione.getDescrizioneContenuto());
			theConcessione.setEditore(concessione.getEditore());
			theConcessione.setFontiFinanziamento(concessione.getFontiFinanziamento());
			theConcessione.setItemId(concessione.getMongo_id());
			theConcessione.setNome(concessione.getNome());
			theConcessione.setAuthors(concessione.getAuthors());
			theConcessione.setCreationTime(DateUtils.asDate(concessione.getCreationTime()));

			theConcessione.setIntroduzione(concessione.getIntroduzione());
			theConcessione.setLastUpdateTime(DateUtils.asDate(concessione.getLastUpdateTime()));
			theConcessione.setLastUpdateUser(toUser(concessione.getLastUpdateUser()));
			theConcessione.setLicenzaID(concessione.getLicenzaID());
			theConcessione.setParoleChiaveICCD(concessione.getParoleChiaveICCD());
			theConcessione.setParoleChiaveLibere(concessione.getParoleChiaveLibere());

			theConcessione.setResponsabile(concessione.getResponsabile());
			theConcessione.setTitolariCopyright(concessione.getTitolareCopyright());
			theConcessione.setTitolariLicenza(concessione.getTitolareLicenza());
			theConcessione.setTitolari(concessione.getTitolari());

			if (includeValidationReport) {
				theConcessione.setValidationReport(toValidationReport(concessione.getReport()));
				if (theConcessione.getValidationReport() != null)
					theConcessione.setValidationStatus(theConcessione.getValidationReport().getStatus());
			}

			LOG.debug("Returning base concessione: " + theConcessione);
			return theConcessione;
		} catch (Exception e) {
			LOG.error("Error on converting concessione: " + concessione, e);
			return null;
		}

	}

	/**
	 * To validation report.
	 *
	 * @param validationReport the validation report
	 * @return the validation report DV
	 * @throws Exception the exception
	 */
	public static ValidationReportDV toValidationReport(ValidationReport validationReport) throws Exception {
		LOG.debug("called toValidationReport for: " + validationReport);

		if (validationReport == null)
			return null;

		ValidationReportDV theVR = new ValidationReportDV();
		theVR.setErrorMessages(validationReport.getErrorMessages());
		theVR.setObjectName(validationReport.getObjectName());
		theVR.setStatus(ValidationStatus.valueOf(validationReport.getStatus().name()));
		theVR.setWarningMessages(validationReport.getWarningMessages());
		theVR.setAsJSONString(toJSON(validationReport));
		return theVR;
	}

	/**
	 * To base concessione.
	 *
	 * @param concessione the concessione
	 * @return the concessione DV
	 * @throws Exception the exception
	 */
	public static BaseConcessioneDV toBaseConcessione(Concessione concessione) throws Exception {
		LOG.debug("called toBaseConcessione for: " + concessione);

		if (concessione == null)
			return null;

		try {

			BaseConcessioneDV theConcessione = new BaseConcessioneDV();
			theConcessione.setPolicy(toPolicy(concessione.getPolicy()));
			theConcessione.setCreationTime(DateUtils.asDate(concessione.getCreationTime()));
			theConcessione.setCreationUser(toUser(concessione.getCreationUser()));
			theConcessione.setItemId(concessione.getMongo_id());
			theConcessione.setNome(concessione.getNome());
			theConcessione.setAuthors(concessione.getAuthors());
			theConcessione.setCreationTime(DateUtils.asDate(concessione.getCreationTime()));

			theConcessione.setIntroduzione(concessione.getIntroduzione());
			theConcessione.setLastUpdateTime(DateUtils.asDate(concessione.getLastUpdateTime()));
			theConcessione.setLastUpdateUser(toUser(concessione.getLastUpdateUser()));
			theConcessione.setLicenzaID(concessione.getLicenzaID());

			LOG.debug("Returning base concessione: " + theConcessione);
			return theConcessione;
		} catch (Exception e) {
			LOG.error("Error on converting concessione: " + concessione, e);
			return null;
		}

	}

	// TO AVOID SERIALIZATION ISSUE AGAINST GWT
//	public static List<String> toListString(List<String> orginList){
//		if(orginList==null)
//			return null;
//		
//		List<String> destList = new ArrayList<String>(orginList.size());
//		for (String orgValue : orginList) {
//			destList.add(orgValue);
//		}
//		
//		return destList;
//	}

	/**
	 * To user.
	 *
	 * @param username the username
	 * @return the string
	 */
	public static String toUser(String username) {

		if (username == null)
			return null;

		return username;

	}

	/**
	 * To policy.
	 *
	 * @param policy the policy
	 * @return the string
	 */
	public static String toPolicy(AccessPolicy policy) {

		if (policy == null)
			return null;

		return policy.name();

	}

	/**
	 * To date format string.
	 *
	 * @param dateTime the date time
	 * @return the string
	 */
	public static String toDateFormatString(LocalDateTime dateTime) {

		if (dateTime == null)
			return null;

		String time = dateTime.toString();
		DateTimeFormatter formatter = null;
		try {
			if (!time.endsWith(NO_TIME)) {
				formatter = DateTimeFormatter.ofPattern(DATE_FORMAT + " " + TIME_FORMAT);
			} else {
				time = time.replace(NO_TIME, "");
				formatter = DateTimeFormatter.ofPattern(DATE_FORMAT);
			}
		} catch (Exception e) {
			LOG.warn("Parsing error: ", e);
		}

		try {
			if (formatter != null)
				return dateTime.format(formatter);
		} catch (Exception e) {
			LOG.warn("Date format error: ", e);
		}

		return dateTime.toString();
	}

	/**
	 * To abstract relazione scavo.
	 *
	 * @param abstractRelazione the abstract relazione
	 * @return the abstract relazione scavo DV
	 */
	public static AbstractRelazioneScavoDV toAbstractRelazioneScavo(AbstractRelazione abstractRelazione) {

		if (abstractRelazione == null)
			return null;

		AbstractRelazioneScavoDV theARelDiScavo = new AbstractRelazioneScavoDV();
		theARelDiScavo.setAbstractSectionIta(abstractRelazione.getAbstractIta());
		theARelDiScavo.setAbstractSectionEng(abstractRelazione.getAbstractEng());

		theARelDiScavo.setCreationTime(DateUtils.asDate(abstractRelazione.getCreationTime()));
		theARelDiScavo.setId(abstractRelazione.getId());
		theARelDiScavo.setLicenseID(abstractRelazione.getLicenseID());
		theARelDiScavo.setPolicy(toPolicy(abstractRelazione.getPolicy()));
		theARelDiScavo.setTitolo(abstractRelazione.getTitolo());

		List<PersistedContent> actContent = abstractRelazione.getActualContent();
		if (actContent != null && actContent.size() > 0) {
			List<WorkspaceContentDV> listWsContent = new ArrayList<WorkspaceContentDV>();

			for (PersistedContent content : actContent) {
				if (content != null && content instanceof WorkspaceContent) {
					listWsContent.add(toWorkspaceContent((WorkspaceContent) content));
				}
			}
			theARelDiScavo.setListWsContent(listWsContent);
		}

		LOG.debug("Returning: " + theARelDiScavo);
		return theARelDiScavo;
	}

	/**
	 * To relazione scavo.
	 *
	 * @param relazioneScavo the relazione scavo
	 * @return the relazione scavo data view
	 */
	public static RelazioneScavoDV toRelazioneScavo(RelazioneScavo relazioneScavo) {

		if (relazioneScavo == null)
			return null;

		RelazioneScavoDV theRelDiScavo = new RelazioneScavoDV();
		theRelDiScavo.setCreationTime(DateUtils.asDate(relazioneScavo.getCreationTime()));
		theRelDiScavo.setId(relazioneScavo.getId());
		theRelDiScavo.setLicenseID(relazioneScavo.getLicenseID());
		theRelDiScavo.setPolicy(toPolicy(relazioneScavo.getPolicy()));
		// theRelazioneDiScavo.setRecord(recordDV);
		theRelDiScavo.setResponsabili(relazioneScavo.getResponsabili());
		theRelDiScavo.setSoggetto(relazioneScavo.getSoggetto());
		theRelDiScavo.setTitolo(relazioneScavo.getTitolo());

		List<PersistedContent> actContent = relazioneScavo.getActualContent();
		if (actContent != null && actContent.size() > 0) {
			List<WorkspaceContentDV> listWsContent = new ArrayList<WorkspaceContentDV>();

			for (PersistedContent content : actContent) {
				if (content != null && content instanceof WorkspaceContent) {
					listWsContent.add(toWorkspaceContent((WorkspaceContent) content));
				}
			}
			theRelDiScavo.setListWsContent(listWsContent);
		}

		LOG.debug("Returning: " + theRelDiScavo);
		return theRelDiScavo;
	}

	/**
	 * To uploaded image.
	 *
	 * @param uploadedImage the uploaded image
	 * @return the uploaded image data-view
	 */
	public static UploadedImageDV toUploadedImage(UploadedImage uploadedImage) {

		if (uploadedImage == null)
			return null;

		UploadedImageDV theUploadedImageDV = new UploadedImageDV();
		theUploadedImageDV.setCreationTime(DateUtils.asDate(uploadedImage.getCreationTime()));
		theUploadedImageDV.setDidascalia(uploadedImage.getDidascalia());
		theUploadedImageDV.setFormat(uploadedImage.getFormat());
		theUploadedImageDV.setId(uploadedImage.getId());
		theUploadedImageDV.setLicenseID(uploadedImage.getLicenseID());
		theUploadedImageDV.setPolicy(toPolicy(uploadedImage.getPolicy()));
		// theUploadedImageDV.setRecord(recordDV);
		theUploadedImageDV.setResponsabili(uploadedImage.getResponsabili());
		theUploadedImageDV.setSoggetto(uploadedImage.getSoggetto());
		theUploadedImageDV.setTitolo(uploadedImage.getTitolo());

		List<PersistedContent> actContent = uploadedImage.getActualContent();
		if (actContent != null && actContent.size() > 0) {
			List<WorkspaceContentDV> listWsContent = new ArrayList<WorkspaceContentDV>();

			for (PersistedContent content : actContent) {
				if (content != null && content instanceof WorkspaceContent) {
					listWsContent.add(toWorkspaceContent((WorkspaceContent) content));
				}
			}
			theUploadedImageDV.setListWsContent(listWsContent);
		}

		LOG.debug("Returning: " + theUploadedImageDV);
		return theUploadedImageDV;
	}

	/**
	 * To workspace content.
	 *
	 * @param wContent the w content
	 * @return the workspace content DV
	 */
	public static WorkspaceContentDV toWorkspaceContent(WorkspaceContent wContent) {
		if (wContent == null)
			return null;

		WorkspaceContentDV theWSDV = new WorkspaceContentDV();
		theWSDV.setLink(wContent.getLink());
		theWSDV.setMimetype(wContent.getMimetype());
		theWSDV.setStorageID(wContent.getStorageID());
		theWSDV.setId(wContent.getId());
		theWSDV.setName(wContent.getName());

		return theWSDV;
	}

	/**
	 * To layer concessione.
	 *
	 * @param layerConcessione        the layer concessione
	 * @param refersToBaseConcessione the refers to base concessione
	 * @return the layer concessione data view
	 */
	public static LayerConcessioneDV toLayerConcessione(LayerConcessione layerConcessione,
			BaseConcessioneDV refersToBaseConcessione) {

		if (layerConcessione == null)
			return null;

		LOG.debug("Converting: " + layerConcessione);
		LayerConcessioneDV theLayerConessione = new LayerConcessioneDV();
		theLayerConessione.setAbstractSection(layerConcessione.getAbstractSection());

		String wmsLink = layerConcessione.getWmsLink();
		String layerName = null;
		BoundsMap bounds = null;

		// reading layer name from wmsLink
		// string bbox
		if (wmsLink != null) {
			layerName = URLParserUtil.extractValueOfParameterFromURL("layers", wmsLink);
			String bbox = URLParserUtil.extractValueOfParameterFromURL("bbox", wmsLink);
			String wmsVersion = URLParserUtil.extractValueOfParameterFromURL("version", wmsLink);
			bounds = toBoundMap(wmsVersion, bbox, ",");
			LOG.debug("Built bounds from wmsLInk: " + bounds);
		}

		if (layerName != null)
			theLayerConessione.setLayerName(layerName);
		else
			theLayerConessione.setLayerName(layerConcessione.getLayerName());

		theLayerConessione.setWmsLink(wmsLink);
		theLayerConessione.setLayerID(layerConcessione.getLayerID());
		theLayerConessione.setLayerUUID(layerConcessione.getLayerUUID());
		theLayerConessione.setAuthors(layerConcessione.getAuthors());

		if (bounds == null)
			bounds = toBoundMap(layerConcessione.getBbox());

		theLayerConessione.setBbox(bounds);
		theLayerConessione.setCreationTime(DateUtils.asDate(layerConcessione.getCreationTime()));
		theLayerConessione.setId(layerConcessione.getId());
		theLayerConessione.setLicenseID(layerConcessione.getLicenseID());
		theLayerConessione.setMetodoRaccoltaDati(layerConcessione.getMetodoRaccoltaDati());
		theLayerConessione.setPolicy(toPolicy(layerConcessione.getPolicy()));
		// theLayerConessione.setRecord(recordDV);
		theLayerConessione.setScalaAcquisizione(layerConcessione.getScalaAcquisizione());
		theLayerConessione.setSubTopic(layerConcessione.getSubTopic());
		theLayerConessione.setTitolo(layerConcessione.getTitolo());
		theLayerConessione.setTopicCategory(layerConcessione.getTopicCategory());
		theLayerConessione.setValutazioneQualita(layerConcessione.getValutazioneQualita());
		theLayerConessione.setRefersTo(refersToBaseConcessione);

		LOG.debug("Returning: " + theLayerConessione);
		return theLayerConessione;

	}

	/**
	 * To other content DV.
	 *
	 * @param otherContent the other content
	 * @return the other content DV
	 */
	public static OtherContentDV toOtherContentDV(OtherContent otherContent) {

		if (otherContent == null)
			return null;

		OtherContentDV theOtherContent = new OtherContentDV();
		theOtherContent.setCreationTime(DateUtils.asDate(otherContent.getCreationTime()));
		theOtherContent.setId(otherContent.getId());
		theOtherContent.setLicenseID(otherContent.getLicenseID());
		theOtherContent.setPolicy(toPolicy(otherContent.getPolicy()));
		// theOtherContent.setRecord(recordDV);
		theOtherContent.setTitolo(otherContent.getTitolo());

		return theOtherContent;
	}

	/**
	 * To bound map.
	 *
	 * @param bbox the bbox
	 * @return the bounds map
	 */
	public static BoundsMap toBoundMap(BBOX bbox) {

		if (bbox == null)
			return null;

		return new BoundsMap(bbox.getMinLong(), bbox.getMinLat(), bbox.getMaxLong(), bbox.getMaxLat(), null);
	}

	/**
	 * To bound map.
	 *
	 * @param wmsVersion the wms version
	 * @param bbox       the bbox
	 * @param separator  the separator
	 * @return the bounds map
	 */
	public static BoundsMap toBoundMap(String wmsVersion, String bbox, String separator) {

		if (bbox == null)
			return null;

		if (wmsVersion == null)
			return null;

		if (separator == null)
			separator = ",";

		try {

			String[] bboxArr = bbox.split(separator);

			if (wmsVersion.startsWith("1.3")) {
				// is 1.3.x
				return new BoundsMap(toDouble(bboxArr[1]), toDouble(bboxArr[0]), toDouble(bboxArr[3]),
						toDouble(bboxArr[2]), wmsVersion);
			} else {
				// should be 1.1.X
				return new BoundsMap(toDouble(bboxArr[0]), toDouble(bboxArr[1]), toDouble(bboxArr[2]),
						toDouble(bboxArr[3]), wmsVersion);
			}
		} catch (Exception e) {
			LOG.warn("Error on creating Bounds for wmsVersion " + wmsVersion + " and bbox " + bbox + " : ", e);
			return null;
		}
	}

	/**
	 * To double.
	 *
	 * @param value the value
	 * @return the double
	 */
	public static Double toDouble(String value) {
		try {
			return Double.parseDouble(value);
		} catch (Exception e) {
			LOG.warn("Error on parsing " + value + " as double: ", e);
			return null;
		}
	}

	/**
	 * To JSON.
	 *
	 * @param theObj the the obj
	 * @return the string
	 */
	public static String toJSON(Object theObj) {
		LOG.debug("toJSON called");
		try {

			if (theObj instanceof Serializable) {
				return org.gcube.application.geoportal.client.utils.Serialization.write(theObj);
			}
			throw new Exception("The input object is not serializable");

		} catch (Exception e) {
			LOG.warn("Error on deserializing: ", e);
			return null;
		}
	}

}
