package org.gcube.data.spd.gbifplugin.search;

import static org.gcube.data.trees.data.Nodes.e;
import static org.gcube.data.trees.data.Nodes.n;
import static org.gcube.data.trees.data.Nodes.t;

import java.io.InputStream;
import java.net.URL;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.Stack;

import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.events.XMLEvent;

import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.data.spd.gbifplugin.Constants;
import org.gcube.data.spd.model.Condition;
import org.gcube.data.spd.model.products.DataSet;
import org.gcube.data.spd.model.products.Product.ProductType;
import org.gcube.data.spd.model.products.ResultItem;
import org.gcube.data.spd.plugin.fwk.Labels;
import org.gcube.data.spd.plugin.fwk.writers.ObjectWriter;
import org.gcube.data.trees.data.Edge;
import org.gcube.data.trees.data.InnerNode;
import org.gcube.data.trees.data.Tree;

public class GBIFSearch extends AbstractSearch<ResultItem>{
	
	protected GCUBELog logger = new GCUBELog(GBIFSearch.class);
	
	
	private String scientificName;
	
	private String baseUrl;
	
	public GBIFSearch(String baseURL, String scientificName, Condition ...properties ) throws Exception{
		super("taxonConcepts", properties);
		this.scientificName = scientificName.replaceAll(" ", "%20");
		this.baseUrl = baseURL;
	}
	
	private static XMLInputFactory ifactory = XMLInputFactory.newInstance();
		
	public void search(ObjectWriter<ResultItem> writer ) throws Exception{
		search(writer, new URL(baseUrl+"taxon/list?scientificname="+scientificName+"&maxresults=100"));
	}
	
	private List<Tree> retrieveTaxons(XMLEventReader eventReader, boolean onlyUri, DataSet dataset) throws Exception{	
		List<Tree> trees = new ArrayList<Tree>();
		while (eventReader.hasNext()){
			
			XMLEvent event = eventReader.nextEvent();
			if (Utils.checkStartElement(event, "TaxonConcept")){
				Tree tree = retrieveEntireTaxon(event.asStartElement().getAttributeByName(Constants.ABOUT_ATTR).getValue());
				if (tree!=null){
					//InnerNode node = n();
					String taxonConceptKey =event.asStartElement().getAttributeByName(Constants.GBIFKEY_ATTR).getValue();
					//logger.trace("taxonKey is "+taxonConceptKey);
					String productsId = Utils.createProductsKey(this.scientificName, this.providerKey, this.dataSetKey, taxonConceptKey, Utils.elaborateProps(this.properties));
					Edge occurrenceEdge= e(Labels.PRODUCT_LABEL, n(productsId, e(Labels.TYPE_LABEL,ProductType.Occurrence), e(Labels.COUNT_LABEL,occurrencesCount(taxonConceptKey))));
					tree.add(e(Labels.PRODUCTS_LABEL,n(occurrenceEdge)));
					Calendar now = Calendar.getInstance();
					SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
					String credits = "Biodiversity occurrence data published by: "+dataset.getDataProvider().getName()+" (Accessed through GBIF Data Portal, data.gbif.org, "+format.format(now.getTime())+")";
					tree.add(e(Labels.CREDITS_LABEL, credits));
					trees.add(tree);
				}
			}if (Utils.checkEndElement(event, "taxonConcepts"))
				break;
			
		}
		return trees;
	}
	
	private Tree retrieveEntireTaxon(String resourceUri) throws Exception{
		URL url = new URL(resourceUri);
		InputStream is =url.openConnection().getInputStream();
		XMLEventReader eventReader =ifactory.createXMLEventReader(is);
		Stack<Edge> stack = new Stack<Edge>();
		String toTaxon= null;
		while (eventReader.hasNext()){
			XMLEvent event = eventReader.nextEvent();
			if (Utils.checkStartElement(event, "TaxonConcept")){
				String taxonKey = event.asStartElement().getAttributeByName(Constants.GBIFKEY_ATTR).getValue();
					
				
				Edge edge = retrieveTaxonIternal(eventReader,taxonKey, toTaxon);
				if (edge!=null){
					toTaxon=edge.target().attribute("taxonRelation");
					stack.push(edge);
				}
			}if (Utils.checkEndElement(event, this.elementsTag))
				break;
		}
		
		if (stack.isEmpty()) return null;
		
		Edge edge = stack.pop();
		while (!stack.isEmpty()){
			Edge tempEdge = stack.pop();
			((InnerNode)tempEdge.target()).add(edge);
			edge = tempEdge;
		}
		return t(edge.target().id(), ((InnerNode)edge.target()).edges().toArray(new Edge[0]));
	}
	
	private Edge retrieveTaxonIternal(XMLEventReader eventReader, String taxonKey, String oldTaxonValue) throws Exception{
		List<Edge> edges = new ArrayList<Edge>();
		
		boolean isChild= false;
		String toTaxon= null;
		String uri = "http://data.gbif.org/ws/rest/taxon/get/"+taxonKey;
		boolean isPrimary = false;
		boolean hasRelationship = false;
		while (eventReader.hasNext()){
			XMLEvent event = eventReader.nextEvent();
			if(Utils.checkStartElement(event, "nameComplete")){
				edges.add(e(Labels.SCIENTIFICNAME_LABEL,Utils.readCharacters(eventReader)));
			}if(Utils.checkStartElement(event, "rankString")){
				edges.add(e(Labels.RANK_LABEL,Utils.readCharacters(eventReader)));
			}if(Utils.checkStartElement(event, "accordingToString")){
				edges.add(e(Labels.CITATION_LABEL,Utils.readCharacters(eventReader)));
			}if (Utils.checkStartElement(event, "primary")){
				isPrimary = Utils.readCharacters(eventReader).equals("true");
			}if (Utils.checkStartElement(event, "Relationship")){
				hasRelationship = true;
				while(!Utils.checkEndElement(event, "Relationship")){
					event = eventReader.nextEvent();
					if (Utils.checkStartElement(event, "relationshipCategory"))
						isChild = event.asStartElement().getAttributeByName(Constants.RESOURCE_ATTR).getValue().equals(Constants.CHILD_RELATIONSHIP_VALUE); 
					if (Utils.checkStartElement(event, "toTaxon"))
						toTaxon = event.asStartElement().getAttributeByName(Constants.RESOURCE_ATTR).getValue();
				}
				if (isChild) break;
			}
			else if (Utils.checkEndElement(event, "TaxonConcept"))
				break;
		}
		if ((isPrimary) || (isChild && uri.equals(oldTaxonValue)) || (!hasRelationship && uri.equals(oldTaxonValue))) {
			Edge returnEdge = e("parent",n(taxonKey, edges.toArray(new Edge[edges.size()])));
			returnEdge.target().setAttribute("taxonRelation", toTaxon );
			return returnEdge;
		}
		else return null;		
	}

	
	private String occurrencesCount(String taxonKey) throws Exception{
		String occurrencesCount = "0";
		URL url = new URL("http://data.gbif.org/ws/rest/occurrence/count?scientificname="+scientificName+"&dataproviderkey="+this.providerKey+"&dataresourcekey="
						+this.dataSetKey+"&taxonconceptkey="+taxonKey+"&coordinatestatus=true"+Utils.elaborateProps(this.properties));
		InputStream is =url.openConnection().getInputStream();
		XMLEventReader countEventReader =ifactory.createXMLEventReader(is);
		while (countEventReader.hasNext()){
			XMLEvent countEvent = countEventReader.nextEvent();
			if (Utils.checkStartElement(countEvent, "summary")){
				occurrencesCount =countEvent.asStartElement().getAttributeByName(Constants.TOTAL_MATCHED_ATTR).getValue();
				break;
			}
		}
		return occurrencesCount;
	}

	@Override
	protected List<ResultItem> retrieveElements(XMLEventReader eventReader,
			boolean b, DataSet dataset) throws Exception {
		//logger.trace("dataSet is "+dataset.getName());
		List<ResultItem> results = new ArrayList<ResultItem>();
		for (Tree tree :retrieveTaxons(eventReader, true, dataset)){
			tree.add(e(Labels.DATASET_TAG,n(dataset.getId(),e(Labels.DATAPROVIDER_TAG, n(dataset.getDataProvider().getId(), e(Labels.NAME_TAG, dataset.getDataProvider().getName()))), e(Labels.CITATION_LABEL, dataset.getCitation()), e(Labels.NAME_TAG, dataset.getName()))));
			results.add(ResultItem.fromTree(tree));
		}
		return results;
	}
	
}
