package org.gcube.data.spd.gbifplugin.search;

import java.net.URL;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import javax.xml.stream.XMLEventReader;
import javax.xml.stream.events.XMLEvent;

import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.data.spd.gbifplugin.Constants;
import org.gcube.data.spd.model.BasisOfRecord;
import org.gcube.data.spd.model.Condition;
import org.gcube.data.spd.model.exceptions.IdNotValidException;
import org.gcube.data.spd.model.products.DataSet;
import org.gcube.data.spd.model.products.OccurrencePoint;
import org.gcube.data.spd.model.util.ElementProperty;
import org.gcube.data.spd.plugin.fwk.readers.LocalReader;
import org.gcube.data.spd.plugin.fwk.writers.ObjectWriter;
import org.gcube.data.spd.plugin.fwk.writers.Writer;
import org.gcube.data.spd.plugin.fwk.writers.rswrapper.LocalWrapper;

public class OccurrenceSearch  extends AbstractSearch<OccurrencePoint>{

	String baseUrl;
	
	public OccurrenceSearch(String baseUrl, Condition ...properties) throws Exception{
		super("occurrenceRecords");
		this.baseUrl= baseUrl;
	}

	public void search(ObjectWriter<OccurrencePoint> writer, String scientificName) throws Exception{
		search(writer, new URL(baseUrl+"occurrence/list?scientificname="+scientificName.replace(" ", "%20")+Utils.elaborateProps(properties)+"&coordinatestatus=true&format=darwin"));
	}
	
	public void searchByKey(ObjectWriter<OccurrencePoint> writer, String key) throws Exception{
		logger.trace(baseUrl+"occurrence/list?"+Utils.elaborateProductsKey(key)+"&coordinatestatus=true&format=darwin");
		search(writer, new URL(baseUrl+"occurrence/list?"+Utils.elaborateProductsKey(key)+"&coordinatestatus=true&format=darwin"));
	}
	
	public OccurrencePoint searchById(String id) throws Exception{
		LocalWrapper<OccurrencePoint> localWrapper = new LocalWrapper<OccurrencePoint>();
		
		search(new Writer<OccurrencePoint>(localWrapper), new URL(baseUrl+"occurrence/get/"+id+"?format=darwin"));
		LocalReader<OccurrencePoint> localReader = new LocalReader<OccurrencePoint>(localWrapper);
		OccurrencePoint returnObj;
		if (localReader.hasNext()) 
			returnObj = localReader.next();
		else throw new IdNotValidException();
		localWrapper.close();
		return returnObj;
		
	}
	
	private static GCUBELog logger = new GCUBELog(OccurrenceSearch.class);

		
	
		
	
	private static BasisOfRecord matchBasisOfRecord(String value){
		for (BasisOfRecord record: BasisOfRecord.values())
			if (record.name().toLowerCase().equals(value.toLowerCase())) return record;
		if (value.toLowerCase().equals("S".toLowerCase())) return BasisOfRecord.PreservedSpecimen;
		else if (value.toLowerCase().equals("O".toLowerCase())) return BasisOfRecord.HumanObservation;
		else if (value.toLowerCase().equals("fossil")) return BasisOfRecord.FossilSpecimen;
		return BasisOfRecord.HumanObservation;
	}

	private int total=0;
	
	@Override
	protected List<OccurrencePoint> retrieveElements(XMLEventReader eventReader, boolean b, DataSet dataset) throws Exception{
		List<OccurrencePoint> occurrencePointList = new ArrayList<OccurrencePoint>();
		while (eventReader.hasNext()){
			XMLEvent event = eventReader.nextEvent();
			if(Utils.checkStartElement(event, "TaxonOccurrence"))
				occurrencePointList.add(retrieveElement(eventReader, dataset, event.asStartElement().getAttributeByName(Constants.GBIFKEY_ATTR).getValue()));	
			else if (Utils.checkEndElement(event, this.elementsTag)){
				logger.trace("returned occurrences are "+occurrencePointList.size());
				total += occurrencePointList.size();
				logger.trace("total for now "+total);
				return occurrencePointList;
			}

		}
		throw new Exception("parsing exception");
	}

	private OccurrencePoint retrieveElement(XMLEventReader eventReader,  DataSet dataset, String occurrenceId) throws Exception{
		OccurrencePoint occurrence = new OccurrencePoint(occurrenceId);
		occurrence.setDataSet(dataset);
		Calendar now = Calendar.getInstance();
		SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
		String credits = "Biodiversity occurrence data published by: "+dataset.getDataProvider().getName()+" (Accessed through GBIF Data Portal, data.gbif.org, "+format.format(now.getTime())+")";
		occurrence.setCredits(credits);
		while (eventReader.hasNext()){
			XMLEvent event = eventReader.nextEvent();
			if(Utils.checkStartElement(event, "collectionCode")){
				occurrence.setCollectionCode(Utils.readCharacters(eventReader));
			}else if(Utils.checkStartElement(event, "institutionCode")){
				occurrence.setInstitutionCode(Utils.readCharacters(eventReader));
			}else if(Utils.checkStartElement(event, "catalogNumber")){
				occurrence.setCatalogueNumber(Utils.readCharacters(eventReader));
			}else if(Utils.checkStartElement(event, "rights")){
				occurrence.addProperty(new ElementProperty("rignts",Utils.readCharacters(eventReader)));
			}else if(Utils.checkStartElement(event, "collector")){
				occurrence.setRecordedBy(Utils.readCharacters(eventReader));
			}else if(Utils.checkStartElement(event, "country")){
				occurrence.setCountry(Utils.readCharacters(eventReader));
			}else if(Utils.checkStartElement(event, "locality")){
				occurrence.setLocality(Utils.readCharacters(eventReader));
			}else if(Utils.checkStartElement(event, "earliestDateCollected")){
				String read= Utils.readCharacters(eventReader);
				try{
					DateFormat df = new SimpleDateFormat("yyyy-MM-dd");
					Calendar calendar= Calendar.getInstance();
					calendar.setTime(df.parse(read.replace(" ", "")));
					occurrence.setEventDate(calendar);
				}catch (ParseException e) {
					logger.warn("EventDate discarded ("+read+")");
				}
			}else if (Utils.checkStartElement(event, "latestDateCollected")){
				String read= Utils.readCharacters(eventReader);
				try{
					DateFormat df = new SimpleDateFormat("yyyy-MM-dd");
					Calendar calendar= Calendar.getInstance();
					calendar.setTime(df.parse(Utils.readCharacters(eventReader).replace(" ", "")));
					occurrence.setModified(calendar);
				}catch (ParseException e) {
					logger.warn("DateModified discarded ("+read+")");
				}
			}else if(Utils.checkStartElement(event, "decimalLatitude")){
				String read= Utils.readCharacters(eventReader);
				try{
					occurrence.setDecimalLatitude(Double.parseDouble(read.replace(" ", "")));
				}catch (NumberFormatException e) {
					logger.warn("DecimalLatitude discarded ("+read+")");
				}
			}else if(Utils.checkStartElement(event, "decimalLongitude")){
				String read= Utils.readCharacters(eventReader);
				try{
					occurrence.setDecimalLongitude(Double.parseDouble(read.replace(" ", "")));
				}catch (NumberFormatException e) {
					logger.warn("DecimalLongitude discarded ("+read+")");
				}
			}else if(Utils.checkStartElement(event, "nameComplete")){
				occurrence.setScientificName(Utils.readCharacters(eventReader));
			}else if(Utils.checkStartElement(event, "maximumDepthInMeters")){
				String read= Utils.readCharacters(eventReader);
				try{
					occurrence.setMaxDepth(Double.parseDouble(read));
				}catch (NumberFormatException e) {
					logger.warn("MaxDepth discarded ("+read+")");
				}
			}else if(Utils.checkStartElement(event, "minimumDepthInMeters")){
				String read= Utils.readCharacters(eventReader);
				try{
					occurrence.setMinDepth(Double.parseDouble(read));
				}catch (NumberFormatException e) {
					logger.warn("MinDepth discarded ("+read+")");
				}
			}else if(Utils.checkStartElement(event, "coordinateUncertaintyInMeters")){
				occurrence.setCoordinateUncertaintyInMeters(Utils.readCharacters(eventReader));
			}else if(Utils.checkStartElement(event, "basisOfRecordString")){
				String basisString = Utils.readCharacters(eventReader);
				occurrence.setBasisOfRecord(matchBasisOfRecord(basisString));
			}else if(Utils.checkStartElement(event, "citation")){
				occurrence.setCitation(Utils.readCharacters(eventReader));
			}else if (Utils.checkEndElement(event, "TaxonOccurrence")){
				return occurrence;	
			}
						
			
		}
		throw new Exception("parsing exception");
	}
	
}
