package org.gcube.application.enm.service.concurrent;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.PriorityQueue;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.gcube.application.enm.common.xml.status.StatusType;
import org.gcube.application.enm.service.GenericJob;
import org.gcube.application.enm.service.conn.JobUpdate;
import org.gcube.application.enm.service.conn.PersistenceClient;
import org.gcube.application.enm.service.conn.StorageClientUtil;
import org.gcube.common.core.utils.logging.GCUBELog;
import org.gcube.contentmanagement.blobstorage.service.IClient;

/**
 * Coordinates the access to the jobs from several different threads. Instead 
 * of using blocking collections (thread-free), this class implements a monitor 
 * to guarantee that no duplicates occur in the {@link PriorityQueue} of 
 * pending jobs, and no collisions occur with the {@link Map} of submitted jobs.
 * Even though, a {@link ConcurrentMap} is used to store jobs, because of its
 * possible use in other classes.
 * 
 * @author Erik Torres <ertorser@upv.es>
 */
public class JobMonitor {

	protected GCUBELog logger = new GCUBELog(JobMonitor.class);

	/**
	 * The list of jobs waiting for execution, ordered by their priority.
	 */
	private PriorityQueue<FIFOEntry<GenericJob>> pendingJobs = 
			new PriorityQueue<FIFOEntry<GenericJob>>();

	/**
	 * The list of all jobs.
	 */
	private ConcurrentMap<UUID, GenericJob> jobsMap = null;

	private final Lock mutex = new ReentrantLock();
	private final Condition hasPendingJobsCondition = mutex.newCondition();

	private final PersistenceClient persistenceClient = new PersistenceClient();

	/**
	 * The singleton instance of the monitor.
	 */
	private static JobMonitor instance;

	/**
	 * Get the singleton instance of the monitor.
	 * @return The singleton instance.
	 */
	public static JobMonitor get() {
		if (instance == null) {
			instance = new JobMonitor();
		}
		return instance;
	}

	private JobMonitor() {
		// Setup logging
		logger.trace("Constructor...");
	}

	/**
	 * Initializes the job monitor, scheduling the execution of those jobs with
	 * the status pending.
	 * @param jobsMap Holds the list of all jobs.
	 * @throws IllegalArgumentException
	 */
	public void init(final ConcurrentMap<UUID, GenericJob> jobsMap) 
			throws IllegalArgumentException {
		mutex.lock();
		try {
			if (this.jobsMap == null) {
				this.jobsMap = jobsMap;
				if (this.jobsMap != null) {
					for (final Map.Entry<UUID, GenericJob> entry : 
						this.jobsMap.entrySet()) {
						final GenericJob job = entry.getValue();
						final StatusType status = job.getStatus().getStatus();
						switch (status) {					
						case PENDING:
						case EXECUTING:
							pendingJobs.add(new FIFOEntry<GenericJob>(job));
							break;
						case FINISHED:
						case FAILED:
						case CANCELLED:
						default:
							// nothing to do
							break;
						}
					}
				}
			}
			else throw new IllegalArgumentException(
					"A jobs map was already initialized");
		} finally {
			mutex.unlock();
		}
	}

	public int numPendingJobs() {
		mutex.lock();
		try {
			return pendingJobs.size();		
		} finally {
			mutex.unlock();
		}
	}

	/**
	 * Retrieves the next job from the queue, waiting if necessary until a job 
	 * becomes available.
	 * @return The next job of the queue.
	 */
	public GenericJob nextPendingJob() {
		mutex.lock();
		try {
			// Wait until next job is available
			while (pendingJobs.isEmpty())
				hasPendingJobsCondition.await();
			return pendingJobs.poll().getEntry(); // TODO : fix the broke tie
		} catch (InterruptedException e) {
			// nothing to do
			return null;
		} finally {
			mutex.unlock();
		}
	}

	/**
	 * Inserts the specified job into the priority queue.
	 * @param job The job to add.
	 */
	public void schedulePendingJob(final GenericJob job) {
		mutex.lock();
		try {
			// Check whether the job exists in the list
			final FIFOEntry<GenericJob> fifoEntry = 
					new FIFOEntry<GenericJob>(job);
			if (!pendingJobs.contains(fifoEntry) 
					&& pendingJobs.offer(fifoEntry))
				hasPendingJobsCondition.signalAll();
		} finally {
			mutex.unlock();
		}
	}

	/**
	 * Register a new job.
	 * @param uuid Unequally identifies the job.
	 * @param job The job to add.
	 */
	public boolean registerNewJob(final GenericJob job) {
		mutex.lock();
		try {
			boolean registered = false;
			// Do not override previous jobs
			final GenericJob previous = jobsMap.putIfAbsent(job.getUUID(), job);
			if (previous == null) {
				try {
					// Write the job to the persistent storage
					persistenceClient.write(job);
					registered = true;
				} catch (IOException e) {
					logger.error("Failed to save job '" 
							+ job.getUUID().toString() + "' to the persistent storage: " 
							+ e.getLocalizedMessage());
				}
			} else {
				registered = false;
				logger.error("Job UUID collision! Offending id: '" 
						+ job.getUUID().toString() + "'");
			}
			return registered;
		} finally {
			mutex.unlock();
		}
	}

	/**
	 * Returns the job to which the specified <code>uuid</code> is mapped, or 
	 * <code>null</code> if this map contains no mapping for the 
	 * <code>uuid</code>. In the case that the job is not contained in the map
	 * loaded in memory, the method uses the persistence client to retrieve the
	 * job from the persistent storage and puts the job in the map, before 
	 * returning it to the caller.
	 * @param uuid Unequally identifies the job.
	 * @return The job to which the specified <code>uuid</code> is mapped, or 
	 *         <code>null</code> if this map contains no mapping for the 
	 *         <code>uuid</code>.
	 */
	public GenericJob getJob(final UUID uuid) {
		mutex.lock();
		try {
			GenericJob job = null;
			if (uuid != null) {
				job = jobsMap.get(uuid);
				if (job == null) {
					final String credentials = persistenceClient
							.findCredentials(uuid);
					job = persistenceClient.read(credentials, uuid);
					if (job != null)
						jobsMap.put(uuid, job);
				}
			}
			return job;	
		} catch (Exception e) {
			logger.error("Failed to load job '" + uuid.toString() + "': " 
					+ e.getLocalizedMessage());
			return null;
		} finally {
			mutex.unlock();
		}
	}

	/**
	 * Cancels a job.
	 * @param uuid Unequally identifies the job.
	 */
	public void cancelJob(final UUID uuid) {
		mutex.lock();
		try {
			final GenericJob job = getJob(uuid);
			if (job != null) {
				final StatusType status = job.getStatus().getStatus();
				switch (status) {					
				case PENDING:
				case EXECUTING:
					job.cancel();
					// Check whether the job exists in the list
					final FIFOEntry<GenericJob> fifoEntry = 
							new FIFOEntry<GenericJob>(job);
					if (!pendingJobs.contains(fifoEntry) 
							&& pendingJobs.offer(fifoEntry))				
						hasPendingJobsCondition.signalAll();
					break;
				case FINISHED:
				case FAILED:
				case CANCELLED:
				default:
					// nothing to do
					logger.trace("Ignoring cancel request for the job '" 
							+ job.getUUID().toString() 
							+ "' that already finished with the status: "
							+ status.toString());
					break;
				}
			}
		} finally {
			mutex.unlock();
		}
	}

	/**
	 * Call this method any time that a job receives an update.
	 * @param job The job to update.
	 */
	public void updateJob(final GenericJob job, final JobUpdate updates) {
		mutex.lock();
		try {
			try {
				persistenceClient.write(job, updates);
			} catch (IOException e) {
				logger.error("Failed to save experiment '" 
						+ job.getUUID().toString() + "': " 
						+ e.getLocalizedMessage());
			}
		} finally {
			mutex.unlock();
		}
	}

	/**
	 * Removes a job from the map and the persistent storage. This effectively
	 * removes the job from the service. This method does not cancels the job
	 * on the remote execution resource. If you want to cancel the job, call
	 * the method {@link JobMonitor#cancelJob(UUID)} wait until the status of
	 * the job change to cancelled, and then call this method.
	 * @param job The job to remove.
	 */
	public void removeJob(final UUID uuid) {
		mutex.lock();
		try {
			GenericJob job = null;
			String credentials = null;
			if (uuid != null) {
				job = jobsMap.get(uuid);
				if (job == null) {
					credentials = persistenceClient.findCredentials(uuid);
					job = persistenceClient.read(credentials, uuid);
				}
			}
			if (job != null) {
				jobsMap.remove(uuid);
				final FIFOEntry<GenericJob> fifoEntry = 
						new FIFOEntry<GenericJob>(job);
				if (pendingJobs.contains(fifoEntry)) {		
					final List<FIFOEntry<GenericJob>> list = 
							new ArrayList<FIFOEntry<GenericJob>>();
					while (!pendingJobs.isEmpty()) {
						final FIFOEntry<GenericJob> item = pendingJobs.poll();
						if (!item.equals(fifoEntry))
							list.add(item);
					}
					pendingJobs.addAll(list);
				}
				persistenceClient.delete(job);
				// delete results from the storage service
				try {
					final IClient storageClient = StorageClientUtil.client();
					storageClient.removeDir().RDir(
							StorageClientUtil.experimentBaseDir(
									job.getRequest().getCredentials(), 
									job.getUUID()));
				} catch (Exception e) {
					logger.error("Failed to delete experiment '" 
							+ job.getUUID().toString() 
							+ "' from the storage service: " 
							+ e.getLocalizedMessage());
				}
				// broadcast signal to possible waiting threads
				hasPendingJobsCondition.signalAll();
			}
		} catch (Exception e) {
			logger.error("Failed to delete job '" + uuid.toString() + "': " 
					+ e.getLocalizedMessage());
		} finally {
			mutex.unlock();
		}
	}

	public void clean() {
		mutex.lock();
		try {
			final Set<UUID> keySet = jobsMap.keySet();
			for (final UUID key : keySet) {
				final GenericJob job = jobsMap.get(key);
				switch (job.getStatus().getStatus()) {
				case PENDING:
				case EXECUTING:				
					break;
				case FAILED:
				case FINISHED:
				case CANCELLED:
				default:
					jobsMap.remove(key,job);
					logger.trace("Job '" + key + "' was taken off the map");
					break;
				}
			}
		} finally {
			mutex.unlock();
		}
	}

}
