package org.gcube.application.enm.service.concurrent;

import java.util.concurrent.CompletionService;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import org.gcube.application.enm.service.GenericJob;
import org.gcube.common.core.utils.logging.GCUBELog;

/**
 * Provides a pool of threads with a fixed number of workers.
 * 
 * @author Erik Torres <ertorser@upv.es>
 */
public class JobExecutionPool {

	protected GCUBELog logger = new GCUBELog(JobExecutionPool.class);

	private static final int NUM_WORKERS = 2;
	private static final int TIMEOUT_SECS = 20;

	private final ExecutorService pool;
	private final CompletionService<String> completion;

	private AtomicBoolean isStarted = new AtomicBoolean(false);
	private AtomicBoolean shouldRun = new AtomicBoolean(true);

	/**
	 * The singleton instance of the thread pool.
	 */
	private static JobExecutionPool instance;

	/**
	 * Get the singleton instance of the thread pool.
	 * @return The singleton instance.
	 */
	public static JobExecutionPool get() {
		if (instance == null) {
			instance = new JobExecutionPool();
		}
		return instance;
	}

	/**
	 * Construct a new thread pool.
	 */
	private JobExecutionPool() {
		// Setup logging
		logger.trace("Constructor with " + NUM_WORKERS + " thread workers");
		// Create the pool
		pool = Executors.newFixedThreadPool(NUM_WORKERS);
		completion = new ExecutorCompletionService<String>(pool);
	}

	public void start() {
		if (!isStarted.get()) {
			isStarted.set(true);
			// Start the pool
			(new Thread(new Runnable() {
				@Override
				public void run() {
					while (shouldRun.get()) {
						try {
							final String res = completion.take().get();
							if (res != null) {
								logger.trace("Job Completed: " + res);								
							}
						} catch (InterruptedException e) {
							// nothing to do
						} catch (ExecutionException e) {
							if (e.getCause() != null) {
								logger.error("Job failed with the exception: "
										+ e.getCause().getLocalizedMessage());					
							} else {
								logger.error("Job failed with unkown causes");
							}
						} catch (Exception e) {
							// nothing to do
						}
					}		
				}
			})).start();			
		}
	}

	public final void shutdownAndAwaitTermination() {
		// shouldRun.set(false);
		pool.shutdown(); // Disable new jobs from being submitted
		try {
			// Wait a while for existing jobs to terminate
			if (!pool.awaitTermination(TIMEOUT_SECS, TimeUnit.SECONDS)) {
				pool.shutdownNow(); // Cancel currently executing jobs
				// Wait a while for jobs to respond to being cancelled
				if (!pool.awaitTermination(TIMEOUT_SECS, TimeUnit.SECONDS))
					logger.warn("Pool did not terminate");
			}
			shouldRun.set(false); // The original method call was commented
		} catch (InterruptedException ie) {
			// (Re-)Cancel if current thread also interrupted
			pool.shutdownNow();
			// Preserve interrupt status
			Thread.currentThread().interrupt();
		} finally {
			logger.trace("Exit Thread Pool");
		}
	}

	public void submit(final GenericJob job) {
		try {
			completion.submit(job);
		} catch (Exception e) {
			// nothing to do
		}
	}

}
